/*
    Copyright 2009 Luigi Auriemma

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    http://www.gnu.org/licenses/gpl-2.0.txt
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <sys/stat.h>
#include <zlib.h>
#include <pcap.h>
#include "ether_hdrlen.h"
#include "ip2.h"
#include "leverage_ssc.h"
#include "show_dump.h"

typedef uint8_t     u8;
typedef uint32_t    u32;



#define VER         "0.2.1"



u8 *skip_pckhead(int if_offset, u8 *pkt);
void z_show_dump(u8 *in, int insz, FILE *fdo);
int my_aa3keydb(int keynum, u8 *buff, int buffsz, u8 *key);
int getxx(u8 *data, u32 *ret, int bits);
int get_num(u8 *str);
u8 *fdload(u8 *fname, int *fsize, int dontquit);
void std_err(void);



void (*ssc_crypt)(unsigned char *, int, unsigned char *, int) = ssc_decrypt;



int main(int argc, char *argv[]) {
    FILE    *fdo            = NULL;
    u32     n,
            crc;
    int     i,
            kn,
            len,
            type,
            buffsz,
            keydbsz,
            if_offset       = 0,
            offset          = 0;
    u8      header_key[]    = "9dfksdj5bb783bhd93jg0d72jhx03k90",
            hash_key[]      = "j48cnh02lx7vm90d5bh590xch5isd854",
            //query_key[]     = "c6mw4it2kg7sz5o0813d9qyufenhj",
            key[32]         = "",
            *aa3keydb,
            *packet_file,
            *keydb,
            *out_file       = NULL,
            *buff,
            *p,
            *l,
            *t,
            *next           = NULL;

    fputs("\n"
        "America's Army 3 auth packets ssc_decrypt "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", stdout);

    if(argc < 3) {
        printf("\n"
            "Usage: %s [options] <aa3.key.db/key> <packet_file>\n"
            "\n"
            "Options:\n"
            "-e       encrypt the input file using ssc_encrypt\n"
            "-f FILE  dump the decrypted data in FILE (default is hex dump on the screen)\n"
            "-o OFF   offset of packet_file from which starting the parsing/decryption\n"
            "\n"
            "aa3.key.db is the file located in the \"America's Army 3\\Binaries\" folder and\n"
            "it contains all the keys required to decrypt/encrypt the packets for the auth\n"
            "server auth.aa3.americasarmy.com (aa3.key.db is the same file everywhere)\n"
            "if the file doesn't exist, it will be considered a custom key to use for\n"
            "decrypting the whole packet_file, example c6mw4it2kg7sz5o0813d9qyufenhj used\n"
            "for the informations queries\n"
            "\n"
            "packet_file is the dump of the raw data sent and received from the auth server\n"
            "or directly a sniffed session in tcpdump format (like the dumps from Wireshark)\n"
            "\n", argv[0]);
        exit(1);
    }

    argc -= 2;
    for(i = 1; i < argc; i++) {
        if(((argv[i][0] != '-') && (argv[i][0] != '/')) || (strlen(argv[i]) != 2)) {
            printf("\nError: wrong argument (%s)\n", argv[i]);
            exit(1);
        }
        switch(argv[i][1]) {
            case 'e': ssc_crypt = ssc_encrypt;          break;
            case 'f': out_file  = argv[++i];            break;
            case 'o': offset    = get_num(argv[++i]);   break;
            default: {
                printf("\nError: wrong argument (%s)\n", argv[i]);
                exit(1);
            }
        }
    }
    aa3keydb    = argv[argc];
    packet_file = argv[argc + 1];

    keydb = fdload(aa3keydb,    &keydbsz, 1);
    buff  = fdload(packet_file, &buffsz,  0);

    if(buffsz < offset) {
        printf("\nError: the offset you specified (%d) is bigger than the file size (%d)\n", offset, buffsz);
        exit(1);
    }

    p = buff + offset;
    l = buff + buffsz;

    getxx(p, &n, 32);
    if(n == 0xa1b2c3d4) {
        p += 4;
        p += getxx(p, &n,   16);    // major version
        p += getxx(p, &n,   16);    // minor version
        p += getxx(p, &n,   32);    // timezone offset
        p += getxx(p, &n,   32);    // timestamp accuracy
        p += getxx(p, &n,   32);    // snapshot length
        p += getxx(p, &n,   32);    // link-layer type
        if_offset = pcap_hdrlen(n);
        printf("- handled as a tcpdump file\n");

        for(t = buff + offset; p < l; p = next) {
            p += getxx(p, &n,   32);    // seconds
            p += getxx(p, &n,   32);    // microseconds
            p += getxx(p, &len, 32);    // caplen
            p += getxx(p, &n,   32);    // pcklen
            next = p + len;
            if((next < p) || (next > l)) break;
            p = skip_pckhead(if_offset, p);
            if(p) {
                memcpy(t, p, next - p);
                t += (next - p);
            }
        }
        l = t;
        p = buff + offset;
    }

    if(keydb) {
        while(p < l) {
            p += getxx(p, &n,   32);
            p += getxx(p, &n,   16);
            p += getxx(p, &len, 32);    // length of the header

            ssc_crypt(header_key, strlen(header_key) + 1, p, len);
            if(!out_file) printf("\n- decrypted command: %.*s\n", len, p);
            p += len;

            p += getxx(p, &kn,  32);    // key number
            p += getxx(p, &len, 32);    // length of the block
            p += getxx(p, &crc, 32);    // crc
            p += getxx(p, &n,   32);
            p += getxx(p, &type,32);

            crc = ascii_calculate_key_hash(hash_key, strlen(hash_key) + 1, p, len);

            if(kn) {
                if(my_aa3keydb(kn, keydb, keydbsz, key) < 0) {
                    printf("\nError: the key with keynum %d has not been found in aa3.key.db\n", kn);
                    exit(1);
                }
                ssc_crypt(key, 32, p, len);
            }
            if(!out_file) {
                if(type & 1) {
                    printf("- decrypted and uncompressed data:\n");
                    z_show_dump(p, len, stdout);
                } else {
                    printf("- decrypted data:\n");
                    show_dump(p, len, stdout);
                }
            }
            p += len;

            p += 4;
        }
    } else {
        keydb = aa3keydb;
        len = l - p;
        ssc_crypt(keydb, strlen(keydb) + 1, p, len);
        if(!out_file) show_dump(p, len, stdout);
    }

    if(out_file) {
        printf("- create output file %s\n", out_file);
        fdo = fopen(out_file, "wb");
        if(!fdo) std_err();
        fwrite(buff + offset, 1, buffsz - offset, fdo);
        fclose(fdo);
    }
    printf("\n- done\n");
    free(buff);
    return(0);
}



u8 *skip_pckhead(int if_offset, u8 *pkt) {
    iph     *ip;
    tcph    *tcp;
    int     iflen,
            iplen,
            tcplen,
            len;
    u8      *data;

    iflen  = ntohs(*(uint16_t *)(pkt + if_offset - 2));
    ip     = (iph *)(pkt + if_offset + ether_hdrlen(iflen));
    if((ip->ihl_ver >> 4) != 4) return(NULL);
    iplen  = (ip->ihl_ver & 15) << 2;

    if(ip->protocol != IPPROTO_TCP) return(NULL);

    tcp    = (tcph *)((uint8_t *)ip + iplen);
    tcplen = tcp->doff >> 2;

    //if(!(tcp->flags & TH_PSH)) goto quit; // never enable it!

    data   = (uint8_t *)((uint8_t *)tcp + tcplen);
    len    = ntohs(ip->tot_len) - (data - (uint8_t *)ip);

    if(len <= 0) return(NULL);
    return(data);
}



void z_show_dump(u8 *in, int insz, FILE *fdo) {
    static int      outsz   = 0;
    static u8       *out    = NULL;
    static z_stream *z      = NULL;
    int     ret;

    if(!z) {
        z = malloc(sizeof(z_stream));
        if(!z) std_err();
        z->zalloc = (alloc_func)0;
        z->zfree  = (free_func)0;
        z->opaque = (voidpf)0;
        if(inflateInit2(z, 15)) {
            printf("\nError: zlib initialization error\n");
            exit(1);
        }
    }

    if(outsz < insz) {
        outsz = insz * 2;
        out = realloc(out, outsz);
        if(!out) std_err();
    }

    for(;;) {
        inflateReset(z);
        z->next_in   = in;
        z->avail_in  = insz;
        z->next_out  = out;
        z->avail_out = outsz;
        ret = inflate(z, Z_SYNC_FLUSH);
        if(ret == Z_STREAM_END) {
            ret = 0;
            break;
        }
        if((ret != Z_OK) && (ret != Z_MEM_ERROR) && (ret != Z_BUF_ERROR)) {
            ret = -1;
            break;
        }

        outsz += insz;
        out = realloc(out, outsz);
        if(!out) std_err();
    }

    if(!ret) {
        show_dump(out, z->total_out, fdo);
    } else {
        show_dump(in,  insz,  fdo);
    }
}



int my_aa3keydb(int keynum, u8 *buff, int buffsz, u8 *key) {    // it's the simplest solution
    u32     start   = 0x3186f265,
            end     = 0x39e0b015;
    int     num;
    u8      fmap_key[]  = "jbkw9nv07s5hyp4skms0uj6t5a9",
            *p,
            *n,
            *l;

    p = buff;
    l = buff + buffsz;
    num = 0;
    while(p < l) {
        if(*(u32 *)p != start) break;

        for(n = p + 4; n < l; n++) {
            if(*(u32 *)n == end) break;
        }

        if(*(u32 *)(p + 4) == 0x01000001) {
            if(num == keynum) {
                memcpy(key, p + 0x34, 32);
                ssc_decrypt(fmap_key, strlen(fmap_key), key, 32);
                return(0);
            }
            num++;
        }
        p = n + 4;
    }
    return(-1);
}



int getxx(u8 *data, u32 *ret, int bits) {
    u32     num;
    int     i,
            bytes;

    bytes = bits >> 3;
    for(num = i = 0; i < bytes; i++) {
        num |= (data[i] << (i << 3));
    }
    *ret = num;
    return(bytes);
}



int get_num(u8 *str) {
    int     offset;

    if(!strncmp(str, "0x", 2) || !strncmp(str, "0X", 2)) {
        sscanf(str + 2, "%x", &offset);
    } else {
        sscanf(str, "%u", &offset);
    }
    return(offset);
}



u8 *fdload(u8 *fname, int *fsize, int dontquit) {
    struct stat xstat;
    FILE    *fd;
    int     size;
    u8      *buff;

    printf("- open %s\n", fname);
    fd = fopen(fname, "rb");
    if(!fd) {
        if(!dontquit) std_err();
        return(NULL);
    }
    fstat(fileno(fd), &xstat);
    size = xstat.st_size;
    buff = malloc(size);
    if(!buff) std_err();
    fread(buff, 1, size, fd);
    fclose(fd);
    *fsize = size;
    return(buff);
}



void std_err(void) {
    perror("\nError");
    exit(1);
}


