/*

by Luigi Auriemma

*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include "ase_udp_decenc.h"
#include "show_dump.h"

#ifdef WIN32
    #include <winsock.h>
    #include "winerr.h"

    #define close   closesocket
    #define sleep   Sleep
#else
    #include <unistd.h>
    #include <sys/socket.h>
    #include <sys/types.h>
    #include <arpa/inet.h>
    #include <netinet/in.h>
    #include <netdb.h>
#endif



#define VER     "0.1"
#define HOST    "tracker.udpsoft.com"
#define PORT    27246
#define BUFFSZ  2048
#define TIMEOUT 3



int timeout(int sock);
u_int resolv(char *host);
void std_err(void);



int main(int argc, char *argv[]) {
    struct  sockaddr_in peer;
    u_int  motdidlo,
            motdidhi,
            userid,
            retid;
    int     sd,
            len,
            state = 0;
    u_char  buff[BUFFSZ];

#ifdef WIN32
    WSADATA    wsadata;
    WSAStartup(MAKEWORD(1,0), &wsadata);
#endif


    setbuf(stdout, NULL);

    fputs("\n"
        "ASE UDP tracker packet sender "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", stdout);

    if(argc < 4) {
        printf("\n"
            "Usage: %s <MotdIdLo> <MotdIdHi> <UserID> [state]\n"
            "\n"
            " State is referred to an user that joins (0, default) or that exits (2)\n"
            " Note: values must be entered in hex mode (with or without 0x is the same)\n"
            "       Your unique values are located in the following registry key:\n"
            "         HKEY_CURRENT_USER\\Software\\UDP Soft\\The All-Seeing Eye\\Settings\n"
            "\n", argv[0]);
        exit(1);
    }

    if(argc > 4) state = atoi(argv[4]);

    sscanf(argv[1], "%lx", &motdidlo);
    sscanf(argv[2], "%lx", &motdidhi);
    sscanf(argv[3], "%lx", &userid);

    printf(
        "  MotdIdLo   0x%08x\n"
        "  MotdIdHi   0x%08x\n"
        "  UserID     0x%08x\n"
        "\n",
        motdidlo, motdidhi, userid);

    peer.sin_addr.s_addr = resolv(HOST);
    peer.sin_port        = htons(PORT);
    peer.sin_family      = AF_INET;

    printf("- " HOST "   %s : %hu\n",
        inet_ntoa(peer.sin_addr), PORT);

    *(u_int *)buff         = 60;
    *(u_int *)(buff + 4)   = 0;
    *(u_int *)(buff + 8)   = motdidlo;
    *(u_int *)(buff + 12)  = motdidhi;
    buff[16]                = state;
    buff[17]                = 0;
    *(u_int *)(buff + 18)  = 0;
    *(u_short *)(buff + 22) = 0;
    *(u_int *)(buff + 24)  = 0xffffffff;
    *(u_int *)(buff + 28)  = userid;

    ase_udp_decenc(buff, 32, 1);

    fputs("- send packet 1\n", stdout);
    sd = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
    if(sd < 0) std_err();
    if(sendto(sd, buff, 32, 0, (struct sockaddr *)&peer, sizeof(peer))
      < 0) std_err();

    if(state) {
        fputs("\nFinished\n\n", stdout);
        return(0);
    }

    *(u_int *)buff         = 63;
    *(u_int *)(buff + 4)   = 0;
    *(u_int *)(buff + 8)   = motdidlo;
    *(u_int *)(buff + 12)  = motdidhi;
    *(u_int *)(buff + 16)  = userid;
    buff[20]                = 2;
    *(u_int *)(buff + 21)  = 0;

    ase_udp_decenc(buff, 25, 1);

    fputs("- send packet 2\n", stdout);
    sd = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
    if(sd < 0) std_err();
    if(sendto(sd, buff, 25, 0, (struct sockaddr *)&peer, sizeof(peer))
      < 0) std_err();

    fputs("- receive reply:\n\n", stdout);
    if(timeout(sd) < 0) {
        fputs("\nError: socket timeout, no reply received\n\n", stdout);
        exit(1);
    }
    len = recvfrom(sd, buff, BUFFSZ, 0, NULL, NULL);
    if(len < 0) std_err();

    close(sd);

    ase_udp_decenc(buff, len, 0);
    show_dump(buff, len, stdout);

    retid = *(u_int *)(buff + len - 4);
    if(retid != userid) {
        printf("\n"
            "- the server seems to have not accepted your packet, in fact the returned\n"
            "  UserID differs from yours:   %08x\n"
            "\n", retid);
    } else {
        fputs("\n- your user has been accepted\n\n", stdout);
    }

    return(0);
}



int timeout(int sock) {
    struct  timeval tout;
    fd_set  fd_read;
    int     err;

    tout.tv_sec = TIMEOUT;
    tout.tv_usec = 0;
    FD_ZERO(&fd_read);
    FD_SET(sock, &fd_read);
    err = select(sock + 1, &fd_read, NULL, NULL, &tout);
    if(err < 0) std_err();
    if(!err) return(-1);
    return(0);
}



u_int resolv(char *host) {
    struct hostent *hp;
    u_int host_ip;

    host_ip = inet_addr(host);
    if(host_ip == INADDR_NONE) {
        hp = gethostbyname(host);
        if(!hp) {
            printf("\nError: Unable to resolv hostname (%s)\n", host);
            exit(1);
        } else host_ip = *(u_int *)hp->h_addr;
    }
    return(host_ip);
}



#ifndef WIN32
    void std_err(void) {
        perror("\nError");
        exit(1);
    }
#endif

