/*
    Copyright 2007,2008,2009 Luigi Auriemma

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    http://www.gnu.org/licenses/gpl-2.0.txt
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <time.h>
#include <errno.h>
#include <sys/stat.h>
#include <ctype.h>
#include <zlib.h>
#include "mycrc32.h"
#include "mydownlib.h"

#ifdef WIN32
    #include <winsock.h>
    #include <direct.h>
    #include "winerr.h"

    #define close       closesocket
    #define sleep       Sleep
    #define usleep      sleep
    #define ONESEC      1000
    #define check_time  clock()
    #define make_dir(x) mkdir(x)
#else
    #include <unistd.h>
    #include <sys/socket.h>
    #include <sys/types.h>
    #include <arpa/inet.h>
    #include <netinet/in.h>
    #include <netdb.h>
    #include <sys/stat.h>
    #include <sys/times.h>
    #include <sys/timeb.h>

    #define stristr     strcasestr
    #define ONESEC      1
    #define check_time  (times(0) * 10)
    #define make_dir(x) mkdir(x, 0755)
#endif



    // http://www.anywebcam.com/awc/awc40040.jar

#define VER             "0.2.1d"
#define HOST            "camsv4.anywebcam.com"  // in the past was "bc.anywebcam.com"
#define PORT            9004
#define WATCH_EVER      "watch_ever.htm"
#define WATCH_THUMB     "watch_thumb.htm"
#define CAMSTEST_FD     "camstest.gz"
#define CAMSTEST_WWW    "http://www.anywebcam.com/res/" CAMSTEST_FD
#define REFRESH_LIST    300
#define BUFFSZ          32767       // max size of the webcam jpg

#define SOCKERR         (-1)
#define TIMEERR         (-2)



typedef struct {
    uint32_t    id;
    uint32_t    crc;
    uint8_t     *name;
} ids_t;



struct  sockaddr_in peer;
int     record = 0;



void check_filter(uint8_t *p);
uint8_t *myitoa(int num);
uint8_t *get_date(void);
void launch_browser(uint8_t *fname);
ids_t *getids(uint8_t *buff, uint8_t *filter);
uint8_t *fgetz(uint8_t **buff);
void create_folder(uint8_t *folder);
uint8_t *fdloadgz(uint8_t *fname);
uint8_t *unzip(uint8_t *in, int *insz);
void awcamrec(ids_t *ids);
int tcp_recv(int sd, uint8_t *data, int len);
int rnd_mem(uint8_t *data, int len, uint32_t rnd);
int save_file(uint8_t *data, int len, uint32_t id, uint32_t frame);
void write_watch_ever(void);
uint32_t mysqrt(uint32_t num);
void write_watch_thumb(ids_t *ids);
int getxx(uint8_t *data, uint32_t *ret, int bits);
int putxx(uint8_t *data, uint32_t num, int bits);
int timeout(int sock);
void show_help(void);
uint32_t resolv(char *host);
void std_err(void);



int main(int argc, char *argv[]) {
    mydown_options  opt;
    ids_t       *ids;
    uint32_t    webcam_id   = 0;
    int         i,
                list        = 0,
                run_browser = 0;
    uint8_t     *fname      = CAMSTEST_FD,
                *filter     = NULL,
                *camstest   = NULL;

#ifdef WIN32
    WSADATA    wsadata;
    WSAStartup(MAKEWORD(1,0), &wsadata);
#endif

    setbuf(stdout, NULL);

    fputs("\n"
        "AWCamREC "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", stderr);

    if(argc < 2) {
        printf("\n"
            "Usage %s [options] [webcam_ID]\n"
            "\n", argv[0]);
        show_help();
    }

    for(i = 1; i < argc; i++) {
        if(((argv[i][0] != '-') && (argv[i][0] != '/')) || (strlen(argv[i]) != 2)) {
            if(i == (argc - 1)) {
                webcam_id = atoi(argv[i]);
                break;
            }
            fprintf(stderr, "\nError: wrong command-line argument (%s)\n", argv[i]);
            exit(1);
        }
        switch(argv[i][1]) {
            case 'r': {
                if(!argv[++i]) show_help();
                webcam_id         = atoi(argv[i]);
                } break;
            case 'f': {
                if(!argv[++i]) show_help();
                filter            = argv[i];
                check_filter(filter);
                } break;
            case 'c': {
                if(!argv[++i]) show_help();
                fname             = argv[i];
                } break;
            case 'l': list        = 1;                  break;
            case 'i': run_browser = 1;                  break;
            default: {
                printf("\nError: wrong command-line argument (%s)\n", argv[i]);
                exit(1);
            }
        }
    }

    if(!list) {
        printf("- resolv %s\n", HOST);
        peer.sin_addr.s_addr = resolv(HOST);
        peer.sin_port        = htons(PORT);
        peer.sin_family      = AF_INET;
    }

    if(webcam_id) {
        printf("- start recording of the webcam ID %u\n", webcam_id);

        ids = malloc(sizeof(ids_t) * 2);
        ids[0].id   = webcam_id;
        ids[0].crc  = 0;
        ids[0].name = "";
        ids[1].id   = 0;

        create_folder(myitoa(webcam_id));
        create_folder(get_date());
        write_watch_ever();
        if(run_browser) launch_browser(WATCH_EVER);

        record = 1;

        awcamrec(ids);

        goto quit;
    }

    printf("- start thumbnails collection\n");

    memset(&opt, 0, sizeof(opt));
    opt.useragent    = "Mozilla/4.0 (Windows XP 5.1) Java/1.5.0_11";
    opt.verbose      = 1;

    create_folder("thumb");

    for(;;) {
        if(fname == (uint8_t *)CAMSTEST_FD) {
            mydown(CAMSTEST_WWW, fname, &opt);
        }

        if(camstest) free(camstest);
        camstest = fdloadgz(fname);
        if(!camstest) {
            printf("\nError: something wrong during the loading of the camstest file\n");
            exit(1);
        }

        ids = getids(camstest, filter);

        if(list) goto quit;

        write_watch_thumb(ids);
        if(run_browser) {
            launch_browser(WATCH_THUMB);
            run_browser = 0;
        }

        awcamrec(ids);
    }

quit:
    free(ids);
    return(0);
}



void check_filter(uint8_t *p) {
    for(; *p && (*p != ','); p++) {
        *p = tolower(*p);
        switch(*p) {
            case 'u':
            case 'm':
            case 'f':
            case 'g':
            case 'o': break;
            default: {
                printf("\nError: wrong filter, recheck it\n");
                exit(1);
                } break;
        }
    }
}



uint8_t *myitoa(int num) {
    static uint8_t  mini[12];
    sprintf(mini, "%u", num);
    return(mini);
}



uint8_t *get_date(void) {
    time_t  datex;
    struct  tm  *tmx;
    static uint8_t  folder[32];
    static const uint8_t *months[12] = {
                "jan","feb","mar","apr","may","jun",
                "jul","aug","sep","oct","nov","dec"
            };

    time(&datex);
    tmx = localtime(&datex);
    sprintf(
        folder,
        "%02u.%s.%02u-%02u.%02u",
        tmx->tm_mday, months[tmx->tm_mon], tmx->tm_year % 100,
        tmx->tm_hour, tmx->tm_min);

    return(folder);
}



void launch_browser(uint8_t *fname) {
    uint8_t     tmp[256];

#ifdef WIN32
    getcwd(tmp, sizeof(tmp) - strlen(fname));
    sprintf(tmp + strlen(tmp), "\\%s", fname);
    printf("- launch browser: %s\n", tmp);

    ShellExecute(
        NULL,
        "open",
        tmp,
        NULL,
        NULL,
        SW_SHOW);
#else
    strcpy(tmp, "$BROWSER ");
    getcwd(tmp + 9, sizeof(tmp) - 9 - strlen(fname) - 3);
    sprintf(tmp + strlen(tmp), "/%s &", fname);
    printf("- launch browser: %s\n", tmp);

    system(tmp);
#endif
}



ids_t *getids(uint8_t *buff, uint8_t *filter) {
    ids_t       *ids;
    uint32_t    id,
                type,
                idsnum;
    int         ok;
    uint8_t     *nick,
                *country,
                *data,
                *p,
                *l;

    idsnum = 1;                 // full allocation
    for(p = buff; *p; p++) {
        if(*p == '\n') idsnum++;
    }
    ids = malloc(sizeof(ids_t) * (idsnum + 1));
    if(!ids) std_err();
    idsnum = 0;

    printf("- webcams:\n\n");

    while((data = fgetz(&buff))) {

#define DOIT(x) p = strchr(data, ',');  \
                if(!p) continue;        \
                *p = 0;                 \
                x;                      \
                data = p + 1;

        DOIT(id      = atoi(data))
        DOIT(nick    = data)
        DOIT(country = data)
        DOIT()
        DOIT(type    = atoi(data))

#undef DOIT

        if(!id) continue;

        if(filter) {
            ok = 0;
            p = strchr(filter, ',');
            if(p) *p = 0;

            switch(type) {
                case 0:  if(strchr(filter, 'u')) ok = 1;    break;
                case 1:  if(strchr(filter, 'm')) ok = 1;    break;
                case 2:  if(strchr(filter, 'f')) ok = 1;    break;
                case 6:  if(strchr(filter, 'g')) ok = 1;    break;
                default: if(strchr(filter, 'o')) ok = 1;    break;
            }

            if(p) {
                *p = ',';
                for(p++; !ok; p = l + 1) {
                    l = strchr(p, ',');
                    if(l) *l = 0;
                    if(stristr(nick, p)) ok = 1;
                    if(!l) break;
                    *l = ',';
                }
            }

            if(!ok) continue;
        }

        printf("  %-10u %-45s %s\n", id, nick, country);
        ids[idsnum].id   = id;
        ids[idsnum].crc   = 0;
        ids[idsnum].name = nick;
        idsnum++;
    }

    if(!idsnum) {
        printf("\n- no webcams in the filter\n");
        free(ids);
        exit(1);
    }

    printf("\n- %u webcams in the filter\n", idsnum);
    ids[idsnum].id = 0;     // delimiter
    return(ids);
}



uint8_t *fgetz(uint8_t **buff) {
    uint8_t     *ret,
                *p;

    for(ret = p = *buff; *p && (*p != '\n') && (*p != '\r'); p++);
    if(!(p - ret)) return(NULL);
    *p = 0;
    *buff = p + 1;
    return(ret);
}



void create_folder(uint8_t *folder) {
    uint8_t     tmp[256];

    printf("- create folder: %s\n", folder);
    make_dir(folder);
    if(chdir(folder) < 0) std_err();
    printf("- folder: %s\n", getcwd(tmp, sizeof(tmp)));
}



uint8_t *fdloadgz(uint8_t *fname) {
    struct stat xstat;
    FILE        *fd;
    int         size,
                gz = 0;
    uint8_t     *buff,
                *zbuff;

    printf("- open file %s\n", fname);
    fd = fopen(fname, "rb");
    if(!fd) std_err();

    fstat(fileno(fd), &xstat);
    size = xstat.st_size;
    if((fgetc(fd) == 0x1f) && (fgetc(fd) == 0x8b) && (fgetc(fd) == 0x08)) {
        fseek(fd, 10, SEEK_SET);
        size -= 10;
        gz = 1;
    } else {
        fseek(fd, 0, SEEK_SET);
    }

    buff = malloc(size + 1);
    if(!buff) std_err();
    fread(buff, 1, size, fd);
    fclose(fd);
    buff[size] = 0;     // delimiter

    if(gz) {
        printf("- decompress file\n");
        zbuff = unzip(buff, &size);
        if(!zbuff) return(NULL);
        free(buff);
        buff = zbuff;
    }

    return(buff);
}



uint8_t *unzip(uint8_t *in, int *insz) {
    z_stream    z;
    int         outsz,
                err;
    uint8_t     *out;
    static const int wbits = -15;

    z.zalloc = (alloc_func)0;
    z.zfree  = (free_func)0;
    z.opaque = (voidpf)0;

    if(inflateInit2(&z, wbits)) return(NULL);

    out = NULL;
    outsz = *insz * 4;

    for(;;) {
        out = realloc(out, outsz);
        if(!out) return(NULL);

        z.next_in   = in;
        z.avail_in  = *insz;
        z.next_out  = out;
        z.avail_out = outsz;

        err = inflate(&z, Z_FINISH);
        if((err == Z_OK) || (err == Z_STREAM_END)) {
            break;
        } else if(err == Z_BUF_ERROR) {
            inflateEnd(&z);
            if(inflateInit2(&z, wbits)) return(NULL);
            outsz += *insz;
        } else {
            free(out);
            return(NULL);
        }
    }

    *insz = z.total_out + 1;
    inflateEnd(&z);
    out = realloc(out, *insz);
    if(!out) return(NULL);
    out[*insz - 1] = 0;     // delimiter
    return(out);
}



void awcamrec(ids_t *ids) {
    time_t      refresh_time;
    uint32_t    crc,
                rnd,
                len,
                frame,
                idsnum;
    int         sd;
    uint8_t     mini[32],
                *buff,
                *p;

    printf("- connect to    %s : %hu\n", inet_ntoa(peer.sin_addr), ntohs(peer.sin_port));

    printf("\n"
        "frames:    . ok    x duplicated    _ not_available    = same_size              \n"
        "-------------------------------------------------------------------------------\n");

    srand(time(NULL));
    frame   = 0;
    idsnum  = 0;
    buff    = malloc(BUFFSZ);
    if(!buff) std_err();

    refresh_time = time(NULL) + REFRESH_LIST;

    for(;;) {
        rnd = rand();
        sd = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
        if(sd < 0) std_err();
        if(connect(sd, (struct sockaddr *)&peer, sizeof(peer))
          < 0) std_err();

        for(;; idsnum++) {
            if(!record) {
                if(time(NULL) > refresh_time) goto quit;
            }

            if(!ids[idsnum].id) idsnum = 0;
            if(ids[idsnum].id == -1) continue;

            p = mini;

            if(rnd) {                           // first packet needs the following data
                p += putxx(p, rnd,        32);  // our ID
                p += rnd_mem(p, 12, rnd);       // password hash
                rnd = 0;
            }
            p += putxx(p, ids[idsnum].id, 32);  // the ID of the webcam we want to see
            p += putxx(p, 0,              32);  // probably milliseconds, usually 0
            p += putxx(p, 10000,          32);  // ever 10000

            if(send(sd, mini, p - mini, 0) < 0) break;

            if(tcp_recv(sd, mini, 2) < 0) break;
            if(mini[1] == 0xff) {
                if(mini[0] == 0)    fputc('_', stdout);     // not available
                if(mini[0] == 0xff) fputc('=', stdout);     // same size
                continue;
            }

            if(tcp_recv(sd, mini, 4) < 0) break;
            getxx(mini, &len, 32);

            if(!len) {
                printf("\n- webcam ID %u seems to not exist or cannot be accessed\n", ids[idsnum].id);
                ids[idsnum].id = -1;
                if(!record) write_watch_thumb(ids);
                continue;
            } else if(len > BUFFSZ) {
                printf("\nError: the size of the frame is too big %u, possible network error\n", len);
                exit(1);
            }

            if(tcp_recv(sd, buff, len) < 0) break;

            crc = mycrc32(buff, len);
            if(crc == ids[idsnum].crc) {
                fputc('x', stdout);
            } else {
                fputc('.', stdout);
                ids[idsnum].crc = crc;
                save_file(buff, len, record ? 0 : ids[idsnum].id, frame);
                frame++;
            }
        }

        printf("\n- connection lost or timed out\n");
        close(sd);
    }

quit:
    close(sd);
    free(buff);
}



int tcp_recv(int sd, uint8_t *data, int len) {
    int     t;

    while(len) {
        if(timeout(sd) < 0) return(TIMEERR);
        t = recv(sd, data, len, 0);
        if(t <= 0) return(SOCKERR);
        data += t;
        len  -= t;
    }

    return(0);
}



int rnd_mem(uint8_t *data, int len, uint32_t rnd) {
    uint8_t      *p;
    static const uint8_t table[] =
                "0123456789"
                "ABCDEFGHIJKLMNOPQRSTUVWXYZ"
                "abcdefghijklmnopqrstuvwxyz";

    for(p = data; len; len--) {
        rnd = (rnd * 0x343FD) + 0x269EC3;
        *p++ = table[rnd % (sizeof(table) - 1)];
    }
    *p++ = 0;

    return(p - data);
}



int save_file(uint8_t *data, int len, uint32_t id, uint32_t frame) {
    FILE        *fd;
    uint8_t     fname[20];

    if(id) {
        sprintf(fname, "%u.jpg", id);
    } else {
        sprintf(fname, "%08x.jpg", frame);
    }
    fd = fopen(fname, "wb");
    if(!fd) std_err();
    fwrite(data, len, 1, fd);
    fclose(fd);
    return(0);
}



void write_watch_ever(void) {
    FILE    *fd;

    fd = fopen(WATCH_EVER, "wb");
    if(!fd) std_err();

    fprintf(fd,
        "<html>\n"
        "<title>AWCamREC</title>\n"
        "<head>\n"
        "<style type=\"text/css\"><!--*{margin:0;padding:0;}--></style>\n"
        "<script language=\"JavaScript\">\n"
        "<!--\n"
        "    var i = 0;\n"
        "    setTimeout(\"loadimg()\", 1);\n"
        "\n"
        "    function d2h(d) {\n"
        "        var hex = \"0123456789abcdef\";\n"
        "        var h = \"\";\n"
        "        var j;\n"
        "        for(j = 0; j < 8; j++) {\n"
        "            h = hex.substr(d & 15, 1) + h;\n"
        "            d >>= 4;\n"
        "        }\n"
        "        return h;\n"
        "    }\n"
        "\n"
        "    function loadimg() {\n"
        "        document.myimg.src = d2h(i) + \".jpg\";\n"
        "        i++;\n"
        "        setTimeout(\"loadimg()\", 300);\n"
        "    }\n"
        "-->\n"
        "</script>\n"
        "</head><body>\n"
        "<img name=\"myimg\" src=\"00000000.jpg\" width=100%% border=0>\n"
        "</body></html>\n");

    fclose(fd);
}



uint32_t mysqrt(uint32_t num) {
     uint32_t   ret    = 0,
                ret_sq = 0,
                b;
     int        s;

     for(s = (32 >> 1) - 1; s >= 0; s--) {
         b = ret_sq + ((uint32_t)1 << (s << 1)) + ((ret << s) << 1);
         if(b <= num) {
             ret_sq = b;
             ret += (uint32_t)1 << s;
         }
     }

     return(ret);
}



void write_watch_thumb(ids_t *ids) {
    FILE        *fd;
    int         idsnum,
                horiz;

    fd = fopen(WATCH_THUMB, "wb");
    if(!fd) std_err();

    fprintf(fd,
        "<html>"
        "<title>AWCamREC</title>"
        "<head><meta http-equiv=\"Refresh\" Content=\"1\"></head>"
        "<style type=\"text/css\"><!--*{margin:0;padding:0;}--></style>\n"
        "<body>");

    for(idsnum = 0; ids[idsnum].id; idsnum++) {
        if(ids[idsnum].id == -1) continue;
    }
    horiz = 100 / mysqrt(idsnum);

    for(idsnum = 0; ids[idsnum].id; idsnum++) {
        if(ids[idsnum].id == -1) continue;
        fprintf(fd,
            "<a href=\"%u.jpg\"><img src=\"%u.jpg\" longdesc=\"%s\" width=%d%% border=0></a>",
            ids[idsnum].id, ids[idsnum].id, ids[idsnum].name, horiz);
    }

    fprintf(fd, "</body></html>");
    fclose(fd);
}



int getxx(uint8_t *data, uint32_t *ret, int bits) {
    uint32_t    num;
    int         i,
                bytes;

    bytes = bits >> 3;

    for(num = i = 0; i < bytes; i++) {
        num |= (data[i] << ((bytes - i - 1) << 3));
    }

    *ret = num;
    return(bytes);
}



int putxx(uint8_t *data, uint32_t num, int bits) {
    int         i,
                bytes;

    bytes = bits >> 3;

    for(i = 0; i < bytes; i++) {
        data[i] = num >> ((bytes - i - 1) << 3);
    }

    return(bytes);
}



int timeout(int sock) {
    struct  timeval tout;
    fd_set  fd_read;
    int     err;

    tout.tv_sec  = 5;   // five seconds are enough
    tout.tv_usec = 0;
    FD_ZERO(&fd_read);
    FD_SET(sock, &fd_read);
    err = select(sock + 1, &fd_read, NULL, NULL, &tout);
    if(err < 0) std_err();
    if(!err) return(-1);
    return(0);
}



void show_help(void) {
    printf(
        "Options:\n"
        "-r ID    record the webcam identified by ID (it's the same of specifying the\n"
        "         webcam_ID as last argument). You must use the ID number NOT the name!\n"
        "-f FLT   show the webcams in the FLT filter:\n"
        "         f=female, m=male, g=groups, u=unknown, o=others, nicknames\n"
        "         example for female and group and hot or ita in the nick: -f fg,hot,ita\n"
        "         example for the webcams with hot or ita in the nick:     -f ,hot,ita\n"
        "-c FILE  load the local camstest file FILE, compressed or not is the same.\n"
        "         by default the "CAMSTEST_FD" file is downloaded all the times from:\n"
        "         "CAMSTEST_WWW"\n"
        "-l       list the webcams and exit\n"
        "-i       automatically launch the browser for watching the webcams\n"
        "\n"
        "Note: the webcam is recorded ONLY with the -r option or with ID argument!\n"
        "\n");
    exit(1);
}



uint32_t resolv(char *host) {
    struct      hostent *hp;
    uint32_t    host_ip;

    host_ip = inet_addr(host);
    if(host_ip == INADDR_NONE) {
        hp = gethostbyname(host);
        if(!hp) {
            printf("\nError: Unable to resolve hostname (%s)\n", host);
            exit(1);
        } else host_ip = *(uint32_t *)hp->h_addr;
    }
    return(host_ip);
}



#ifndef WIN32
    void std_err(void) {
        perror("\nError");
        exit(1);
    }
#endif


