/*
    Copyright 2008 Luigi Auriemma

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    http://www.gnu.org/licenses/gpl-2.0.txt
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <ctype.h>
#include <sys/stat.h>

#ifdef WIN32
#else
    #define stricmp strcasecmp
#endif

typedef uint8_t     u8;
typedef uint16_t    u16;
typedef uint32_t    u32;



#define VER         "0.1"



u8 *get_ext(u8 *fname);
void bsr_decrypt(u8 *fname, u8 *data, int datasz);
int check_overwrite(u8 *fname);
u8 *fdload(u8 *fname, int *fsize);
void fddump(u8 *fname, u8 *buff, int buffsz);
void std_err(void);



int main(int argc, char *argv[]) {
    int     buffsz;
    u8      *buff,
            *fin,
            *fout;

    setbuf(stdout, NULL);

    fputs("\n"
        "Big Scale Racing files decoder "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", stdout);

    if(argc < 3) {
        printf("\n"
            "Usage: %s <file> <output_file>\n"
            "\n", argv[0]);
        exit(1);
    }
    fin  = argv[1];
    fout = argv[2];

    buff = fdload(fin, &buffsz);

    bsr_decrypt(fin, buff, buffsz);

    fddump(fout, buff, buffsz);

    printf("- %d bytes decoded\n", buffsz);
    free(buff);
    return(0);
}



u8 *get_ext(u8 *fname) {
    u8      *p;

    p = strrchr(fname, '.');
    if(!p) return(fname + strlen(fname));
    return(p + 1);
}



void bsr_decrypt(u8 *fname, u8 *data, int datasz) {
    static const u8 table[] = "\x47\xC3\xF5\x12\x38\xE9\xB5\x91\x25\x63\x06\xD9\xAA\x6F\x3A\x73";
    int     i;
    u8      *ext;

    ext = get_ext(fname);
    if(!stricmp(ext, "FS3")) {
        printf("- the output file is a MP3\n");
    } else if(!stricmp(ext, "FSW")) {
        printf("- the output file is a WAV\n");
        data   += 0x2e; // quick way
        datasz -= 0x2e;
    } else if(!stricmp(ext, "FSP")) {
        data   += 0x14;
        datasz -= 0x14;
    } else if(!stricmp(ext, "FSM") || !stricmp(ext, "FSO")) {
        printf("- the files with this extension are not encrypted, I quit\n");
        exit(1);
    }

    for(i = 0; i < datasz; i++) {
        data[i] ^= table[i & 15];
    }
}



int check_overwrite(u8 *fname) {
    FILE    *fd;
    u8      ans[16];

    fd = fopen(fname, "rb");
    if(!fd) return(0);
    fclose(fd);
    printf("  the file already exists, do you want to overwrite it (y/N)? ");
    fgets(ans, sizeof(ans), stdin);
    if(tolower(ans[0]) != 'y') return(-1);
    return(0);
}



u8 *fdload(u8 *fname, int *fsize) {
    struct stat xstat;
    FILE    *fd;
    int     size;
    u8      *buff;

    printf("- open %s\n", fname);
    fd = fopen(fname, "rb");
    if(!fd) std_err();
    fstat(fileno(fd), &xstat);
    size = xstat.st_size;
    buff = malloc(size);
    if(!buff) std_err();
    fread(buff, 1, size, fd);
    fclose(fd);
    *fsize = size;
    return(buff);
}



void fddump(u8 *fname, u8 *buff, int buffsz) {
    FILE    *fd;

    printf("- create file %s\n", fname);
    if(check_overwrite(fname) < 0) exit(1);
    fd = fopen(fname, "wb");
    if(!fd) std_err();
    fwrite(buff, 1, buffsz, fd);
    fclose(fd);
}



void std_err(void) {
    perror("\nError");
    exit(1);
}


