/*
    Copyright 2009 Luigi Auriemma

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    http://www.gnu.org/licenses/gpl-2.0.txt
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <time.h>
#include <errno.h>
#include <sys/stat.h>
#include <ctype.h>
#include "mycrc32.h"
#include "mydownlib.h"

#ifdef WIN32
    #include <winsock.h>
    #include <direct.h>

    #define close       closesocket
    #define sleep       Sleep
    #define usleep      sleep
    #define ONESEC      1000
    #define check_time  clock()
    #define make_dir(x) mkdir(x)
#else
    #include <unistd.h>
    #include <sys/socket.h>
    #include <sys/types.h>
    #include <arpa/inet.h>
    #include <netinet/in.h>
    #include <netdb.h>
    #include <sys/stat.h>
    #include <sys/times.h>
    #include <sys/timeb.h>

    #define stricmp     strcasecmp
    #define strnicmp    strncasecmp
    #define stristr     strcasestr
    #define ONESEC      1
    #define check_time  (times(0) * 10)
    #define make_dir(x) mkdir(x, 0755)
#endif

typedef uint8_t     u8;
typedef uint16_t    u16;
typedef uint32_t    u32;



#define VER             "0.1"
#define HOST            "relay.cameraware.com"
#define PORT            80

#define CAMLIST_FD      "all_cat.cfg"
#define CAMLIST_WWW     HOST "/cwjava/" CAMLIST_FD "?1"

#define WATCH_THUMB     "watch_thumb.htm"
#define WATCH_EVER      "watch_ever.htm"
#define WATCH_NOW       "watch_now.htm"

#define REFRESH_LIST    300

#define SOCKERR         (-1)
#define TIMEERR         (-2)



typedef struct {
    u8      *name;
    u32     crc;
    u8      *url;
    u8      disabled;
} ids_t;



int     record = 0;



u8 *SLDecrypt(u8 *data, u32 *datalen);
void check_filter(u8 *p);
u8 *myitoa(int num);
u8 *get_date(void);
void launch_browser(u8 *fname);
ids_t *getids(u8 *buff, u8 *filter);
void myfree(u8 **buff);
u8 *fgetz(u8 **buff);
void create_folder(u8 *folder);
u8 *fdload(u8 *fname);
void camerawarec(ids_t *ids);
int save_file(u8 *data, int len, u8 *id, u32 frame);
u32 mysqrt(u32 num);
void write_watch_ever(void);
void write_watch_now(u8 *fname);
void write_watch_thumb(ids_t *ids);
int timeout(int sock);
void show_help(void);
void std_err(void);



int main(int argc, char *argv[]) {
    mydown_options  opt;
    ids_t   *ids;
    int     i,
            list        = 0,
            run_browser = 0;
    u8      *fname      = CAMLIST_FD,
            *filter     = NULL,
            *camlist    = NULL,
            *webcam_id  = NULL,
            *p;

#ifdef WIN32
    WSADATA    wsadata;
    WSAStartup(MAKEWORD(1,0), &wsadata);
#endif

    setbuf(stdout, NULL);

    fputs("\n"
        "CameraWaREC "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", stderr);

    if(argc < 2) {
        printf("\n"
            "Usage %s [options] [nickname/URL]\n"
            "\n", argv[0]);
        show_help();
    }

    for(i = 1; i < argc; i++) {
        if(((argv[i][0] != '-') && (argv[i][0] != '/')) || (strlen(argv[i]) != 2)) {
            if(i == (argc - 1)) {
                webcam_id = argv[i];
                break;
            }
            fprintf(stderr, "\nError: wrong command-line argument (%s)\n", argv[i]);
            exit(1);
        }
        switch(argv[i][1]) {
            case 'r': {
                if(!argv[++i]) show_help();
                webcam_id         = argv[i];
                } break;
            case 'f': {
                if(!argv[++i]) show_help();
                filter            = argv[i];
                check_filter(filter);
                } break;
            case 'c': {
                if(!argv[++i]) show_help();
                fname             = argv[i];
                } break;
            case 'l': list        = 1;                  break;
            case 'i': run_browser = 1;                  break;
            default: {
                printf("\nError: wrong command-line argument (%s)\n", argv[i]);
                exit(1);
            }
        }
    }

    memset(&opt, 0, sizeof(opt));
    opt.useragent  = "CameraWare Viewer";
    opt.verbose    = 1;
    opt.onflyunzip = 1;

    if(webcam_id) {
        if(strstr(webcam_id, "://")) {  // it's the url
            ids = calloc(sizeof(ids_t), 2);
            ids[0].name = strdup(webcam_id);
            ids[0].crc  = 0;
            ids[0].url  = webcam_id;
            p = strrchr(ids[0].name, '/');
            if(p) {
                ids[0].name = p + 1;
                p = strrchr(ids[0].name, '.');
                if(p) *p = 0;
            }
        } else {
            if(fname == (u8 *)CAMLIST_FD) {
                mydown(CAMLIST_WWW, fname, &opt);
            }

            camlist = fdload(fname);
            if(!camlist) {
                printf("\nError: something wrong during the loading of the camlist file\n");
                exit(1);
            }
            // unlink(fname);  // remove the temporary file... I prefer to not remove files, even the temporaries

            filter = malloc(1 + strlen(webcam_id) + 1);
            sprintf(filter, ",%s", webcam_id);

            ids = getids(camlist, filter);
            for(i = 0; ids[i].name; i++);
            if(i != 1) {
                printf("\n"
                    "Error: the webcam id/nickname is used by %d webcams\n"
                    "       you must specify a more exact nickname\n",
                    i);
                exit(1);
            }
        }
        webcam_id = ids[0].name;

        printf("- start recording of the webcam %s\n", webcam_id);

        create_folder(webcam_id);
        create_folder(get_date());
        write_watch_ever();
        if(run_browser) launch_browser(WATCH_EVER);

        record = 1;

        camerawarec(ids);

        goto quit;
    }

    printf("- start thumbnails collection\n");

    create_folder("thumb");

    for(;;) {
        if(fname == (u8 *)CAMLIST_FD) {
            mydown(CAMLIST_WWW, fname, &opt);
        }

        if(camlist) free(camlist);
        camlist = fdload(fname);
        if(!camlist) {
            printf("\nError: something wrong during the loading of the camlist file\n");
            exit(1);
        }

        ids = getids(camlist, filter);

        if(list) goto quit;

        write_watch_thumb(ids);
        if(run_browser) {
            launch_browser(WATCH_THUMB);
            run_browser = 0;
        }

        camerawarec(ids);
    }

quit:
    free(ids);
    return(0);
}



u8 *SLDecrypt(u8 *data, u32 *datalen) {
    int     endian = 1; // big endian
    u32     i,
            len,
            xor,
            sub,
            rem,
            *p32;
    u8      *p;

    p32 = (u32 *)data;
    len = *datalen >> 2;

    if(!*(char *)&endian) {     // big endian compatibility
        for(p = data, i = 0; i < len; i++, p += 4) {
            p32[i] = (p[3] << 24) | (p[2] << 16) | (p[1] << 8) | p[0];
        }
    }

    xor = p32[0x7e] ^ *datalen;
    for(i = 0; i < len; i++) {
        if(i == 0x7e) continue; // useless, skip the xor number at 0x7e
        p32[i] ^= xor;
    }

    sub = p32[0x7f];
    for(i = 0; i < len; i++) {
        if(i == 0x7e) continue; // useless, skip the xor number at 0x7e
        p32[i] -= sub;
    }

    if(p32[1] > *datalen) {     // useless compatibility
        p32[1] = 0x400;
        if(p32[1] > *datalen) {
            p32[1] = 0x200;
            if(p32[1] > *datalen) p32[1] = 0;
        }
    }
    rem = p32[1];

    if(!*(char *)&endian) {     // big endian compatibility
        for(p = data, i = 0; i < len; i++, p += 4) {
            p32[i] = (p[3] << 24) | (p[2] << 16) | (p[1] << 8) | p[0];
        }
    }

    *datalen -= rem;
    return(data + rem);
}



void check_filter(u8 *p) {
    for(; *p && (*p != ','); p++) {
        *p = tolower(*p);
        switch(*p) {
            case 'f':
            case 'c':
            case 'm':
            case 'p':
            case 'e':
            case 't': break;
            default: {
                printf("\nError: wrong filter, recheck it\n");
                exit(1);
                } break;
        }
    }
}



u8 *myitoa(int num) {
    static u8   mini[12];
    sprintf(mini, "%u", num);
    return(mini);
}



u8 *get_date(void) {
    time_t  datex;
    struct  tm  *tmx;
    static u8  folder[32];
    static const u8 *months[12] = {
                "jan","feb","mar","apr","may","jun",
                "jul","aug","sep","oct","nov","dec"
            };

    time(&datex);
    tmx = localtime(&datex);
    sprintf(
        folder,
        "%02hhu.%s.%02hhu-%02hhu.%02hhu",
        tmx->tm_mday, months[tmx->tm_mon], tmx->tm_year % 100,
        tmx->tm_hour, tmx->tm_min);

    return(folder);
}



void launch_browser(u8 *fname) {
    u8      tmp[256];

#ifdef WIN32
    getcwd(tmp, sizeof(tmp) - strlen(fname));
    sprintf(tmp + strlen(tmp), "\\%s", fname);
    printf("- launch browser: %s\n", tmp);

    ShellExecute(
        NULL,
        "open",
        tmp,
        NULL,
        NULL,
        SW_SHOW);
#else
    strcpy(tmp, "$BROWSER ");
    getcwd(tmp + 9, sizeof(tmp) - 9 - strlen(fname) - 3);
    sprintf(tmp + strlen(tmp), "/%s &", fname);
    printf("- launch browser: %s\n", tmp);

    system(tmp);
#endif
}



ids_t *getids(u8 *buff, u8 *filter) {
    ids_t   *ids;
    u32     type    = 0,
            idsnum;
    int     ok,
            id      = 0,
            ready   = 0;
    u8      *nick   = "",
            *data   = NULL,
            *p      = NULL,
            *l      = NULL,
            *param  = NULL,
            *value  = NULL,
            *url    = NULL;

    idsnum = 1;                 // full allocation
    for(p = buff; *p; p++) {
        if(*p == '\n') idsnum++;
    }
    ids = calloc(sizeof(ids_t), idsnum + 1);
    if(!ids) std_err();
    idsnum = 0;

    printf("- webcams:\n\n");

    while((data = fgetz(&buff))) {

#define DOIT(X,Y)   p = strchr(data, X); \
                    if(!p) continue;     \
                    *p = 0;              \
                    Y;                   \
                    data = p + 1;

        DOIT('=', param   = data)
                  value   = data;   // do not touch data, ready for DOIT

        if(!strnicmp(param, "image", 5)) {
            id  = atoi(param + 5);  // in case there is no nick
            url = value;
            ready++;
        } else if(!strnicmp(param, "channelinfo", 11)) {
            DOIT(',', )
            DOIT(',', type    = tolower(data[0]))
            ready++;
        } else if(!strnicmp(param, "label", 5)) {
            nick = value;
            p = strrchr(nick, '{');
            if(p) {
                id = atoi(p + 1);
                //*p = 0;
                //p--;
                //if(*p <= ' ') *p = 0;
            }
            ready++;
        }
        if(ready < 3) continue;

#undef DOIT
        ready = 0;  // needed

        if(filter) {
            ok = 0;
            p = strchr(filter, ',');
            if(p) *p = 0;

            if(strchr(filter, type)) ok = 1;    // if type is zero it's catched automatically

            if(p) {
                *p = ',';
                for(p++; !ok; p = l + 1) {
                    l = strchr(p, ',');
                    if(l) *l = 0;
                    if(stristr(nick, p)) ok = 1;
                    if(!l) break;
                    *l = ',';
                }
            }

            if(!ok) continue;
        }

        printf("  %-3u %s\n", id, nick);
        ids[idsnum].name = nick;
        ids[idsnum].crc  = 0;
        ids[idsnum].url  = url;
        idsnum++;

        url   = NULL;
        type  = 0;
    }

    if(!idsnum) {
        printf("\n- no webcams in the filter\n");
        free(ids);
        exit(1);
    }

    printf("\n- %u webcams in the filter\n", idsnum);
    return(ids);
}



void myfree(u8 **buff) {
    if(!*buff) return;
    free(*buff);
    *buff = NULL;
}



u8 *fgetz(u8 **buff) {
    u8      *ret,
            *p;

    for(ret = p = *buff; *p && (*p != '\n') && (*p != '\r'); p++);
    if(!(p - ret)) return(NULL);
    *p = 0;
    *buff = p + 1;
    return(ret);
}



void create_folder(u8 *folder) {
    u8      tmp[256];

    printf("- create folder: %s\n", folder);
    make_dir(folder);
    if(chdir(folder) < 0) std_err();
    printf("- folder: %s\n", getcwd(tmp, sizeof(tmp)));
}



u8 *fdload(u8 *fname) {
    struct stat xstat;
    FILE    *fd;
    int     size;
    u8      *buff;

    printf("- open file %s\n", fname);
    fd = fopen(fname, "rb");
    if(!fd) std_err();

    fstat(fileno(fd), &xstat);
    size = xstat.st_size;

    buff = malloc(size + 1);
    if(!buff) std_err();
    fread(buff, 1, size, fd);
    fclose(fd);
    buff[size] = 0;     // delimiter

    return(buff);
}



void camerawarec(ids_t *ids) {
    mydown_options  opt;
    time_t  refresh_time,
            onesec_time;
    u32     crc,
            frame,
            idsnum;
    int     sd,
            len,
            code,
            waitsec,
            seq;
    u8      seq_url[2000],
            *buff,
            *jpg;

    printf("\n"
        "frames:    . ok    x duplicated    ? timeout_or_other_network_problems         \n"
        "-------------------------------------------------------------------------------\n");

    srand(time(NULL));
    frame       = 0;
    idsnum      = 0;
    sd          = 0;

    memset(&opt, 0, sizeof(opt));
    opt.verbose    = -1;
    opt.useragent  = "CameraWare Viewer";
    opt.filedata   = &buff;
    opt.keep_alive = &sd;
    opt.timeout    = 20,   // max 15
    opt.ret_code   = &code;
    opt.onflyunzip = 1;

    refresh_time = time(NULL) + REFRESH_LIST;
    onesec_time  = time(NULL) + 1;

    seq = 10000000;
    for(;;) {
        for(;; idsnum++) {
            if(!record) {
                if(time(NULL) > refresh_time) goto quit;
            }

            if(!ids[idsnum].name) {
                idsnum = 0;
                if((time(NULL) - onesec_time) < 1) sleep(ONESEC);
                onesec_time = time(NULL) + 1;
            }
            if(ids[idsnum].disabled) continue;

            if(!idsnum) seq++;
            sprintf(seq_url, "%s?seq=%u", ids[idsnum].url, seq);
            len = mydown(seq_url, NULL, &opt);
            //len = mydown(ids[idsnum].url, NULL, &opt);

            if(len == MYDOWN_ERROR) {
                if(code == 403) {
                    printf("- %s temporary disabled\n", ids[idsnum].name);
                    ids[idsnum].disabled = 1;
                    continue;
                }

                if((code == 404) || (code == 403)) {
                    if(record) {
                        printf("- webcam temporary offline or protected, wait mode... ");
                        for(waitsec = 15; waitsec; waitsec--) {
                            printf("%-3d\b\b\b", waitsec);
                            sleep(ONESEC);
                        }
                        printf("\n");
                    } else {
                        printf("- %s temporary disabled\n", ids[idsnum].name);
                        ids[idsnum].disabled = 1;
                    }
                } else {
                    // timeout webcams are rescanned later
                    fputc('?', stdout);
                }
                continue;
            }
            if(!len) {
                idsnum--;
                sd = 0;
                continue;
            }

            jpg = SLDecrypt(buff, &len);
            crc = mycrc32(jpg, len);
            if(crc == ids[idsnum].crc) {
                fputc('x', stdout);
            } else {
                fputc('.', stdout);
                ids[idsnum].crc = crc;
                save_file(jpg, len, record ? NULL : ids[idsnum].name, frame);
                frame++;
            }

            myfree((void *)&buff);
        }

        printf("\n- connection lost or timed out\n");
    }

quit:
    if(sd) close(sd);
    myfree((void *)&buff);
}



int save_file(u8 *data, int len, u8 *id, u32 frame) {
    FILE    *fd;
    u8      fname[128];

    if(id) {
        sprintf(fname, "%.*s.jpg", sizeof(fname) - 5, id);
    } else {
        sprintf(fname, "%08x.jpg", frame);
        write_watch_now(fname);
    }
    fd = fopen(fname, "wb");
    if(!fd) std_err();
    fwrite(data, len, 1, fd);
    fclose(fd);
    return(0);
}



void write_watch_ever(void) {
    FILE    *fd;

    fd = fopen(WATCH_EVER, "wb");
    if(!fd) std_err();
    printf(
        "- Note: if you use Firefox remember to set to FALSE the parameter\n"
        "  dom.disable_image_src_set available in about:config or "WATCH_EVER"\n"
        "  will not work\n");

    fprintf(fd,
        "<html>\n"
        "<title>CameraWaREC</title>\n"
        "<head>\n"
        "<style type=\"text/css\"><!--*{margin:0;padding:0;}--></style>\n"
        "<script language=\"JavaScript\">\n"
        "<!--\n"
        "    var i = 0;\n"
        "    setTimeout(\"loadimg()\", 1);\n"
        "\n"
        "    function d2h(d) {\n"
        "        var hex = \"0123456789abcdef\";\n"
        "        var h = \"\";\n"
        "        var j;\n"
        "        for(j = 0; j < 8; j++) {\n"
        "            h = hex.substr(d & 15, 1) + h;\n"
        "            d >>= 4;\n"
        "        }\n"
        "        return h;\n"
        "    }\n"
        "\n"
        "    function loadimg() {\n"
        "        document.myimg.src = d2h(i) + \".jpg\";\n"
        "        i++;\n"
        "        setTimeout(\"loadimg()\", 300);\n"
        "    }\n"
        "-->\n"
        "</script>\n"
        "</head><body>\n"
        "<img name=\"myimg\" src=\"00000000.jpg\" width=100%% border=0>\n"
        "</body></html>\n");

    fclose(fd);
}



void write_watch_now(u8 *fname) {
    FILE    *fd;

    fd = fopen(WATCH_NOW, "wb");
    if(!fd) return;

    fprintf(fd,
        "<html>"
        "<title>CameraWaREC</title>"
        "<head>"
        "<style type=\"text/css\"><!--*{margin:0;padding:0;}--></style>"
        "<meta http-equiv=\"Refresh\" Content=\"1\">"
        "</head>"
        "<body>"
        "<img src=\"%s\" width=100%%>\n"
        "</body>"
        "</html>",
        fname);

    fclose(fd);
}



u32 mysqrt(u32 num) {
     u32    ret    = 0,
            ret_sq = 0,
            b;
     int    s;

     for(s = (32 >> 1) - 1; s >= 0; s--) {
         b = ret_sq + ((u32)1 << (s << 1)) + ((ret << s) << 1);
         if(b <= num) {
             ret_sq = b;
             ret += (u32)1 << s;
         }
     }

     return(ret);
}



void write_watch_thumb(ids_t *ids) {
    FILE    *fd;
    int     idsnum,
            horiz;

    fd = fopen(WATCH_THUMB, "wb");
    if(!fd) std_err();

    fprintf(fd,
        "<html>"
        "<title>CameraWaREC</title>"
        "<head><meta http-equiv=\"Refresh\" Content=\"1\"></head>"
        "<style type=\"text/css\"><!--*{margin:0;padding:0;}--></style>\n"
        "<body>");

    for(idsnum = 0; ids[idsnum].name; idsnum++);
    horiz = 100 / mysqrt(idsnum);

    for(idsnum = 0; ids[idsnum].name; idsnum++) {
        fprintf(fd,
            "<a href=\"%s.jpg\"><img src=\"%s.jpg\" width=%d%% border=0></a>",
            ids[idsnum].name, ids[idsnum].name, horiz);
    }

    fprintf(fd, "</body></html>");
    fclose(fd);
}



int timeout(int sock) {
    struct  timeval tout;
    fd_set  fd_read;
    int     err;

    tout.tv_sec  = 5;   // five seconds are enough
    tout.tv_usec = 0;
    FD_ZERO(&fd_read);
    FD_SET(sock, &fd_read);
    err = select(sock + 1, &fd_read, NULL, NULL, &tout);
    if(err < 0) std_err();
    if(!err) return(-1);
    return(0);
}



void show_help(void) {
    printf(
        "Options:\n"
        "-r ID    record the webcam identified by ID (it's the same of specifying the\n"
        "         webcam_ID as last argument), ID is just the full or partial nickname\n"
        "-f FLT   show the webcams in the FLT filter:\n"
        "         f=female, c=couples, m=male, p=places, e=events, t=things, nicknames\n"
        "         example for female, couple and hot or ita in the nick:   -f fc,hot,ita\n"
        "         example for the webcams with hot or ita in the nick:     -f ,hot,ita\n"
        "-c FILE  load the local camlist file FILE, compressed or not is the same.\n"
        "         by default the "CAMLIST_FD" file is downloaded all the times from:\n"
        "         "CAMLIST_WWW"\n"
        "-l       list the webcams and exit, useful to get the nickname to record\n"
        "-i       automatically launch the browser for watching the webcams\n"
        "\n"
        "Note: the webcam is recorded ONLY with the -r option or with ID argument!\n"
        "\n");
    exit(1);
}



void std_err(void) {
    perror("\nError");
    exit(1);
}


