/*
    Copyright 2007,2008,2009 Luigi Auriemma

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    http://www.gnu.org/licenses/gpl-2.0.txt
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <sys/stat.h>
#include <ctype.h>
#include "unlzss.h"

#ifdef WIN32
    #include <direct.h>
    #define PATHSLASH   '\\'
    #define make_dir(x) mkdir(x)
#else
    #include <unistd.h>
    #define stricmp     strcasecmp
    #define PATHSLASH   '/'
    #define make_dir(x) mkdir(x, 0755)
#endif

typedef uint8_t     u8;
typedef uint16_t    u16;
typedef uint32_t    u32;



#define VER         "0.2a"



int cauldron_dump(FILE *fd, FILE *fdo, u8 *out, int outsz, int zlen);
int cauldron_extract(FILE *fd);
void xor(u8 *data, int len, u8 chr);
u8 **get_names(FILE *fd, int len, int num);
u32 fgetxx(FILE *fd, int bytes);
u8 *create_dir(u8 *name);
int check_wildcard(u8 *fname, u8 *wildcard);
int check_overwrite(u8 *fname);
void myalloc(u8 **data, unsigned wantsize, unsigned *currsize);
void myfw(FILE *fd, void *data, unsigned size);
void myfr(FILE *fd, void *data, unsigned size);
void std_err(void);



int     listonly    = 0,
        overwrite   = 0;
u8      *gwildcard  = NULL;
int     xor_chr     = 0;



int main(int argc, char *argv[]) {
    FILE    *fd;
    int     i,
            files;
    u8      *fname,
            *fdir;

    setbuf(stdout, NULL);

    fputs("\n"
        "Cauldron FS files extractor "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", stdout);

    if(argc < 3) {
        printf("\n"
            "Usage: %s [options] <file.FS> <output_folder>\n"
            "\n"
            "Options:\n"
            "-l      list files without extracting them\n"
            "-f \"W\"  extract only the files which match the wildcard W like \"*.mp3\"\n"
            "-o      overwrites existent files without asking\n"
            "-x NUM  the byte used for obfuscating the compressed demo files.\n"
            "        the demo of Battle Isle uses 0x48 while the one of Chaser 0x44\n"
            "\n", argv[0]);
        exit(1);
    }

    argc -= 2;
    for(i = 1; i < argc; i++) {
        if(((argv[i][0] != '-') && (argv[i][0] != '/')) || (strlen(argv[i]) != 2)) {
            printf("\nError: wrong argument (%s)\n", argv[i]);
            exit(1);
        }
        switch(argv[i][1]) {
            case 'l': listonly  = 1;                    break;
            case 'f': gwildcard = argv[++i];            break;
            case 'o': overwrite = 1;                    break;
            case 'x': sscanf(argv[++i], "%x", &xor_chr);break;
            default: {
                printf("\nError: wrong argument (%s)\n", argv[i]);
                exit(1);
            }
        }
    }
    fname = argv[argc];
    fdir  = argv[argc + 1];

    printf("- open %s\n", fname);
    fd = fopen(fname, "rb");
    if(!fd) std_err();

    if(!listonly) {
        printf("- set output folder %s\n", fdir);
        if(chdir(fdir) < 0) std_err();
    }

    printf("- start %s:\n", listonly ? "listing" : "extraction");

    files = cauldron_extract(fd);

    fclose(fd);
    printf("\n- %s %d files\n", listonly ? "listed" : "extracted", files);
    return(0);
}



int cauldron_dump(FILE *fd, FILE *fdo, u8 *out, int outsz, int zlen) {
    static u8   *zbuff = NULL;
    int     len;

    if(!zbuff) {
        zbuff = malloc(outsz);
        if(!zbuff) std_err();
    }

    if(zlen > outsz) {
        printf("\nError: there is something wrong with a compressed size %u (max %u)\n", zlen, outsz);
        exit(1);
    }

    if(zlen) {  // chaser:0048fc58 gt:100e5051
        myfr(fd, zbuff, zlen);
        len = unlzss(out, outsz, zbuff, zlen);
        if(xor_chr) xor(out, len, xor_chr);
    } else {
        zlen = outsz;
        len  = outsz;
        myfr(fd, out, len);
    }

    myfw(fdo, out, len);
    return(zlen);
}



int cauldron_extract(FILE *fd) {
    FILE    *fdo;
    u32     len,
            zlen,
            files,
            chunk,
            current_offset,
            offset,
            zoffset,
            section_size,
            sector_size,
            directories_len,
            directories_num,
            filenames_len,
            filenames_num,
            comp_len,
            file_len,
            folder,
            compress,
            xnum;
    int     fnamesz         = 0,
            fsvercheck      = 0,
            tot             = 0,
            fsver           = 1,        // default for 1
            nsz             = 2,        // default for 1
            buffsz          = 0x4000;   // default for 1
    u8      *fname          = NULL,
            **directories   = NULL,
            **filenames     = NULL,
            *zbuff          = NULL,
            *buff           = NULL;

    if(fseek(fd, -4, SEEK_END)) std_err();
    offset = fgetxx(fd, 4);
    if(fseek(fd, offset, SEEK_SET)) std_err();

    section_size    = fgetxx(fd, 4);
    sector_size     = fgetxx(fd, 4);
    directories_len = fgetxx(fd, 4);
    directories_num = fgetxx(fd, 4);
    filenames_len   = fgetxx(fd, 4);
    filenames_num   = fgetxx(fd, 4);

    printf("\n"
        "  info offset     %08x\n"
        "  section_size    %08x\n"
        "  sector_size     %08x\n"
        "  directories_len %08x\n"
        "  directories_num %08x\n"
        "  filenames_len   %08x\n"
        "  filenames_num   %08x\n"
        "\n",
        offset,
        section_size,
        sector_size,
        directories_len,
        directories_num,
        filenames_len,
        filenames_num);

    directories = get_names(fd, directories_len, directories_num);
    filenames   = get_names(fd, filenames_len,   filenames_num);

    current_offset = ftell(fd);
redo:
    if(fseek(fd, current_offset, SEEK_SET)) std_err();
    if(fsvercheck) {
        printf("- use FS compatibility version %d: %d %08x\n", fsver, nsz * 8, buffsz);
        zbuff = malloc(buffsz);
        buff  = malloc(buffsz);
        if(!zbuff || !buff) std_err();
        if(xor_chr) printf("- all the compressed files will be XORed with 0x%02x\n", xor_chr);
        printf("\n"
            "  offset   size       filename\n"
            "  ----------------------------\n");
    }

    for(files = 0; files < filenames_num; files++) {
        offset   = fgetxx(fd, 4);
        comp_len = fgetxx(fd, 4);
        file_len = fgetxx(fd, 4);
        folder   = fgetxx(fd, nsz);
        compress = fgetxx(fd, nsz);
        compress &= 0xff;

        if(!fsvercheck) {       // guess the version of the FS archive
            if(files < 3) {     // check at least 3 files
                if((folder > directories_num) || (compress > 3)) {
                    fsver++;
                    switch(fsver) {
                        case 1: {   // chaser and the others
                            nsz     = 2;
                            buffsz  = 0x4000;
                            } break;
                        case 2: {   // battle isle
                            nsz     = 4;
                            buffsz  = 0x8000;
                            } break;
                        default: {
                            printf("\nError: unsupported FS file version, contact me\n");
                            exit(1);
                            } break;
                    }
                    printf("- change FS compatibility to version %d\n", fsver);
                } else {
                    continue;
                }
            } else {
                fsvercheck = 1; // start the real extraction/listing
            }
            goto redo;
            continue;
        }

        #ifdef DEBUG
        printf("\n"
            "  offset   %08x\n"
            "  comp_len %08x\n"
            "  file_len %08x\n"
            "  folder   %08x\n"
            "  compress %08x\n",
            offset, comp_len, file_len, folder, compress);
        #endif

        myalloc(&fname, strlen(directories[folder]) + 1 + strlen(filenames[files]) + 1, &fnamesz);
        sprintf(fname, "%s\\%s", directories[folder], filenames[files]);

        if(check_wildcard(fname, gwildcard) < 0) continue;
        printf("  %08x %-10u %s\n", offset, file_len, fname);
        if(listonly) {
            tot++;
            continue;
        }

        if(!overwrite && (check_overwrite(fname) < 0)) continue;
        create_dir(fname);
        fdo = fopen(fname, "wb");
        if(!fdo) std_err();

        current_offset = ftell(fd);
        if(fseek(fd, offset, SEEK_SET)) std_err();

        if(!compress) {
            for(len = buffsz; file_len; file_len -= len) {
                if(file_len < len) len = file_len;
                myfr(fd, buff, len);
                myfw(fdo, buff, len);
            }

        } else if(compress == 2) {
            for(chunk = 0; ; chunk++) {
                if(fseek(fd, offset + (chunk * (4 + nsz + nsz)), SEEK_SET)) std_err();

                zoffset = fgetxx(fd, 4);    // chunk offset
                xnum    = fgetxx(fd, nsz);  // not important, this value is not handled even by the game
                zlen    = fgetxx(fd, nsz);  // chunk length

                if(fseek(fd, offset + zoffset, SEEK_SET)) std_err();
                zlen = cauldron_dump(fd, fdo, buff, buffsz, zlen);
                if((zoffset + zlen) >= comp_len) break;
            }

        } else if(compress == 3) {
            for(chunk = 0; ; chunk++) {
                if(fseek(fd, offset + (chunk * (4 + nsz)), SEEK_SET)) std_err();

                zoffset = fgetxx(fd, 4);    // chunk offset
                zlen    = fgetxx(fd, nsz);  // chunk length

                if(fseek(fd, offset + zoffset, SEEK_SET)) std_err();
                zlen = cauldron_dump(fd, fdo, buff, buffsz, zlen);
                if((zoffset + zlen) >= comp_len) break;
            }

        } else {
            printf("\nError: unknown compression method (%u)\n", compress);
            exit(1);
        }

        fclose(fdo);
        tot++;
        if(fseek(fd, current_offset, SEEK_SET)) std_err();
    }

    if(!fsvercheck) {
        fsvercheck = 1;
        goto redo;
    }

    if(zbuff) free(zbuff);
    if(buff)  free(buff);
    if(fname) free(fname);
    return(tot);
}



void xor(u8 *data, int len, u8 chr) {
    int     i;

    for(i = 0; i < len; i++) {
        data[i] ^= chr;
    }
}



u8 **get_names(FILE *fd, int len, int num) {
    int     i;
    u8      *namebuff,
            **names,
            *p;

    namebuff = malloc(len); // do NOT free!
    if(!namebuff) std_err();

    names = malloc(sizeof(char *) * num);
    if(!names) std_err();

    myfr(fd, namebuff, len);

    p = namebuff;
    for(i = 0; i < num; i++) {
        names[i] = p;
        p += strlen(p) + 1;
    }
    return(names);
}



u32 fgetxx(FILE *fd, int bytes) {
    u32     num;
    int     i;
    u8      tmp[8];

    myfr(fd, tmp, bytes);
    for(num = i = 0; i < bytes; i++) {
        num |= (tmp[i] << (i << 3));
    }
    return(num);
}



u8 *create_dir(u8 *fname) {
    u8      *p,
            *l;

    p = strchr(fname, ':'); // unused
    if(p) {
        *p = '_';
        fname = p + 1;
    }
    for(p = fname; *p && strchr("\\/. \t:", *p); p++) *p = '_';
    fname = p;

    for(p = fname; ; p = l + 1) {
        for(l = p; *l && (*l != '\\') && (*l != '/'); l++);
        if(!*l) break;
        *l = 0;

        if(!strcmp(p, "..")) {
            p[0] = '_';
            p[1] = '_';
        }

        make_dir(fname);
        *l = PATHSLASH;
    }
    return(fname);
}



int check_wildcard(u8 *fname, u8 *wildcard) {
    u8      *f,
            *w,
            *a;

    if(!wildcard) return(0);
    f = fname;
    w = wildcard;
    a = NULL;
    while(*f || *w) {
        if(!*w && !a) return(-1);
        if(*w == '?') {
            if(!*f) break;
            w++;
            f++;
        } else if(*w == '*') {
            w++;
            a = w;
        } else {
            if(!*f) break;
            if(tolower(*f) != tolower(*w)) {
                if(!a) return(-1);
                f++;
                w = a;
            } else {
                f++;
                w++;
            }
        }
    }
    if(*f || *w) return(-1);
    return(0);
}



int check_overwrite(u8 *fname) {
    FILE    *fd;
    u8      ans[16];

    fd = fopen(fname, "rb");
    if(!fd) return(0);
    fclose(fd);
    printf("  the file already exists, do you want to overwrite it (y/N)? ");
    fgets(ans, sizeof(ans), stdin);
    if(tolower(ans[0]) != 'y') return(-1);
    return(0);
}



void myalloc(u8 **data, unsigned wantsize, unsigned *currsize) {
    if(wantsize <= *currsize) return;
    *data = realloc(*data, wantsize);
    if(!*data) std_err();
    *currsize = wantsize;
}



void myfw(FILE *fd, void *data, unsigned size) {
    if(fwrite(data, 1, size, fd) != size) {
        printf("\nError: impossible to write %d bytes to the output file, check your space\n", size);
        fclose(fd);
        exit(1);
    }
}



void myfr(FILE *fd, void *data, unsigned size) {
    if(fread(data, 1, size, fd) != size) {
        printf("\nError: incomplete input file, can't read %u bytes at offset %08x\n", size, (u32)ftell(fd));
        fclose(fd);
        exit(1);
    }
}



void std_err(void) {
    perror("\nError");
    exit(1);
}


