/*
    Copyright 2005,2006,2007,2008,2009,2010 Luigi Auriemma

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    http://www.gnu.org/licenses/gpl.txt
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <ctype.h>

#ifdef WIN32
    #include <direct.h>
    #define PATHSLASH   '\\'
#else
    #include <unistd.h>
    #include <dirent.h>
    #include <sys/stat.h>

    #define strnicmp    strncasecmp
    #define stristr     strcasestr
    #define PATHSLASH   '/'
#endif



#define VER     "0.2.2"
#define SIGN    "BIGF"



int unlzw(void *out, int outsize, void *in, int insize);
int myfgets(uint8_t *buff, int buffsz, FILE *fd);
int check_wildcard(uint8_t *fname, uint8_t *wildcard);
uint64_t fdxx(FILE *fd, uint8_t *data, int bits);
void cbf_file_read(FILE *fd, uint8_t *fname, uint64_t off, uint64_t len);
void cbf_head_dec(uint8_t *data, uint16_t len);
void cbf_file_dec(uint8_t *data, int len);
void crp_convertfile(uint8_t *data, int len);
uint8_t *create_dir(uint8_t *name);
void std_err(void);



int     cbf_ver = 0;



int main(int argc, char *argv[]) {
    FILE        *fd;
    uint64_t    file_offset,
                data_offset,
                data_length;
    uint32_t    i,
                file_num,
                extracted = 0;
    int         listonly  = 0;
    uint16_t    len;
    uint8_t     sign[4],
                *buff,
                *fname,
                *folder   = NULL,
                *pattern  = NULL;

    setbuf(stdout, NULL);

    fputs("\n"
        "CBF files extractor "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", stdout);

    if(argc < 2) {
        printf("\n"
            "Usage: %s [options] <file.CBF>\n"
            "\n"
            "Options:\n"
            "-l       list files without extracting them\n"
            "-d DIR   the folder where the files will be extracted\n"
            "-p PAT   extract only the files which match the wildcard PAT like \"*.dds\"\n"
            "\n",
            argv[0]);
        exit(1);
    }

    argc--;
    for(i = 1; i < argc; i++) {
        if(((argv[i][0] != '-') && (argv[i][0] != '/')) || (strlen(argv[i]) != 2)) {
            printf("\nError: recheck your options (%s is not valid)\n", argv[i]);
            exit(1);
        }
        switch(argv[i][1]) {
            case 'l': listonly = 1;         break;
            case 'd': folder   = argv[++i]; break;
            case 'p': pattern  = argv[++i]; break;
            default: {
                printf("\nError: wrong command-line argument (%s)\n\n", argv[i]);
                exit(1);
                } break;
        }
    }

    printf("- open file %s\n", argv[argc]);
    fd = fopen(argv[argc], "rb");
    if(!fd) std_err();

    if(folder) {
        printf("- enter in folder %s\n", folder);
        if(chdir(folder) < 0) std_err();
    }

    fread(sign, 1, 4, fd);
    if(memcmp(sign, SIGN, sizeof(SIGN) - 1)) {
        printf("- sign is not "SIGN", the file seems NOT valid!\n");
    }

    cbf_ver     = fdxx(fd, NULL, 8);
    fseek(fd, 3 + 8, SEEK_CUR); // "ZBL + full archive size
    file_num    = fdxx(fd, NULL, 32);
    file_offset = fdxx(fd, NULL, 64);

    printf("- file info offset: 0x%08x\n", (uint32_t)file_offset);

    buff = malloc(0xffff + 3);
    if(!buff) std_err();
    fname = buff;
    if(cbf_ver) fname += 40;

    printf("\n"
        "  size       filename\n"
        "  -------------------\n");

    for(i = 0; i < file_num; i++) {
        if(fseek(fd, file_offset, SEEK_SET) < 0) std_err();

        if(cbf_ver) {
            len = fdxx(fd, NULL, 16);
            file_offset += 2;
        } else {
            len = 40;
        }
        fread(buff, 1, len, fd);
        file_offset += len;

        cbf_head_dec(buff, len);
        data_offset  = fdxx(NULL, buff, 64);
        data_length  = fdxx(NULL, buff + 8 + 4 + 8, 64);
        if(!cbf_ver) {
            len = myfgets(buff, 0xffff, fd);
            file_offset += len;
        }

        if(pattern && (check_wildcard(fname, pattern) < 0)) continue;

        printf("  %-10u %s\n", (uint32_t)data_length, fname);

        extracted++;
        if(listonly) continue;

        cbf_file_read(fd, fname, data_offset, data_length);
    }

    printf("\n"
        "- %u files %s\n\n",
        extracted,
        listonly ? "listed" : "extracted");
    fclose(fd);
    free(buff);
    return(0);
}



int myfgets(uint8_t *buff, int buffsz, FILE *fd) {
    int     i,
            c;

    buffsz--;
    for(i = 0;; i++) {
        c = fgetc(fd);
        if(c <= 0) c = 0;
        if(i < buffsz) buff[i] = c;
        if(!c) break;
    }
    if(i >= buffsz) buff[buffsz] = 0;
    return(i + 1);
}



int check_wildcard(uint8_t *fname, uint8_t *wildcard) {
    uint8_t *f,
            *w,
            *a;

    f = fname;
    w = wildcard;
    a = NULL;
    while(*f || *w) {
        if(!*w && !a) return(-1);
        if(*w == '?') {
            if(!*f) break;
            w++;
            f++;
        } else if(*w == '*') {
            w++;
            a = w;
        } else {
            if(!*f) break;
            if(tolower(*f) != tolower(*w)) {
                if(!a) return(-1);
                f++;
                w = a;
            } else {
                f++;
                w++;
            }
        }
    }
    if(*f || *w) return(-1);
    return(0);
}



uint64_t fdxx(FILE *fd, uint8_t *data, int bits) {
    uint64_t    num,
                t;
    int         i,
                bytes;
    uint8_t     tmp[8];

    bytes = bits >> 3;
    if(fd) {
        fread(tmp, 1, bytes, fd);
    } else {
        memcpy(tmp, data, sizeof(tmp));
    }

    num = 0;
    for(i = 0; i < bytes; i++) {
        if(i >= 4) continue;
        t = tmp[i];
        num |= (t << (i << 3));
    }

    return(num);
}



void cbf_file_read(FILE *fd, uint8_t *fname, uint64_t off, uint64_t len) {
    FILE        *fdo;
    uint32_t    zlen,
                ulen;
    int         comp;
    uint8_t     sign[4],
                *buff,
                *zbuff;

    if(fseek(fd, off, SEEK_SET) < 0) std_err();

    fread(sign, 1, 4, fd);

    comp = 0;
    if(!memcmp(sign, "[..]", 4)) {  // compressed files
        comp = 1;
    }

    fdo = fopen(create_dir(fname), "wb");
    if(!fdo) std_err();

    if(comp) {
        fread(&zlen, 1, 4, fd);
        fread(&ulen, 1, 4, fd);
        len = zlen;
    } else {
        fseek(fd, -4, SEEK_CUR);
    }

    buff = malloc(len);
    if(!buff) std_err();
    fread(buff, 1, len, fd);

    if(comp) {
        zbuff = malloc(ulen);
        if(!zbuff) std_err();
        len = unlzw(zbuff, ulen, buff, zlen);
        free(buff);
        buff = zbuff;
        if(len != ulen) {
            printf("\nError: wrong lzw output size\n");
            exit(1);
        }
    } else {
        cbf_file_dec(buff, len);
        if(!strnicmp(fname, "INI\\DAT\\", 8) || !strnicmp(fname, "INI/DAT/", 8)) {
            crp_convertfile(buff, len);
        }
    }

    fwrite(buff, len, 1, fdo);
    fclose(fdo);
    free(buff);
}



void cbf_head_dec(uint8_t *data, uint16_t len) {
    static const uint8_t key[16] = {
                0x32, 0xf3, 0x1e, 0x06,
                0x45, 0x70, 0x32, 0xaa,
                0x55, 0x3f, 0xf1, 0xde,
                0xa3, 0x44, 0x21, 0xb4 };
    uint8_t     t,
                e;

    if(!cbf_ver) return;
    for(e = len; len--; data++) {
        t = *data;
        *data = t ^ key[e & 15];
        e = t;
    }
}



void cbf_file_dec(uint8_t *data, int len) {
    uint8_t     t1,
                t2,
                *limit;

    if(!cbf_ver) return;
    t1 = len;
    t2 = 90 - t1;
    for(limit = data + len; data < limit; data++) {
        *data = (*data - t2) ^ t1;
    }
}



void crp_convertfile(uint8_t *data, int len) {
    int         i;
    uint8_t     e;

    if(!cbf_ver) return;
    e = 234;
    for(i = 0; i < len; i++) {
        data[i] -= e;
        switch(i % 3) {
            case 0: e += 13;  break;
            case 1: e += len; break;
            case 2: e -= i;   break;
        }
    }
}



uint8_t *create_dir(uint8_t *name) {
    uint8_t     *p,
                *l;

    for(p = name; (*p == '\\') || (*p == '/') || (*p == '.'); p++);
    name = p;

    for(;;) {
        if((p[0] && (p[1] == ':')) || ((p[0] == '.') && (p[1] == '.'))) p[0] = '_';

        l = strchr(p, '\\');
        if(!l) {
            l = strchr(p, '/');
            if(!l) break;
        }
        *l = 0;
        p = l + 1;

#ifdef WIN32
        mkdir(name);
#else
        mkdir(name, 0755);
#endif
        *l = PATHSLASH;
    }
    return(name);
}



void std_err(void) {
    perror("\nError");
    exit(1);
}


