/*
    Copyright 2004,2005,2006 Luigi Auriemma

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    http://www.gnu.org/licenses/gpl.txt
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <openssl/blowfish.h>
#include "cogs_chall.h"
#include "cogs_irc_chall.h"

#ifdef WIN32
    #include <winsock.h>
    #include "winerr.h"

    #define close       closesocket
    #define sleep   Sleep
    #define ONESEC      1000
#else
    #include <unistd.h>
    #include <sys/socket.h>
    #include <sys/types.h>
    #include <arpa/inet.h>
    #include <netinet/in.h>
    #include <netdb.h>
    #include <pthread.h>

    #define strnicmp    strncasecmp
    #define stricmp     strcasecmp
    #define ONESEC      1
#endif

#include "sockline.h"

#ifdef WIN32
    #define quick_thread(NAME, ARG) DWORD WINAPI NAME(ARG)
    #define thread_id   DWORD
#else
    #define quick_thread(NAME, ARG) void *NAME(ARG)
    #define thread_id   pthread_t
#endif

thread_id quick_threadx(void *func, void *data) {
    thread_id       tid;
#ifdef WIN32
    if(!CreateThread(NULL, 0, func, data, 0, &tid)) return(0);
#else
    pthread_attr_t  attr;
    pthread_attr_init(&attr);
    pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_DETACHED);
    if(pthread_create(&tid, &attr, func, data)) return(0);
#endif
    return(tid);
}



#define VER         "0.2.2a"
#define BUFFSZ      8192
#define COGSINI     "cogs_irc.ini"

#define HOST_LOGIN  "144.140.155.6" // could require update!
#define PORT_LOGIN  11120

#define HOST_IRC    "thearena-chat.gamearena.com.au"
#define PORT_IRC    4445

#define PINGTIME    180             // 3 minutes
#define NICKSZ      256
#define COGSVER     "225"           // could require update!
#define LOGIN       "HELO WIN32 "COGSVER"\n"    \
                    "%s\n"      /* username */  \
                    "%s\n"      /* password */  \
                    "%012X\n"   /* uniqueid */  \
                    "2\n"                       \
                    "%012X\n"   /* uniqueid */  \
                    "LOCFAILURE\n"              \
                    "%08X\n"    /* random   */
#define IRCLOGIN    "PASS fred\r\n" \
                    "USER cogs%d cogs cogs :%d\r\n" \
                    "NICK %s\r\n"
#define PRESSRET    fputs("\nPress RETURN to exit\n\n", stdout); \
                    fgetc(stdin);



void load_ini(u_char *username, u_char *password, u_char **logserver, u_short *logport, u_char **ircserver, u_short *ircport, u_short *proxyport, int *keeplogin);
u_char *get_host(u_char *data, u_short *port);
quick_thread(client, int sda);
quick_thread(login_ping, void);
void cogs_login(int *userid, u_char *nickname);
u_char *cogs_show(u_char *title, u_char *data);
int myrecv(int sd, u_char *data, int max, u_char chr);
void bf(u_char *data, int len, int enc);
void cogs_send(int sd, u_char *data, int len);
int cogs_recv(int sd, u_char *data, int max);
void delimit(u_char *data);
u_char *base64_encode(u_char *data, int *length);
int Base64Decode(u_char *buf, u_char *ret);
u_char *mydelimiter(u_char *data);
void myreadini(u_char *line, u_char **par, u_char **value);
u_char *parse_config_line(u_char *data, u_char *value);
u_int resolv(char *host);
void std_err(void);



struct sockaddr_in  peer,
                    loginpeer;
BF_KEY  bfx;
int     uniqueid,
        userid,
        verbose = 0;
u_char  nickname[NICKSZ + 1],
        username[256],
        password[256];



int main(int argc, char *argv[]) {
    struct  sockaddr_in peerl;
    int     sdl,
            sda,
            i,
            on         = 1,
            keeplogin  = 0,
            psz;
    u_short proxyport  = 6667,
            logport    = PORT_LOGIN,
            ircport    = PORT_IRC;
    u_char  *logserver = HOST_LOGIN,
            *ircserver = HOST_IRC,
            *iface     = "127.0.0.1";
    const static unsigned char cogs_key[32] =
        "\xa3\xd5\x6e\xf0\xbf\x3a\xed\xd8\xc5\x49\x2a\x34\x3c\x21\xcd\xf1"
        "\xb2\x38\xfc\xdb\x73\x16\xba\x85\x49\x39\x8c\x99\x54\x78\xdf\x72";

#ifdef WIN32
    WSADATA    wsadata;
    WSAStartup(MAKEWORD(1,0), &wsadata);
#endif


    srand(time(NULL));  /* random */
    setbuf(stdout, NULL);

    fputs("\n"
        "COGS Gamearena IRC proxy "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", stdout);

    printf("\n"
        "Usage: %s [options]\n"
        "\n"
        "Options:\n"
        "-u U P    specify username (U) and password (P)\n"
        "-k        enable keep-alive login (no longer required)\n"
        "-p PORT   local port bound by this proxy server (default %hu)\n"
        "-b IP     local interface to bind, use 0 for any (default %s)\n"
        "-v        verbose, shows every (decrypted) data sent and received\n"
        "-s H[:P]  change the default login server (%s:%hu)\n"
//        "          For example Xtra (http://games.xtra.co.nz) uses 210.55.92.2:11130\n"
        "-S H[:P]  change the default IRC server (%s:%hu)\n"
//        "          For example Xtra (http://games.xtra.co.nz) uses 210.55.92.2:11140\n"
        "\n",
        argv[0], proxyport, iface, logserver, logport, ircserver, ircport);

    *username = 0;
    *password = 0;

    load_ini(username, password, &logserver, &logport, &ircserver, &ircport, &proxyport, &keeplogin);
    fputc('\n', stdout);

    for(i = 1; i < argc; i++) {
        switch(argv[i][1]) {
            case 'u': {
                sprintf(username, "%.*s", sizeof(username) - 1, argv[++i]);
                sprintf(password, "%.*s", sizeof(password) - 1, argv[++i]);
                } break;
            case 'k': keeplogin = 1;                                break;
            case 'p': proxyport = atoi(argv[++i]);                  break;
            case 'b': iface     = argv[++i];                        break;
            case 'v': verbose   = 1;                                break;
            case 's': logserver = get_host(argv[++i], &logport);    break;
            case 'S': ircserver = get_host(argv[++i], &ircport);    break;
            default: {
                printf("\nError: Wrong command-line argument (%s)\n\n", argv[i]);
                exit(1);
                } break;
        }
    }

    if(!*username) {
        fputs("- Enter your username: ", stdout);
        fflush(stdin);
        fgets(username, sizeof(username), stdin);
        delimit(username);
    } else {
        printf("- Loaded username:     %s\n",
            username);
    }

    if(!*password) {
        fputs("- Enter your password: ", stdout);
        fflush(stdin);
        fgets(password, sizeof(password), stdin);
        delimit(password);
    } else {
        printf("- Loaded password:     %s\n",
             password);
    }

    uniqueid = rand() * time(NULL);
    printf("- random uniqueID generated: %012X\n", uniqueid);

    printf("- resolv hostname %s = ", logserver);
    loginpeer.sin_addr.s_addr = resolv(logserver);
    loginpeer.sin_family      = AF_INET;
    loginpeer.sin_port        = htons(logport);
    printf("%s\n", inet_ntoa(loginpeer.sin_addr));

    printf("- resolv hostname %s = ", ircserver);
    peer.sin_addr.s_addr      = resolv(ircserver);
    peer.sin_family           = AF_INET;
    peer.sin_port             = htons(ircport);
    printf("%s\n", inet_ntoa(peer.sin_addr));

    BF_set_key(&bfx, sizeof(cogs_key), cogs_key);

    printf("- send login request to %s : %hu\n",
        inet_ntoa(loginpeer.sin_addr), logport);
    cogs_login(&userid, nickname);

    if(keeplogin) {
        quick_threadx(login_ping, NULL);
    }

    peerl.sin_addr.s_addr = inet_addr(iface);
    peerl.sin_port        = htons(proxyport);
    peerl.sin_family      = AF_INET;
    psz                   = sizeof(peerl);

    if(peerl.sin_addr.s_addr == INADDR_ANY) {
        printf("- start proxy on port %hu of all the interfaces\n",
            proxyport);
    } else {
        printf("- start proxy on interface %s and port %hu\n",
            inet_ntoa(peerl.sin_addr), proxyport);
    }

    sdl = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
    if(sdl < 0) std_err();
    if(setsockopt(sdl, SOL_SOCKET, SO_REUSEADDR, (char *)&on, sizeof(on))
      < 0) std_err();
    if(bind(sdl, (struct sockaddr *)&peerl, sizeof(peerl))
      < 0) std_err();
    if(listen(sdl, SOMAXCONN)
      < 0) std_err();

    printf(
        "- IRC proxy enabled\n"
        "- connect your IRC client to irc://%s:%hu\n"
        "  if some chans require a keyword use cogs (/join #quake4 cogs)\n",
        inet_ntoa(peerl.sin_addr), proxyport);

    for(;;) {
        sda = accept(sdl, (struct sockaddr *)&peerl, &psz);
        if(sda < 0) std_err();

        printf("  %s:%hu\n",
            inet_ntoa(peerl.sin_addr), ntohs(peerl.sin_port));

        if(!quick_threadx(client, (void *)sda)) close(sda);
    }

    close(sdl);
    return(0);
}



void load_ini(u_char *username, u_char *password, u_char **logserver, u_short *logport, u_char **ircserver, u_short *ircport, u_short *proxyport, int *keeplogin) {
    FILE    *fd;
    int     found = 0;
    u_char  buff[256 + 10],
            *p;

    fd = fopen(COGSINI, "rb");
    if(!fd) return;

    fputs("- load configuration from "COGSINI"\n", stdout);

    while(fgets(buff, sizeof(buff), fd)) {
        p = parse_config_line(buff, "username");
        if(p && *p) {
            strcpy(username, p);
            fputs("  cfg: username found\n", stdout);
            found++;
            continue;
        }

        p = parse_config_line(buff, "password");
        if(p && *p) {
            strcpy(password, p);
            fputs("  cfg: password found\n", stdout);
            found++;
            continue;
        }

        p = parse_config_line(buff, "login_server");
        if(p && *p) {
            *logserver = strdup(get_host(p, logport));
            fputs("  cfg: login server found\n", stdout);
            found++;
            continue;
        }

        p = parse_config_line(buff, "irc_server");
        if(p && *p) {
            *ircserver = strdup(get_host(p, ircport));
            fputs("  cfg: IRC server found\n", stdout);
            found++;
            continue;
        }

        p = parse_config_line(buff, "proxy_port");
        if(p && *p) {
            *proxyport = atoi(p);
            fputs("  cfg: proxy port found\n", stdout);
            found++;
            continue;
        }

        p = parse_config_line(buff, "keep-alive");
        if(p && *p) {
            *keeplogin = atoi(p);
            fputs("  cfg: keep-alive found\n", stdout);
            found++;
            continue;
        }
    }

    if(!found) fputs("  cfg: no parameters found\n", stdout);

    fclose(fd);
}



u_char *get_host(u_char *data, u_short *port) {
    u_char  *p;

    p = strchr(data, ':');
    if(p) {
        *p = 0;
        *port = atoi(p + 1);
    }

    return(data);
}



quick_thread(client, int sda) {
    int     sd,
            t,
            len;
    u_char  buff[BUFFSZ + 1],
            *p;

    sd = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
    if(sd < 0) std_err();
    if(connect(sd, (struct sockaddr *)&peer, sizeof(peer))
      < 0) std_err();

    len = snprintf(
        buff,
        BUFFSZ,
        IRCLOGIN,
        userid,
        userid,
        nickname);
    if((len < 0) || (len > sizeof(buff))) {
        fputs("\nError: buffer too small\n\n", stdout);
        exit(1);
    }

    if(send(sd, buff, len, 0)
      < 0) goto quit;
    if(verbose) fputs(buff, stdout);

    len = BUFFSZ;
    sockline(NULL, &len, '\n', sd, sda, 0);

    for(;;) {
        t = sockline(buff, &len);
        if(len <= 0) break;
        if(verbose) fputs(buff, stdout);

        if(t == sd) {                   /* SERVER */
            p = strchr(buff, ' ');
            if(p && !strnicmp(++p, "CRYP ", 5)) {
                p += 5;                         // create "CRYP :RESP\r\n"
                len -= (p - buff);
                memmove(buff + 6, p, ++len);
                memcpy(buff, "CRYP :", 6);
                p = buff + 6;
                len += 6;
                printf("- CRYP challenge: %s", p);
                cogs_irc_chall(p, len - 8);
                printf("- CRYP response:  %s", p);
                if(send(sd, buff, len, 0)
                  < 0) break;
                continue;
            }

            if(send(sda, buff, len, 0)
               <= 0) break;
        } else {                        /* CLIENT */
            if( /* if you use USER or NICK you will be kicked! */
              !strnicmp(buff, "USER ", 5) ||
              !strnicmp(buff, "NICK ", 5)
            ) {
                fputs("- your IRC command has been skipped to avoid kicking\n", stdout);
                continue;
            }

            if(send(sd, buff, len, 0)
              <= 0) break;
        }
    }

quit:
    sockline(buff, 0);
    fputs("- disconnected\n", stdout);
    return(0);
}



quick_thread(login_ping, void) {
    for(;;) {
        sleep(PINGTIME * ONESEC);
        fputs("- send heartbeat login\n", stdout);
        cogs_login(0, NULL);
    }
    return(0);
}



void cogs_login(int *userid, u_char *nickname) {
    int     sd,
            len;
    u_char  buff[BUFFSZ + 1],
            *p,
            *t;

    sd = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
    if(sd < 0) std_err();
    if(connect(sd, (struct sockaddr *)&loginpeer, sizeof(loginpeer))
      < 0) std_err();

    len = snprintf(
        buff,
        BUFFSZ,
        LOGIN,
        username,
        password,
        uniqueid,
        uniqueid,
        (u_int)(~time(NULL) & 0xfffffff));  // something random
    if((len < 0) || (len > sizeof(buff))) {
        fputs("\nError: buffer too small\n\n", stdout);
        exit(1);
    }

    cogs_send(sd, buff, len);

    len = cogs_recv(sd, buff, BUFFSZ);
    if(len < 0) std_err();

    if(*buff == '0') goto login_failed;

    printf("- Login challenge: %s\n", buff);
    len = cogs_chall(buff, strlen(buff));
    printf("- Login response:  %s\n", buff);

    cogs_send(sd, buff, len);

    len = cogs_recv(sd, buff, BUFFSZ);
    if(len < 0) std_err();

    close(sd);

    if(*buff == '0') goto login_failed;

    if(nickname) {
        p = buff;
        p = cogs_show("Login Accepted", p);
        p = cogs_show("Username",       p);
        t = p;
        p = cogs_show("Nickname",       p);
        strncpy(nickname, t, NICKSZ);       // save nickname
        p = cogs_show("E-mail",         p);
        t = p;
        p = cogs_show("UserID",         p);
        *userid = atoi(t);                  // save userid
    }

    return;

login_failed:
    printf("\n"
        "Error: Login failed!\n"
        "\n"
        "%s\n"
        "\n", buff);
    PRESSRET;
    exit(1);
}



u_char *cogs_show(u_char *title, u_char *data) {
    u_char  *p;

    p = strchr(data, '\n');
    if(!p) {
        fputs("\nError: no login value found, don't know why but is better if you report this problem to me!\n\n", stdout);
        exit(1);
    }
    *p = 0;
    printf("- %20s: %s\n", title, data);
    return(p + 1);
}



int myrecv(int sd, u_char *data, int max, u_char chr) {
    int     len;
    u_char  *p = data;

    len = 0;
    do {
        if(recv(sd, p, 1, 0) <= 0) return(-1);
        if(++len == max) break;
    } while(*p++ != chr);

    *p = 0;
    return(p - data);
}



void bf(u_char *data, int len, int enc) {
    while(len > 0) {
        BF_ecb_encrypt(data, data, &bfx, enc);
        data += 8;
        len  -= 8;
    }
}



void cogs_send(int sd, u_char *data, int len) {
    int     blen,
            diff;
    u_char  *b64;

    blen = len;
    diff = len & 7;
    if(diff) {
        diff = 8 - diff;
        memset(data + blen, 0, diff);
        blen += diff;
    }

    if(verbose) printf("%s\n", data);

    bf(data, blen, BF_ENCRYPT);         // encrypt
    b64 = base64_encode(data, &blen);   // base64
    b64[blen++] = '=';                  // WORK-AROUND FOR COGS!!!
    send(sd, b64, blen, 0);
    free(b64);
}



int cogs_recv(int sd, u_char *data, int max) {
    int     len;

    len = myrecv(sd, data, max, '=');
    if(len < 0) return(len);
    if(!(len & 1)) {                    // wrong but this is what COGS wants
        recv(sd, data + len, 1, 0);
        len++;
    }
    data[len] = 0;

    len = Base64Decode(data, data);     // base64
    bf(data, len, BF_DECRYPT);          // decrypt

    if(verbose) printf("%s\n", data);
    return(len);
}



void delimit(u_char *data) {
    while(*data > '\r') data++;
    *data = 0;
}



u_char *base64_encode(u_char *data, int *length) {
    int             r64len,
                    len = *length;
    u_char          *p64;
    static u_char   *r64;
    const static char   enctab[64] = {
        'A','B','C','D','E','F','G','H','I','J','K','L','M','N','O','P',
        'Q','R','S','T','U','V','W','X','Y','Z','a','b','c','d','e','f',
        'g','h','i','j','k','l','m','n','o','p','q','r','s','t','u','v',
        'w','x','y','z','0','1','2','3','4','5','6','7','8','9','+','/'
    };

    r64len = ((len / 3) << 2) + 5;
    r64 = malloc(r64len);
    if(!r64) return(NULL);
    p64 = r64;

    do {
        *p64++ = enctab[(*data >> 2) & 63];
        *p64++ = enctab[(((*data & 3) << 4) | ((*(data + 1) >> 4) & 15)) & 63];
        data++;
        *p64++ = enctab[(((*data & 15) << 2) | ((*(data + 1) >> 6) & 3)) & 63];
        data++;
        *p64++ = enctab[*data & 63];
        data++;

        len -= 3;
    } while(len > 0);

    for(; len < 0; len++) *(p64 + len) = '=';
    *p64 = 0;

    *length = p64 - r64;
    return(r64);
}



int Base64Decode(u_char *buf, u_char *ret) {
#define B64DECX     bb = *bf; \
                    for(; \
                    bb && \
                    ((bb == ' ') || \
                    (bb == '\t') || \
                    (bb == '\r') || \
                    (bb == '\n')); \
                    bf++);
    int     c,
            d,
            i,
            Base[256];
    u_char  bb,
            *bf,
            *rf;

    if(!*buf) {
        *ret = 0;
        return(0);
    }

    for(i = 0; i < 26; i++) {
        Base['A' + i] = i;
        Base['a' + i] = i + 26;
    }
    for(i = 0; i < 10; i++) {
        Base['0' + i] = i + 52;
    }
    Base['+'] = 62;
    Base['/'] = 63;

    bf = buf;
    rf = ret;
    B64DECX;
    for(;;) {
        bb = *bf++;
        if((bb == '=') || !bb) break;
        c = Base[bb];

        B64DECX;
        bb = *bf++;
        if((bb == '=') || !bb) break;
        d = Base[bb];
        *rf++ = (c << 2) | (d >> 4);

        B64DECX;
        bb = *bf++;
        if((bb == '=') || !bb) break;
        c = Base[bb];
        *rf++ = ((d & 15) << 4) | (c >> 2);

        B64DECX;
        bb = *bf++;
        if((bb == '=') || !bb) break;
        *rf++ = ((c & 3) << 6) | Base[bb];
    }
    *rf = 0;

    return(rf - ret);

#undef B64DECX
}



u_char *mydelimiter(u_char *data) {
    while(*data && (*data != '\r') && (*data != '\n')) data++;
    *data = 0;
    return(data);
}



void myreadini(u_char *line, u_char **par, u_char **value) {
    u_char  *divx,      // =
            *end;

    *par   = NULL;
    *value = NULL;
    end = mydelimiter(line);

    while(*line && (*line <= ' ')) line++;          // -> PAR
    *par = line;

    while(*line && (*line != '=')) line++;          // =
    divx = line;

    for(line--; *line && (*line <= ' '); line--);   // PAR <-
    *(line + 1) = 0;

    line = divx + 1;

    while(*line && (*line <= ' ')) line++;          // -> VAL
    *value = line;

    line = end - 1;                                 // end

    while(*line && (*line <= ' ')) line--;          // VAL <-
    *(line + 1) = 0;
}



u_char *parse_config_line(u_char *data, u_char *value) {
    u_char  *p,
            *v;

    myreadini(data, &p, &v);
    if(stricmp(p, value)) return(NULL);
    return(v);
}



u_int resolv(char *host) {
    struct  hostent *hp;
    u_int   host_ip;

    host_ip = inet_addr(host);
    if(host_ip == INADDR_NONE) {
        hp = gethostbyname(host);
        if(!hp) {
            printf("\nError: Unable to resolv hostname (%s)\n", host);
            exit(1);
        } else host_ip = *(u_int *)(hp->h_addr);
    }
    return(host_ip);
}



#ifndef WIN32
    void std_err(void) {
        perror("\nError");
        exit(1);
    }
#endif


