/*
    Copyright 2009 Luigi Auriemma

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    http://www.gnu.org/licenses/gpl-2.0.txt
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include "dgrid_sha1.h"

#ifdef WIN32
#else
    #define stricmp strcasecmp
#endif

typedef unsigned char   u8;



#define VER     "0.2"



void std_err(void) {
    perror("\nError");
    exit(1);
}



u8 *fdload(u8 *fname, int *fsize) {
    struct stat xstat;
    FILE    *fd;
    int     size;
    u8      *buff;

    printf("- open file %s\n", fname);
    fd = fopen(fname, "rb");
    if(!fd) std_err();
    fstat(fileno(fd), &xstat);
    size = xstat.st_size;
    buff = calloc(1, size + 20);    // for DefenseGrid!
    if(!buff) std_err();
    fread(buff, 1, size, fd);
    fclose(fd);
    *fsize = size;
    return(buff);
}



void fddump(u8 *fname, u8 *buff, int size) {
    FILE    *fd;

    printf("- dump file %s\n", fname);
    fd = fopen(fname, "wb");
    if(!fd) std_err();
    fwrite(buff, 1, size, fd);
    fclose(fd);
}



int main(int argc, char *argv[]) {
    int     i,
            buffsz,
            has_hash    = 0,
            check_it    = 0;
    u8      hash[20],
            *buff,
            *fin        = NULL,
            *fout       = NULL,
            *p;

    fputs("\n"
        "DefenseGrid dgp files hash calculator "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", stdout);

    if(argc < 2) {
        printf("\n"
            "Usage: %s [options] <file>\n"
            "\n"
            "-o FILE  create a new file with the new calculated hash added/updated, if the\n"
            "         file already exists it will be overwritten automatically\n"
            "-a       use it if the input file isn't a dgp file and so doesn't have the\n"
            "         hash already at its end (default for files without the dgp extension)\n"
            "-r       use it if the input file is a dgp file and so has the hash already at\n"
            "         its end (default for files with the dgp extension)\n"
            "-d       activate the modified sha1 algorithm used in the demo\n"
            "-c       if the input file already has the hash check if it's correct\n"
            "\n", argv[0]);
        exit(1);
    }

    argc--;
    fin = argv[argc];
    p = strrchr(fin, '.');
    if(p && !stricmp(p, ".dgp")) has_hash = 1;

    for(i = 1; i < argc; i++) {
        if(((argv[i][0] != '-') && (argv[i][0] != '/')) || (strlen(argv[i]) != 2)) {
            printf("\nError: wrong argument (%s)\n", argv[i]);
            exit(1);
        }
        switch(argv[i][1]) {
            case 'o': fout      = argv[++i];    break;
            case 'a': has_hash  = 0;            break;
            case 'r': has_hash  = 1;            break;
            case 'd': sha1_is_demo(1);          break;
            case 'c': check_it  = 1;            break;
            default: {
                printf("\nError: wrong argument (%s)\n", argv[i]);
                exit(1);
            }
        }
    }

    buff = fdload(fin, &buffsz);
    if(has_hash) {
        printf("- input file handled with hash at the end\n");
        buffsz -= 20;
    } else {
        printf("- input file handled without hash at the end\n");
    }

    sha1(buff, buffsz, hash);
    printf("- hash: ");
    for(i = 0; i < 20; i++) {
        printf("%02x", hash[i]);
    }
    printf("\n");

    if(has_hash && check_it) {
        if(!memcmp(buff + buffsz, hash, 20)) {
            printf("- the hash is CORRECT\n");
        } else {
            printf("- the hash is WRONG\n");
        }
    } else if(fout && fout[0]) {
        memcpy(buff + buffsz, hash, 20);
        buffsz += 20;
        fddump(fout, buff, buffsz);
    }

    printf("- done\n");
    return(0);
}


