/*
    Copyright 2007 Luigi Auriemma

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    http://www.gnu.org/licenses/gpl.txt
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include "show_dump.h"

#ifdef WIN32
    #include <winsock.h>
    #include "winerr.h"

    #define close   closesocket
    #define sleep   Sleep
    typedef uint32_t    in_addr_t;
#else
    #include <unistd.h>
    #include <sys/socket.h>
    #include <sys/types.h>
    #include <arpa/inet.h>
    #include <netinet/in.h>
    #include <netdb.h>
    #include <sys/times.h>
#endif



#define VER         "0.1a"
#define BUFFSZ      4096
#define PORT        47624
#define COOLGUID    0x00000000, 0x0000, 0x0000, 0, 0, 0, 0, 0, 0, 0, 0
#define SOCKADDR    struct sockaddr_in
#define SOCKADDRSZ  sizeof(struct sockaddr_in)



/* info from the Wine project */

#define DPLAY_ID        0xfab0
#define ACTION_STRING   "play"

#define DPMSGCMD_ENUMSESSIONSREPLY    1
#define DPMSGCMD_ENUMSESSIONSREQUEST  2
#define DPMSGCMD_GETNAMETABLEREPLY    3
#define DPMSGCMD_PLAYERSLISTREQUEST   4
#define DPMSGCMD_REQUESTNEWPLAYERID   5
#define DPMSGCMD_REQUESTNEWPLAYERID2  6
#define DPMSGCMD_NEWPLAYERIDREPLY     7
#define DPMSGCMD_CREATESESSION        8
#define DPMSGCMD_CREATENEWPLAYER      9
#define DPMSGCMD_SYSTEMMESSAGE        10
#define DPMSGCMD_DELETEPLAYER         11
#define DPMSGCMD_DELETEGROUP          12
#define DPMSGCMD_ENUMGROUPS           17
#define DPMSGCMD_FORWARDADDPLAYER     19
#define DPMSGCMD_PLAYERCHAT           22
#define DPMSGCMD_PLAYERCHAT2          23
#define DPMSGCMD_PLAYERCHATREPLY      24
#define DPMSGCMD_FORWARDADDPLAYERNACK 36
#define DPMSGCMD_PLAYERSLISTREPLY     41
#define DPMSGCMD_JUSTENVELOPE         1000
#define DPMSGCMD_JUSTENVELOPEREPLY    1001

typedef enum _dplay_version {
    DPLAY7  = 0x000b,
    DPLAY9  = 0x000e
} DPlayVersion;

/* */

typedef struct {
    uint32_t    g1;
    uint16_t    g2, g3;
    uint8_t     g4, g5, g6, g7, g8, g9, g10, g11;
} guid_t;



uint16_t dp_recv(int sd, uint8_t *buff);
void dp_info(uint8_t *data, int size);
uint8_t *dp_try_get_name(uint8_t **data, int players);
uint16_t dp_bind(int sd, struct sockaddr_in *peer);
void dp_query(in_addr_t host, uint16_t port, uint16_t lport, uint16_t type);
int tcp_recv(int sd, uint8_t *data, int size);
int getmm(uint8_t *data, uint8_t **val, int len);
int putmm(uint8_t *data, uint8_t *val, int len);
int getxx(uint8_t *data, void *ret, int bits);
int putxx(uint8_t *data, uint32_t num, int bits);
int getgi(uint8_t *data, guid_t *guid);
int putgi(uint8_t *data, uint32_t g1, uint16_t g2, uint16_t g3, uint8_t g4, uint8_t g5, uint8_t g6, uint8_t g7, uint8_t g8, uint8_t g9, uint8_t g10, uint8_t g11);
int getuc(uint8_t *data, uint8_t **ret);
uint8_t *show_guid(guid_t *g);
uint8_t *myitoa(unsigned num);
uint8_t *show_dp_type(uint16_t type);
void show_dp_flags(uint8_t *info, uint32_t flags);
void show_hex(uint8_t *info, uint8_t *data, int len);
int timeout(int sock);
in_addr_t resolv(char *host);
void std_err(void);



int main(int argc, char *argv[]) {
    struct  sockaddr_in peer;
    int         sl,
                sd,
                i,
                psz,
                retry   = 1;
    in_addr_t   host;
    uint16_t    port    = PORT,
                lport,
                len16;
    uint8_t     buff[BUFFSZ];

    uint16_t    query[] = {
                    DPMSGCMD_ENUMSESSIONSREQUEST,
                    DPMSGCMD_PLAYERSLISTREQUEST,
                    // DPMSGCMD_REQUESTNEWPLAYERID,
                    // DPMSGCMD_REQUESTNEWPLAYERID2,
                    // DPMSGCMD_PLAYERCHAT,
                    // DPMSGCMD_PLAYERCHAT2,
                    0
                };

#ifdef WIN32
    WSADATA    wsadata;
    WSAStartup(MAKEWORD(1,0), &wsadata);
#endif

    setbuf(stdout, NULL);

    fputs("\n"
        "DirectPlay 6/7 Info "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", stdout);

    if(argc < 2) {
        printf("\n"
            "Usage: %s <host> [port(%hu)]\n"
            "\n", argv[0], port);
        exit(1);
    }

    if(argc > 2) port = atoi(argv[2]);

    printf("- resolv %s", argv[1]);
    host = resolv(argv[1]);
    printf(" -> %s:%hu\n", inet_ntoa(*(struct in_addr *)&host), port);

    sl = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
    if(sl < 0) std_err();
    sd = 0;

    printf("- bind the listening port\n");
    lport = dp_bind(sl, &peer);
    printf("- port %hu bound\n", lport);

    for(i = 0; query[i]; i++) {
        printf("- send query %s\n", show_dp_type(query[i]));
        dp_query(host, port, lport, query[i]);

        if(!sd) {
            psz = SOCKADDRSZ;
            if(timeout(sl) < 0) {
                if(retry) {
                    i--;    // it will be reincremented later
                    retry = 0;
                    printf("- no connection received, I retry to send the packet\n");
                    continue;
                }
                printf("\nError: the server has not connected to us, it's probably not online\n");
                close(sl);
                exit(1);
            }
            sd = accept(sl, (struct sockaddr *)&peer, &psz);
            if(sd < 0) std_err();
        }

        len16 = dp_recv(sd, buff);
        if(len16 == (uint16_t)-1) goto error;

        dp_info(buff, len16);
    }

    close(sl);
    close(sd);
    return(0);
error:
    close(sl);
    close(sd);
    printf("\nError: connection lost\n");
    return(1);
}



uint16_t dp_recv(int sd, uint8_t *buff) {
    uint16_t    len16;

    if(timeout(sd) < 0)                             return(-1);
    if(tcp_recv(sd, buff,       2)          < 0)    return(-1);
    getxx(buff, &len16, 16);
    if((len16 < 2) || (len16 > BUFFSZ))             return(-1);
    printf("- receive %u bytes\n", len16);
    if(tcp_recv(sd, buff + 2,   len16 - 2)    < 0)  return(-1);
    return(len16);
}



void dp_info(uint8_t *data, int size) { /* size is not really used for lazyness... */
    int         i;
    uint8_t     *p,
                *players_data;

    struct {
        uint16_t    sign;
        SOCKADDR    *peer;
        uint8_t     *action;
        uint16_t    type;
        uint16_t    version;
    } x;
    struct {
        uint32_t    dwSize;
        uint32_t    dwFlags;
        guid_t      guidInstance;
        guid_t      guidApplication;
        uint32_t    dwMaxPlayers;
        uint32_t    dwCurrentPlayers;
        uint32_t    lpszSessionName;
        uint32_t    lpszPassword;
        uint32_t    dwReserved1;
        uint32_t    dwReserved2;
        uint32_t    dwUser1;
        uint32_t    dwUser2;
        uint32_t    dwUser3;
        uint32_t    dwUser4;
        uint32_t    dwUnknown;
        uint8_t     *SessionName;
    } es;
    struct {
        uint8_t     *name;
        SOCKADDR    *peer1;
        SOCKADDR    *peer2;
    } pl;
    struct {
        uint32_t    dpidNewPlayerId;
        uint8_t     *Unknown;
    } np;

    p = data;
    p += 2;                                         /* packet size      */
    p += getxx(p, &x.sign,          16);            /* dplay 6/7 sign   */
    p += getmm(p, (void *)&x.peer,  SOCKADDRSZ);    /* address          */
    p += getmm(p, &x.action,        4);             /* action           */
    p += getxx(p, &x.type,          16);            /* type             */
    p += getxx(p, &x.version,       16);            /* version          */

    fputc('\n', stdout);
    printf("  signature/ID      0x%04hx\n", x.sign);
    printf("  action            %.4s\n",    x.action);
    printf("  type              %s\n",      show_dp_type(x.type));
    printf("  version           0x%04hx\n", x.version);

    if(x.type == DPMSGCMD_ENUMSESSIONSREPLY) {
        p += getxx(p, &es.dwSize,               32);
        p += getxx(p, &es.dwFlags,              32);
        p += getgi(p, &es.guidInstance);
        p += getgi(p, &es.guidApplication);
        p += getxx(p, &es.dwMaxPlayers,         32);
        p += getxx(p, &es.dwCurrentPlayers,     32);
        p += getxx(p, &es.lpszSessionName,      32);
        p += getxx(p, &es.lpszPassword,         32);
        p += getxx(p, &es.dwReserved1,          32);
        p += getxx(p, &es.dwReserved2,          32);
        p += getxx(p, &es.dwUser1,              32);
        p += getxx(p, &es.dwUser2,              32);
        p += getxx(p, &es.dwUser3,              32);
        p += getxx(p, &es.dwUser4,              32);
        p += getxx(p, &es.dwUnknown,            32);
        p += getuc(p, &es.SessionName);

        show_dp_flags("  flags             ",   es.dwFlags);
        printf("  instance guid     %s\n",      show_guid(&es.guidInstance));
        printf("  application guid  %s\n",      show_guid(&es.guidApplication));
        printf("  players           %u/%u\n",   es.dwCurrentPlayers, es.dwMaxPlayers);
        printf("  session name      %s\n",      es.SessionName);

    } else if(x.type == DPMSGCMD_NEWPLAYERIDREPLY) {
        p += getxx(p, &np.dpidNewPlayerId,      32);
        p += getmm(p, &np.Unknown,              36);

        printf("  new player ID     0x%08x\n",  np.dpidNewPlayerId);

    } else if(x.type == DPMSGCMD_PLAYERSLISTREPLY) {
        p += 28;
        p += getxx(p, &es.dwSize,               32);
        p += getxx(p, &es.dwFlags,              32);
        p += getgi(p, &es.guidInstance);
        p += getgi(p, &es.guidApplication);
        p += getxx(p, &es.dwMaxPlayers,         32);
        p += getxx(p, &es.dwCurrentPlayers,     32);
        p += getxx(p, &es.lpszSessionName,      32);
        p += getxx(p, &es.lpszPassword,         32);
        p += getxx(p, &es.dwReserved1,          32);
        p += getxx(p, &es.dwReserved2,          32);
        p += getxx(p, &es.dwUser1,              32);
        p += getxx(p, &es.dwUser2,              32);
        p += getxx(p, &es.dwUser3,              32);
        p += getxx(p, &es.dwUser4,              32);
        p += getuc(p, &es.SessionName);

        show_dp_flags("  flags             ",   es.dwFlags);
        printf("  instance guid     %s\n",      show_guid(&es.guidInstance));
        printf("  application guid  %s\n",      show_guid(&es.guidApplication));
        printf("  players           %u/%u\n",   es.dwCurrentPlayers, es.dwMaxPlayers);
        printf("  dwUser data       0x%08x 0x%08x 0x%08x 0x%08x\n",
            es.dwUser1, es.dwUser2, es.dwUser3, es.dwUser4);
        printf("  session name      %s\n",      es.SessionName);
        free(es.SessionName);

        players_data = p + (es.dwCurrentPlayers * 53);  // almost exact...

        for(i = 0; i < es.dwCurrentPlayers; i++) {
            printf("  player %d:\n", i + 1);
            p += 21;                                    // 20 + one size byte ignored here
            p += getmm(p, (void *)&pl.peer1,    SOCKADDRSZ);
            p += getmm(p, (void *)&pl.peer2,    SOCKADDRSZ);
            printf("    addr            %s:%hu",
                inet_ntoa(pl.peer1->sin_addr), ntohs(pl.peer1->sin_port));
            printf(" / %s:%hu\n",
                inet_ntoa(pl.peer2->sin_addr), ntohs(pl.peer2->sin_port));

            pl.name = dp_try_get_name(&players_data, es.dwCurrentPlayers - i);
            if(pl.name) {
                printf("    name            %s\n", pl.name);
                free(pl.name);
            }
        }

    } else if(x.type == DPMSGCMD_GETNAMETABLEREPLY) {
        /* actually not supported */

    }

    size -= (p - data);
    if(size > 0) {
        printf("  raw data not handled:\n");
        show_dump(p, size, stdout);
    }

    fputc('\n', stdout);
}



            /* EXPERIMENTAL!!!!!!!!!!!!!!!!!!!!! */
uint8_t *dp_try_get_name(uint8_t **data, int players) {
    uint32_t    num32;
    uint8_t     *name,
                *p;

    p = *data;

    p += getxx(p, &num32,               32);
    if(num32 != 0x10)                       return(NULL);
    p += getxx(p, &num32,               32);
    p += 12;
    if((num32 != 0x4) && (num32 != 0xc))    return(NULL);
    p += getuc(p, &name);
    p += 33;

    *data = p;
    return(name);
}



uint16_t dp_bind(int sd, struct sockaddr_in *peer) {
    uint16_t    port    = 2300;

    peer->sin_addr.s_addr = INADDR_ANY;
    peer->sin_port        = htons(port);
    peer->sin_family      = AF_INET;

    for(;;) {
        if(!bind(sd, (struct sockaddr *)peer, SOCKADDRSZ)) break;
        peer->sin_port    = htons(++port);
    }
    if(listen(sd, 1)
      < 0) std_err();

    return(port);
}



void dp_query(in_addr_t host, uint16_t port, uint16_t lport, uint16_t type) {
    struct  sockaddr_in peer,
                        client;
    int         sd;
    uint8_t     buff[BUFFSZ],
                *p;

    peer.sin_addr.s_addr = host;
    peer.sin_port        = htons(port);
    peer.sin_family      = AF_INET;

    memset(&client, 0, SOCKADDRSZ);
    client.sin_port      = htons(lport);
    client.sin_family    = AF_INET;

    p = buff;
    p += 2;
    p += putxx(p, DPLAY_ID,         16);            /* dplay 6/7 sign   */
    p += putmm(p, (void *)&client,  SOCKADDRSZ);    /* client address   */
    p += putmm(p, ACTION_STRING,    4);             /* action           */
    p += putxx(p, type,             16);            /* type             */
    p += putxx(p, DPLAY9,           16);            /* version          */
    p += putgi(p, COOLGUID);                        /* GUID             */
    p += putxx(p, 0,                32);            /* password size    */
    p += putxx(p, 0xffffffff,       32);            /* flags (all!)     */
    putxx(buff,   p - buff,         16);            /* size (max 4095)  */

    sd = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
    if(sd < 0) std_err();
    if(sendto(sd, buff, p - buff, 0, (struct sockaddr *)&peer, SOCKADDRSZ)
      < 0) std_err();
    close(sd);
}



int tcp_recv(int sd, uint8_t *data, int size) {
    int     t;

    while(size) {
        t = recv(sd, data, size, 0);
        if(t <= 0) return(-1);
        data += t;
        size -= t;
    }
    return(0);
}



int getmm(uint8_t *data, uint8_t **val, int len) {
    *val = data;
    return(len);
}



int putmm(uint8_t *data, uint8_t *val, int len) {
    memcpy(data, val, len);
    return(len);
}



int getxx(uint8_t *data, void *ret, int bits) {
    uint32_t    num;
    int         i,
                bytes;

    bytes = bits >> 3;

    for(num = i = 0; i < bytes; i++) {
        num |= data[i] << (i << 3);
    }

    switch(bits) {
        case 8:  *(uint8_t  *)ret = num;    break;
        case 16: *(uint16_t *)ret = num;    break;
        case 32: *(uint32_t *)ret = num;    break;
    }
    return(bytes);
}



int putxx(uint8_t *data, uint32_t num, int bits) {
    int         i,
                bytes;

    bytes = bits >> 3;

    for(i = 0; i < bytes; i++) {
        data[i] = num >> (i << 3);
    }

    return(bytes);
}



int getgi(uint8_t *data, guid_t *guid) {
    uint8_t     *p;

    p = data;
    p += getxx(p, &guid->g1,  32);
    p += getxx(p, &guid->g2,  16);
    p += getxx(p, &guid->g3,  16);
    p += getxx(p, &guid->g4,  8);
    p += getxx(p, &guid->g5,  8);
    p += getxx(p, &guid->g6,  8);
    p += getxx(p, &guid->g7,  8);
    p += getxx(p, &guid->g8,  8);
    p += getxx(p, &guid->g9,  8);
    p += getxx(p, &guid->g10, 8);
    p += getxx(p, &guid->g11, 8);

    return(p - data);
}



int putgi(uint8_t *data, uint32_t g1, uint16_t g2, uint16_t g3, uint8_t g4, uint8_t g5, uint8_t g6, uint8_t g7, uint8_t g8, uint8_t g9, uint8_t g10, uint8_t g11) {
    uint8_t     *p;

    p = data;
    p += putxx(p, g1,  32);
    p += putxx(p, g2,  16);
    p += putxx(p, g3,  16);
    p += putxx(p, g4,  8);
    p += putxx(p, g5,  8);
    p += putxx(p, g6,  8);
    p += putxx(p, g7,  8);
    p += putxx(p, g8,  8);
    p += putxx(p, g9,  8);
    p += putxx(p, g10, 8);
    p += putxx(p, g11, 8);

    return(p - data);
}



int getuc(uint8_t *data, uint8_t **ret) {
    int         len;
    uint8_t     *i,
                *o,
                *mem;

    for(i = data; *i; i += 2);
    len = i + 2 - data;
    mem = malloc(len >> 1); // remember to free later

    for(i = data, o = mem; *i; i += 2, o++) {
        *o = *i;
    }
    *o = 0;

    *ret = mem;
    return(len);
}



uint8_t *show_guid(guid_t *g) {
    static uint8_t  mini[37];

    sprintf(mini,
        "%08x-%04x-%04x-%02x%02x-%02x%02x%02x%02x%02x%02x",
        g->g1, g->g2, g->g3, g->g4, g->g5, g->g6, g->g7, g->g8, g->g9, g->g10, g->g11);
    return(mini);
}



uint8_t *myitoa(unsigned num) {
    static uint8_t  mini[11];

    sprintf(mini, "0x%08x", num);
    return(mini);
}



uint8_t *show_dp_type(uint16_t type) {
    switch(type) {
        case DPMSGCMD_ENUMSESSIONSREPLY:    return("ENUMSESSIONSREPLY");
        case DPMSGCMD_ENUMSESSIONSREQUEST:  return("ENUMSESSIONSREQUEST");
        case DPMSGCMD_GETNAMETABLEREPLY:    return("GETNAMETABLEREPLY");
        case DPMSGCMD_PLAYERSLISTREQUEST:   return("PLAYERSLISTREQUEST");
        case DPMSGCMD_REQUESTNEWPLAYERID:   return("REQUESTNEWPLAYERID");
        case DPMSGCMD_REQUESTNEWPLAYERID2:  return("REQUESTNEWPLAYERID2");
        case DPMSGCMD_NEWPLAYERIDREPLY:     return("NEWPLAYERIDREPLY");
        case DPMSGCMD_CREATESESSION:        return("CREATESESSION");
        case DPMSGCMD_CREATENEWPLAYER:      return("CREATENEWPLAYER");
        case DPMSGCMD_SYSTEMMESSAGE:        return("SYSTEMMESSAGE");
        case DPMSGCMD_DELETEPLAYER:         return("DELETEPLAYER");
        case DPMSGCMD_DELETEGROUP:          return("DELETEGROUP");
        case DPMSGCMD_ENUMGROUPS:           return("ENUMGROUPS");
        case DPMSGCMD_FORWARDADDPLAYER:     return("FORWARDADDPLAYER");
        case DPMSGCMD_PLAYERCHAT:           return("PLAYERCHAT");
        case DPMSGCMD_PLAYERCHAT2:          return("PLAYERCHAT2");
        case DPMSGCMD_PLAYERCHATREPLY:      return("PLAYERCHATREPLY");
        case DPMSGCMD_FORWARDADDPLAYERNACK: return("FORWARDADDPLAYERNACK");
        case DPMSGCMD_PLAYERSLISTREPLY:     return("PLAYERSLISTREPLY");
        case DPMSGCMD_JUSTENVELOPE:         return("JUSTENVELOPE");
        case DPMSGCMD_JUSTENVELOPEREPLY:    return("JUSTENVELOPEREPLY");
        default:                            return(myitoa(type));
    }
}



void show_dp_flags(uint8_t *info, uint32_t flags) {
#define DPCAPS_ISHOST                   0x00000002
#define DPCAPS_GROUPOPTIMIZED           0x00000008
#define DPCAPS_KEEPALIVEOPTIMIZED       0x00000010
#define DPCAPS_GUARANTEEDOPTIMIZED      0x00000020
#define DPCAPS_GUARANTEEDSUPPORTED      0x00000040
#define DPCAPS_SIGNINGSUPPORTED         0x00000080
#define DPCAPS_ENCRYPTIONSUPPORTED      0x00000100
#define DPPLAYERCAPS_LOCAL			    0x00000800
#define DPCAPS_ASYNCCANCELSUPPORTED     0x00001000
#define DPCAPS_ASYNCCANCELALLSUPPORTED  0x00002000
#define DPCAPS_SENDTIMEOUTSUPPORTED     0x00004000
#define DPCAPS_SENDPRIORITYSUPPORTED    0x00008000
#define DPCAPS_ASYNCSUPPORTED 		    0x00010000
#define DPCAPS_PRINT(A,B) if(flags & A) fputs(B, stdout);
    fputs(info, stdout);
    DPCAPS_PRINT(DPCAPS_ISHOST,                     "ISHOST,");
    DPCAPS_PRINT(DPCAPS_GROUPOPTIMIZED,             "GROUPOPTIMIZED,");
    DPCAPS_PRINT(DPCAPS_KEEPALIVEOPTIMIZED,         "KEEPALIVEOPTIMIZED,");
    DPCAPS_PRINT(DPCAPS_GUARANTEEDOPTIMIZED,        "GUARANTEEDOPTIMIZED,");
    DPCAPS_PRINT(DPCAPS_GUARANTEEDSUPPORTED,        "GUARANTEEDSUPPORTED,");
    DPCAPS_PRINT(DPCAPS_SIGNINGSUPPORTED,           "SIGNINGSUPPORTED,");
    DPCAPS_PRINT(DPCAPS_ENCRYPTIONSUPPORTED,        "ENCRYPTIONSUPPORTED,");
    DPCAPS_PRINT(DPPLAYERCAPS_LOCAL,                "LOCAL,");
    DPCAPS_PRINT(DPCAPS_ASYNCCANCELSUPPORTED,       "ASYNCCANCELSUPPORTED,");
    DPCAPS_PRINT(DPCAPS_ASYNCCANCELALLSUPPORTED,    "ASYNCCANCELALLSUPPORTED,");
    DPCAPS_PRINT(DPCAPS_SENDTIMEOUTSUPPORTED,       "SENDTIMEOUTSUPPORTED,");
    DPCAPS_PRINT(DPCAPS_SENDPRIORITYSUPPORTED,      "SENDPRIORITYSUPPORTED,");
    DPCAPS_PRINT(DPCAPS_ASYNCSUPPORTED,             "ASYNCSUPPORTED,");
    fputc('\n', stdout);
}



void show_hex(uint8_t *info, uint8_t *data, int len) {
    int     i,
            ilen,
            rest;

    ilen = strlen(info);
    rest = (79 - len) / 3;

    fputs(info, stdout);
    for(i = 0; i < len;) {
        printf("%02x ", data[i]);
        if(!(++i % rest)) {
            printf("\n%*s", ilen, "");
        }
    }
    fputc('\n', stdout);
}



int timeout(int sock) {
    struct  timeval tout;
    fd_set  fd_read;
    int     err;

    tout.tv_sec  = 3;
    tout.tv_usec = 0;
    FD_ZERO(&fd_read);
    FD_SET(sock, &fd_read);
    err = select(sock + 1, &fd_read, NULL, NULL, &tout);
    if(err < 0) std_err();
    if(!err) return(-1);
    return(0);
}



in_addr_t resolv(char *host) {
    struct      hostent *hp;
    in_addr_t   host_ip;

    host_ip = inet_addr(host);
    if(host_ip == INADDR_NONE) {
        hp = gethostbyname(host);
        if(!hp) {
            printf("\nError: Unable to resolv hostname (%s)\n",
                host);
            exit(1);
        } else host_ip = *(in_addr_t *)(hp->h_addr);
    }
    return(host_ip);
}



#ifndef WIN32
    void std_err(void) {
        perror("\nError");
        exit(1);
    }
#endif

