/*
    Copyright 2009 Luigi Auriemma

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    http://www.gnu.org/licenses/gpl-2.0.txt
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include "des.h"
#include "show_dump2.h"

typedef uint8_t     u8;
typedef uint16_t    u16;
typedef uint32_t    u32;
typedef uint64_t    u64;



#define VER         "0.1a"
#define DEBUG       // hexdump of key, iv and data
#define PRINTF64(x) (u32)(((x) >> 32) & 0xffffffff), (u32)((x) & 0xffffffff)    // I64x, llx? blah



extern void tiger(u8 *, u64, u8 *);
void hex2byte(u8 *in, u8 *out);
void bdTicket_info(u8 *bdTicket);
void bdcrypto(u8 *server_key, u8 *bdTicket);
u8 *base64_encode(u8 *data, int *size);
u8 *base64_decode(u8 *data, int *size);



int main(int argc, char *argv[]) {
    u8      *server_key,
            *bdTicket;

    fputs("\n"
        "DemonWare bdcrypto bdTicket decryption "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", stdout);

    if(argc < 3) {
        printf("\n"
            "Usage: %s <server_cdkey/hash> <bdTicket>\n"
            "\n"
            "this tool shows the operations performed on a server based on the DemonWare\n"
            "SDK like, for example, Call of Duty World at War where bdTicket is a parameter\n"
            "passed in the \"connect\" packet and this tool can decrypt its content\n"
            "examples:\n"
            "  %s AAAA-BBBB-CCCC-DDDD-EEEE MDAwMDExMTEyMzMzMzMzMzM0NDQ0NDQ0NE==\n"
            "  %s 2c482ceab8c48a9e846ef8d89342189264f01db9412e84ff MDAwMDExMTEy...\n"
            "\n", argv[0], argv[0], argv[0]);
        exit(1);
    }

    server_key = argv[1];
    bdTicket   = argv[2];

    bdcrypto(server_key, bdTicket);
    bdTicket_info(bdTicket);
    return(0);
}



void hex2byte(u8 *in, u8 *out) {
    int     t;

    for(; in[0] && in[1]; in += 2, out++) {
        if(sscanf(in, "%02x", &t) != 1) break;
        *out = t;
    }
}



u32 net32(u32 num) {
    int         endian = 1; // big endian

    if(!*(char *)&endian) return(num);
    return(((num & 0xff000000) >> 24) |
           ((num & 0x00ff0000) >>  8) |
           ((num & 0x0000ff00) <<  8) |
           ((num & 0x000000ff) << 24));
}



void bdTicket_info(u8 *bdTicket) {
    #pragma pack(1)
    typedef struct {
        u32     challenge;
        u8      license_type;
        u64     zero;
        u32     timestamp;
        u64     userid;
        u64     licenseid;
    } bdcrypto_ticket_t;
    bdcrypto_ticket_t   *bdcrypto_ticket;
    #pragma pack()

    bdcrypto_ticket = (bdcrypto_ticket_t *)bdTicket;
    printf("\n"
        "- Ticket challenge is %08x (must be %08x)\n"
        "- License type is %u\n"
        "- timestamp is %08x\n"
        "- UserID is %08x%08x\n" //%llx\n"
        "- LicenseID is %08x%08x\n", //%llx\n",
        bdcrypto_ticket->challenge, 0xefbdadde,
        bdcrypto_ticket->license_type,
        bdcrypto_ticket->timestamp,
        PRINTF64(bdcrypto_ticket->userid),
        PRINTF64(bdcrypto_ticket->licenseid));
}



void bdcrypto(u8 *server_key, u8 *bdTicket_input) {
    des3_context    ctx;
    int     bdTicket_len;
    u8      client_hash[24],
            server_hash[24],
            *bdTicket;

    if(strlen(server_key) == 48) {  // if it's a tiger192 hash
        hex2byte(server_key, server_hash);
    } else {
        tiger(server_key, strlen(server_key), server_hash);
    }
    des3_set3key_dec(&ctx, server_hash);

    bdTicket_len = strlen(bdTicket_input);
    bdTicket = base64_decode(bdTicket_input, &bdTicket_len);
    if(bdTicket_len < 4) {
        printf("\nError: bdTicket is too short\n"); 
        exit(1);
    }

    //printf("- Got IV %u from client!\n", *(u32 *)bdTicket);
    tiger(bdTicket, 4, client_hash);
    bdTicket_len -= 4;

#ifdef DEBUG
    printf("\n- 3des key:\n");
    show_dump(2, server_hash, 24, stdout);
    printf("\n- 3des IV:\n");
    show_dump(2, client_hash, 24, stdout);
    printf("\n- encrypted data:\n");
    show_dump(2, bdTicket + 4, bdTicket_len, stdout);
#endif

    des3_crypt_cbc(&ctx, DES_DECRYPT, bdTicket_len, client_hash, bdTicket + 4, bdTicket);

#ifdef DEBUG
    printf("\n- decrypted data:\n");
    show_dump(2, bdTicket, bdTicket_len, stdout);
#endif

    memcpy(bdTicket_input, bdTicket, bdTicket_len);
    free(bdTicket);
}



u8 *base64_encode(u8 *data, int *size) {
    int     len,
            a,
            b,
            c;
    u8      *buff,
            *p;
    static const u8 base[64] = {
        'A','B','C','D','E','F','G','H','I','J','K','L','M','N','O','P',
        'Q','R','S','T','U','V','W','X','Y','Z','a','b','c','d','e','f',
        'g','h','i','j','k','l','m','n','o','p','q','r','s','t','u','v',
        'w','x','y','z','0','1','2','3','4','5','6','7','8','9','+','/'
    };

    if(!size || (*size < 0)) {      // use size -1 for auto text size!
        len = strlen(data);
    } else {
        len = *size;
    }
    buff = malloc(((len / 3) << 2) + 6);
    if(!buff) return(NULL);

    p = buff;
    do {
        a = data[0];
        b = data[1];
        c = data[2];
        *p++ = base[(a >> 2) & 63];
        *p++ = base[(((a &  3) << 4) | ((b >> 4) & 15)) & 63];
        *p++ = base[(((b & 15) << 2) | ((c >> 6) &  3)) & 63];
        *p++ = base[c & 63];
        data += 3;
        len  -= 3;
    } while(len > 0);
    *p = 0;

    for(; len < 0; len++) *(p + len) = '=';

    if(size) *size = p - buff;
    return(buff);
}



u8 *base64_decode(u8 *data, int *size) {
    int     len,
            xlen,
            a,
            b,
            c;
    u8      *buff,
            *limit,
            *p;
    static const u8 base[128] = {
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x3e, 0x00, 0x00, 0x00, 0x3f,
        0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b,
        0x3c, 0x3d, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06,
        0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e,
        0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16,
        0x17, 0x18, 0x19, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20,
        0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28,
        0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, 0x30,
        0x31, 0x32, 0x33, 0x00, 0x00, 0x00, 0x00, 0x00
    };

    if(!size || (*size < 0)) {      // use size -1 for auto size!
        len = strlen(data);
    } else {
        len = *size;
    }
    xlen = ((len >> 2) * 3) + 1;    // NULL included in output for text
    buff = malloc(xlen);
    if(!buff) return(NULL);

    p = buff;
    limit = data + len;
    for(;;) {
#define B64NEXT do {            /* auto chars cleaning */   \
                    c = *data;                              \
                    if((data == limit) || (c == '=')) {     \
                        c = 0;                              \
                        break;                              \
                    }                                       \
                    data++;                                 \
                } while(c && ((c <= ' ') || (c > 0x7f)));   \
                if(!c) break;

        B64NEXT;
        a = base[c];

        B64NEXT;
        b = base[c];
        *p++ = (a << 2)        | (b >> 4);

        B64NEXT;
        a = base[c];
        *p++ = ((b & 15) << 4) | (a >> 2);

        B64NEXT;
        *p++ = ((a & 3) << 6)  | base[c];

#undef B64NEXT
    }
    *p = 0;

    len = p - buff;
    if(size) *size = len;
    if((len + 1) != xlen) {             // save memory
        buff = realloc(buff, len + 1);  // NULL included!
        if(!buff) return(NULL);
    }
    return(buff);
}


