/*
    Copyright 2008 Luigi Auriemma

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    http://www.gnu.org/licenses/gpl-2.0.txt
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <ctype.h>
#include <sys/stat.h>
#include <zlib.h>

#ifdef WIN32
    #include <direct.h>
    #define PATHSLASH   '\\'
    #define make_dir(x) mkdir(x)
#else
    #include <unistd.h>
    #define stricmp     strcasecmp
    #define PATHSLASH   '/'
    #define make_dir(x) mkdir(x, 0755)
#endif

typedef uint8_t     u8;
typedef uint16_t    u16;
typedef uint32_t    u32;



#define VER         "0.1"



u8 *get_ext(u8 *fname);
u8 *mgetz(u8 *data, int datasz);
u8 *gunzip(u8 *in, int insz, int *ret);
int egosoft_x_dumpa(FILE *fd, u8 *fname, u32 fsize);
void xor(u8 *in, u8 *out, int len, u8 x);
void egosoft_x_cat(u8 *data, int datasz);
void clean_fname(u8 *fname);
void create_dir(u8 *fname);
int check_wildcard(u8 *fname, u8 *wildcard);
int check_overwrite(u8 *fname);
u8 *fdload(u8 *fname, int *fsize);
void myalloc(u8 **data, unsigned wantsize, unsigned *currsize);
void myfw(FILE *fd, void *data, unsigned size);
void myfr(FILE *fd, void *data, unsigned size);
void std_err(void);



int     listonly    = 0,
        overwrite   = 0;
u8      *wildcard   = NULL;



int main(int argc, char *argv[]) {
    FILE    *fd     = NULL;
    int     i,
            fsize,
            catsz,
            files;
    u8      *cat,
            *dname,
            *fname,
            *fdir,
            *line,
            *ext,
            *p;

    setbuf(stdout, NULL);

    fputs("\n"
        "Egosoft X series CAT/DAT files extractor "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", stdout);

    if(argc < 3) {
        printf("\n"
            "Usage: %s [options] <file.CAT> <output_folder>\n"
            "\n"
            "Options:\n"
            "-l      list files without extracting them\n"
            "-f \"W\"  extract only the files which match the wildcard W like \"*.xml\"\n"
            "-o      overwrites existent files without asking\n"
            "\n", argv[0]);
        exit(1);
    }

    argc -= 2;
    for(i = 1; i < argc; i++) {
        if(((argv[i][0] != '-') && (argv[i][0] != '/')) || (strlen(argv[i]) != 2)) {
            printf("\nError: wrong argument (%s)\n", argv[i]);
            exit(1);
        }
        switch(argv[i][1]) {
            case 'l': listonly  = 1;            break;
            case 'f': wildcard  = argv[++i];    break;
            case 'o': overwrite = 1;            break;
            default: {
                printf("\nError: wrong argument (%s)\n", argv[i]);
                exit(1);
            }
        }
    }
    fname = argv[argc];
    fdir  = argv[argc + 1];

    ext = get_ext(fname);   // in case the user select a DAT file instead of the CAT one
    if(!stricmp(ext, "dat")) strcpy(ext, "cat");

    cat = fdload(fname, &catsz);
    egosoft_x_cat(cat, catsz);

    line = mgetz(cat, catsz);       // get the DAT filename, the first line
    if(!listonly) {
        p = strrchr(fname, '\\');   // remove the file.cat from the input name
        if(!p) p = strrchr(fname, '/');
        if(!p) p = fname;
        *p = 0;

        p = strrchr(line, '\\');    // remove possible ..\ from the DAT name
        if(!p) p = strrchr(line, '/');
        if(p) line = p + 1;

        dname = malloc(strlen(fname) + strlen(line) + 1);
        if(fname[0]) {
            sprintf(dname, "%s%c%s", fname, PATHSLASH, line);
        } else {
            strcpy(dname, line);
        }

        printf("- open DAT file %s\n", dname);
        fd = fopen(dname, "rb");
        if(!fd) std_err();

        printf("- set output folder %s\n", fdir);
        if(chdir(fdir) < 0) std_err();
    }

    printf("- start %s:\n\n", listonly ? "listing" : "extraction");
    printf("  size       filename\n"
           "---------------------\n");

    files = 0;
    while((line = mgetz(cat, catsz))) {
        p = strrchr(line, ' ');
        if(!p) {
            printf("- alert: no filesize for the line: %s\n", line);
            continue;
        }
        *p = 0;
        fsize = atoi(p + 1);
        files += egosoft_x_dumpa(fd, line, fsize);
    }

    if(fd) fclose(fd);
    free(cat);
    printf("\n- %s %d files\n", listonly ? "listed" : "extracted", files);
    return(0);
}



u8 *get_ext(u8 *fname) {
    u8      *p;

    p = strrchr(fname, '.');
    if(!p) return(fname + strlen(fname));
    return(p + 1);
}



u8 *mgetz(u8 *data, int datasz) {
    static  int off = 0;
    u8      *ret;

    if(off >= datasz) return(NULL);
    for(ret = data + off; off < datasz; off++) {
        if(data[off] == '\r') {
            data[off] = 0;
        } else if(data[off] == '\n') {
            data[off] = 0;
            off++;
            break;
        }
    }
    return(ret);
}



u8 *gunzip(u8 *in, int insz, int *ret) {
    static int  outsz   = 0;
    static u8   *out    = NULL;
    z_stream    z;
    int         outlen;
    u8          flags,
                *p;

    flags = in[3];
    outlen = in[insz - 4] | (in[insz - 3] << 8) | (in[insz - 2] << 16) | (in[insz - 1] << 24);

    p = in + 10;
    if(flags & 2)  p += 2;
    if(flags & 4)  p += 2 + (p[0] | (p[1] << 8));
    if(flags & 8)  p += strlen(p) + 1;
    if(flags & 16) p += strlen(p) + 1;
    insz -= ((p - in) + 4);
    if(insz < 0) return(NULL);
    myalloc(&out, outlen, &outsz);

    z.zalloc = (alloc_func)0;
    z.zfree  = (free_func)0;
    z.opaque = (voidpf)0;
    if(inflateInit2(&z, -15)) {
        printf("\nError: zlib initialization error\n");
        exit(1);
    }
    z.next_in   = p;
    z.avail_in  = insz;
    z.next_out  = out;
    z.avail_out = outlen;
    if(inflate(&z, Z_FINISH) != Z_STREAM_END) {
        printf("- invalid gzipped data\n");
        inflateEnd(&z);
        exit(1);    // is better to show the error
        return(NULL);
    }
    *ret = z.total_out;
    inflateEnd(&z);
    return(out);
}



int egosoft_x_dumpa(FILE *fd, u8 *fname, u32 fsize) {
    static const u8 gzip_sign[3] = "\x1f\x8b\x08";
    static int  insz    = 0;
    static u8   *in     = NULL;
    FILE    *fdo;
    int     x2      = 0,
            gzipped = 0;
    u8      tmp[sizeof(gzip_sign)],
            *out    = NULL,
            x2_xor;

    clean_fname(fname);
    if(check_wildcard(fname, wildcard) < 0) {
        if(fseek(fd, fsize, SEEK_CUR)) std_err();   // needed if we skip a file
        return(0);
    }
    printf("  %-10u %s\n", fsize, fname);
    if(listonly) return(1);

    if(!overwrite && (check_overwrite(fname) < 0)) {
        if(fseek(fd, fsize, SEEK_CUR)) std_err();   // needed if we skip a file
        return(0);
    }
    create_dir(fname);
    fdo = fopen(fname, "wb");
    if(!fdo) std_err();

    myalloc(&in, fsize, &insz);
    myfr(fd, in, fsize);
    xor(in, in, fsize, 0x33);

    if(!memcmp(in, gzip_sign, sizeof(gzip_sign))) {
        gzipped = 1;
    } else {
        x2_xor = in[0] ^ 0xc8;
        xor(in + 1, tmp, sizeof(gzip_sign), x2_xor);
        if(!memcmp(tmp, gzip_sign, sizeof(gzip_sign))) {
            x2 = 1;
            gzipped = 1;
        }
    }

    if(x2) {
        fsize--;
        xor(in + 1, in, fsize, x2_xor);
    }
    if(gzipped) {
        out = gunzip(in, fsize, &fsize);    // fsize is untouched if gunzip fails
    }

    myfw(fdo, out ? out : in, fsize);
    fclose(fdo);
    return(1);
}



void xor(u8 *in, u8 *out, int len, u8 x) {
    int     i;

    for(i = 0; i < len; i++) {
        out[i] = in[i] ^ x;
    }
}



void egosoft_x_cat(u8 *data, int datasz) {
    int     i;

    for(i = 0; i < datasz; i++) {
        data[i] ^= i - 0x25;
    }
}



void clean_fname(u8 *fname) {
    u8      *p;

    for(p = fname; (*p == '\\') || (*p == '/') || (*p == '.'); p++);
    if(p != fname) memmove(fname, p, strlen(p) + 1);
    for(p = fname; *p; p++) {
        if((p[0] == ':') || ((p[0] == '.') && (p[1] == '.'))) p[0] = '_';
    }
}



void create_dir(u8 *fname) {
    struct  stat    xstat;
    u8      *p,
            *l;

    if(!stat(fname, &xstat)) return;

    for(p = fname;; p = l + 1) {
        l = strchr(p, '\\');
        if(!l) l = strchr(p, '/');
        if(!l) break;
        *l = 0;
        make_dir(fname);
        *l = PATHSLASH;
    }
}



int check_wildcard(u8 *fname, u8 *wildcard) {
    u8      *f,
            *w,
            *a;

    if(!fname || !wildcard) return(0);
    f = fname;
    w = wildcard;
    a = NULL;
    while(*f || *w) {
        if(!*w && !a) return(-1);
        if(*w == '?') {
            if(!*f) break;
            w++;
            f++;
        } else if(*w == '*') {
            w++;
            a = w;
        } else {
            if(!*f) break;
            if(tolower(*f) != tolower(*w)) {
                if(!a) return(-1);
                f++;
                w = a;
            } else {
                f++;
                w++;
            }
        }
    }
    if(*f || *w) return(-1);
    return(0);
}



int check_overwrite(u8 *fname) {
    FILE    *fd;
    u8      ans[16];

    fd = fopen(fname, "rb");
    if(!fd) return(0);
    fclose(fd);
    printf("  the file already exists, do you want to overwrite it (y/N)? ");
    fgets(ans, sizeof(ans), stdin);
    if(tolower(ans[0]) != 'y') return(-1);
    return(0);
}



u8 *fdload(u8 *fname, int *fsize) {
    struct stat xstat;
    FILE    *fd;
    int     size;
    u8      *buff;

    printf("- open %s\n", fname);
    fd = fopen(fname, "rb");
    if(!fd) std_err();
    fstat(fileno(fd), &xstat);
    size = xstat.st_size;
    buff = malloc(size);
    if(!buff) std_err();
    fread(buff, 1, size, fd);
    fclose(fd);
    *fsize = size;
    return(buff);
}



void myalloc(u8 **data, unsigned wantsize, unsigned *currsize) {
    if(wantsize <= *currsize) return;
    *data = realloc(*data, wantsize);
    if(!*data) std_err();
    *currsize = wantsize;
}



void myfw(FILE *fd, void *data, unsigned size) {
    if(fwrite(data, 1, size, fd) != size) {
        printf("\nError: impossible to write %d bytes to the output file, check your space\n", size);
        fclose(fd);
        exit(1);
    }
}



void myfr(FILE *fd, void *data, unsigned size) {
    if(fread(data, 1, size, fd) != size) {
        printf("\nError: incomplete input file, can't read %u bytes at offset %08x\n", size, (u32)ftell(fd));
        fclose(fd);
        exit(1);
    }
}



void std_err(void) {
    perror("\nError");
    exit(1);
}


