/*
    Copyright 2008,2009 Luigi Auriemma

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    http://www.gnu.org/licenses/gpl-2.0.txt
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <ctype.h>
#include <sys/stat.h>
#include "show_dump.h"

typedef uint8_t     u8;
typedef uint16_t    u16;
typedef uint32_t    u32;



#define VER         "0.2"



extern u8 *enctype1_decoder(u8 *, u8 *, int *); // for enctype1 decoding
extern u8 *enctype2_decoder(u8 *, u8 *, int *); // for enctype2 decoding
extern unsigned char *enctypex_decoder(unsigned char *, unsigned char *, unsigned char *, int *, void *);
extern int enctypex_decoder_convert_to_ipport(unsigned char *, int, unsigned char *, unsigned char *, int, int);
u8 *keyval(u8 *data, u8 *key);
u8 *fdload(u8 *fname, int *fsize);
void std_err(void);



int main(int argc, char *argv[]) {
    FILE    *fd;
    int     i,
            len,
            buffsz,
            enctype     = 0,
            list_mode   = 0;
    u8      *buff,
            *input      = NULL,
            *secure     = NULL,
            *validate   = NULL,
            *gamekey    = NULL,
            *output     = NULL,
            *lista      = NULL,
            *limit      = NULL,
            *p;

    setbuf(stdout, NULL);

    fputs("\n"
        "Enctype decoder/tester "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", stdout);

    if(argc < 3) {
        printf("\n"
            "Usage: %s [options] <enctype> <input_file>\n"
            "\n"
            "Options:\n"
            "-s SECURE    the server's challenge\n"
            "-v VALIDATE  the client's challenge\n"
            "-g GAMEKEY   the gamekey used by the requester gamename (the one used to log\n"
            "             into gamespy, not the requested game), for example Xn221z if\n"
            "             has been used gslive\n"
            "-l           parse the decrypted data and return the list of IP and ports in\n"
            "             binary form (4 bytes IP and 2 port), needed only for enctypeX\n"
            "-L           as above but the list is returned in textual form, IP:port\n"
            "-o FILE      specify the output file (default: hex dump)\n"
            "\n"
            "Required fields and supported enctypes:\n"
            "- enctype1: secure\n"
            "- enctype2: gamekey\n"
            "- enctypeX: validate gamekey\n"
            "use \"\" or anything else for the other not required fields\n"
            "\n"
            "input_file must contain ALL the data received from the master server,\n"
            "this allows the tool to automatically guessing the secure field for enctype1\n"
            "while for enctypeX you can just dump also the sent query so if guesses the"
            "validate field\n"
            "\n", argv[0]);
        exit(1);
    }

    argc -= 2;
    for(i = 1; i < argc; i++) {
        if(((argv[i][0] != '-') && (argv[i][0] != '/')) || (strlen(argv[i]) != 2)) {
            printf("\nError: wrong argument (%s)\n", argv[i]);
            exit(1);
        }
        switch(argv[i][1]) {
            case 's': secure    = argv[++i];    break;
            case 'v': validate  = argv[++i];    break;
            case 'g': gamekey   = argv[++i];    break;
            case 'l': list_mode = 1;            break;
            case 'L': list_mode = 2;            break;
            case 'o': output    = argv[++i];    break;
            default: {
                printf("\nError: wrong argument (%s)\n", argv[i]);
                exit(1);
            }
        }
    }
    p = argv[argc];
    p += strlen(p) - 1;
    enctype  = atoi(p);
    if(tolower(p[0] == 'x')) enctype = -1;
    input    = argv[argc + 1];

    buff = fdload(input, &buffsz);

    p = buff;
    if(enctype == 1) {
        printf("- enctype 1\n");
        if(!memcmp(p, "\\basic\\", 7)) {
            secure = keyval(p, "secure");
            printf("- secure %s\n", secure);
            p += strlen(p) + 1;
        }
        if(!secure) {
            printf("\nError: you must specify -s SECURE\n");
            exit(1);
        }
        len = buffsz - (p - buff);
        p = enctype1_decoder(secure, p, &len);

    } else if(enctype == 2) {
        printf("- enctype 2\n");
        if(!gamekey) {
            printf("\nError: you must specify -g GAMEKEY\n");
            exit(1);
        }
        if(!memcmp(p, "\\basic\\", 7)) {
            p += strlen(p) + 1;
        }
        len = buffsz - (p - buff);
        p = enctype2_decoder(gamekey, p, &len);

    } else if(enctype < 0) {
        printf("- enctypeX\n");
        if(!p[0]) {     // usually the data of the client is shorter than 255 bytes
            len = (p[0] << 8) | p[1];
            p += 2;
            p += 7;
            p += strlen(p) + 1;
            p += strlen(p) + 1;
            validate = strdup(p);
            printf("- validate %s\n", validate);
            p = buff + len;
        }
        if(!gamekey || !validate) {
            printf("\nError: you must specify -g GAMEKEY -v VALIDATE\n");
            exit(1);
        }
        len = buffsz - (p - buff);
        p = enctypex_decoder(gamekey, validate, p, &len, NULL);

        if(list_mode) {
            lista = malloc((len / 5) * 6);
            len = enctypex_decoder_convert_to_ipport(p, len, lista, NULL, 0, 0);
            limit = lista + len;
            p = lista;
        }

    } else {
        printf("\nError: unsupported enctype\n");
        exit(1);
    }

    // p contains the buffer with the decrypted data
    // len contains its length

    if(list_mode == 2) {
        lista = calloc(32, (len / 6) + 1);
        limit = p + len;
        for(len = 0; p < limit; p += 6) {
            len += sprintf(lista + len, "%u.%u.%u.%u:%u\r\n", p[0], p[1], p[2], p[3], (p[4] << 8) | p[5]);
        }
        p = lista;
    }

    if(output) {
        printf("- write file %s\n", output);
        fd = fopen(output, "wb");
        if(!fd) std_err();
        fwrite(p, 1, len, fd);
        fclose(fd);
    } else {
        printf("- hex dump:\n");
        show_dump(p, len, stdout);
    }

    free(buff);
    printf("- done\n");
    return(0);
}



u8 *keyval(u8 *data, u8 *key) {
    int     len,
            nt   = 0,
            skip = 1;
    u8      *p,
            *val;

    for(;;) {
        p = strchr(data, '\\');

        if(nt & 1) {
            // key
            if(p && !strnicmp(data, key, p - data)) {
                skip = 0;
            }
        } else {
            // value
            if(!skip) {
                if(!p) p = data + strlen(data);
                len = p - data;
                val = malloc(len + 1);
                if(!val) std_err();
                memcpy(val, data, len);
                val[len] = 0;
                return(val);
            }
        }

        if(!p) break;
        nt++;
        data = p + 1;
    }
    return(NULL);
}



u8 *fdload(u8 *fname, int *fsize) {
    struct stat xstat;
    FILE    *fd;
    int     size;
    u8      *buff;

    printf("- open file %s\n", fname);
    fd = fopen(fname, "rb");
    if(!fd) std_err();
    fstat(fileno(fd), &xstat);
    size = xstat.st_size;
    buff = malloc(size);
    if(!buff) std_err();
    fread(buff, 1, size, fd);
    fclose(fd);
    *fsize = size;
    return(buff);
}



void std_err(void) {
    perror("\nError");
    exit(1);
}


