/*

    Copyright 2004,2005,2006 Luigi Auriemma

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    http://www.gnu.org/licenses/gpl.txt

Windows Winpcap version
*/

#include <stdio.h>
#include <stdlib.h>
#include <pcap.h>
#include <string.h>
#include <winsock.h>
#include <zlib.h>
#include "ether_hdrlen.h"



#define VER         "0.1"
#define PORT        17900
#define GCFILE      "gcsniff.log"



void dispatcher_handler(u_char *temp1, const struct pcap_pkthdr *header, const u_char *pkt_data);
u_char *gamecenter_dec(u_char *encdata, u_char *data, int *size);
int do_inflate(u_char *in, int inlen, u_char *out, int outlen);
u_int resolv(char *host);
void std_err(void);



struct iphdr {
    u_int   ihl:4;
    u_int   version:4;
    u_char  tos;
    u_short tot_len;
    u_short id;
    u_short frag_off;
    u_char  ttl;
    u_char  protocol;
    u_short check;
    u_int   saddr;
    u_int   daddr;
} *iphdr;

struct tcphdr {
    u_short source;
    u_short dest;
    u_int   seq;
    u_int   ack_seq;
    u_short res1:4;
    u_short doff:4;
    u_short fin:1;
    u_short syn:1;
    u_short rst:1;
    u_short psh:1;
    u_short ack:1;
    u_short urg:1;
    u_short res2:2;
    u_short window;
    u_short check;
    u_short urg_ptr;
} *tcphdr;

FILE    *fdlog;
u_char  gchash[33];
u_short port;



int main(int argc, char *argv[]) {
    char        errbuf[PCAP_ERRBUF_SIZE];
    pcap_t      *fp;
    int         i,
                inum;
    pcap_if_t   *alldevs,
                *d;


    setbuf(stdout, NULL);

    fputs("\n"
        "Cyanide GameCenter real-time data decoder "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", stderr);

    if(pcap_findalldevs(&alldevs, errbuf) < 0) {
        fprintf(stderr,"Error in pcap_findalldevs: %s\n", errbuf);
        exit(1);
    }

    fprintf(stderr, "Interfaces list:\n\n");
    for(i = 0, d = alldevs; d; d = d->next) {
        fprintf(stderr, "%d. %s", ++i, d->name);
        if(d->description) fprintf(stderr, " (%s)\n", d->description);
            else fprintf(stderr, " (No description available)\n");
    }

    if(!i) {
        fprintf(stderr, "\nNo interfaces found! Make sure WinPcap is installed.\n");
        pcap_freealldevs(alldevs);
        exit(1);
    } else if(i > 1) {
        fprintf(stderr, "\nEnter the interface number (1 - %d): ",i);
        scanf("%d", &inum);

        if(inum < 1 || inum > i) {
            fprintf(stderr, "\nInterface number out of range.\n");
            pcap_freealldevs(alldevs);
            exit(1);
        }
    } else inum = 1;

    for(inum--, d = alldevs, i = 0; i < inum; d = d->next, i++);
    fprintf(stderr, "\nAdapter used: %s\n\n", d->name);

    fp = pcap_open_live(d->name, 65535, 1, 1000, errbuf);
    if(!fp) {
        printf("\nError: %s\n", errbuf);
        pcap_freealldevs(alldevs);
        exit(1);
    }

    pcap_freealldevs(alldevs);

    fdlog = fopen(GCFILE, "ab");
    if(!fdlog) {
        fputs("\nError: impossible to create or open the log file "GCFILE"\n", stdout);
        exit(1);
    }
    fputs("\n\n===\nNEW DUMP\n===\n\n", fdlog);
    fflush(fdlog);
    port = htons(PORT);
    memset(gchash, 0, sizeof(gchash));
    pcap_loop(fp, 0, dispatcher_handler, NULL);

    return(0);
}



void dispatcher_handler(u_char *temp1, const struct pcap_pkthdr *header, const u_char *pkt_data) {
    struct  iphdr   *ip;
    struct  tcphdr  *tcp;
    u_char  *data,
            *output,
            tmp[64];    // for faster performances
    int     len,
            off,
            tmpnum;

    off = 14 + ether_hdrlen(ntohs(*(u_short *)(pkt_data + 12)));

    ip   = (struct iphdr *)(pkt_data + off);
    tcp  = (struct tcphdr *)((u_char *)ip + sizeof(struct iphdr));
    data = (u_char *)((u_char *)tcp + sizeof(struct tcphdr));

    if(ip->protocol != IPPROTO_TCP) return;

    if((tcp->source != port) && (tcp->dest != port)) return;

    if(!tcp->psh) return;

    len =
        header->len -           // full packet length
        off -                   // Ethernet offset
        sizeof(struct iphdr) -  // IP
        sizeof(struct tcphdr);
    if(len <= 0) return;

        /* SHOW IP PORT */
    tmpnum = sprintf(
        tmp,
        "\n%s:%u --> ",
        inet_ntoa(*(struct in_addr *)&ip->saddr),
        ntohs(tcp->source));
    sprintf(
        tmp + tmpnum,
        "%s:%u\n",
        inet_ntoa(*(struct in_addr *)&ip->daddr),
        ntohs(tcp->dest));
    fputs(tmp, stdout);
    fputs(tmp, fdlog);

        /* this is a sniffer for a TCP stream so is not very easy to */
        /* manage the data in the stream so I must use some tricks */
    if(
      (*(u_short *)data != 0xffff) &&
      (len == 32) &&
      (tcp->source == port)) {
        memcpy(gchash, data, 32);
        fprintf(stdout, "Encoding string:   %32s\n", gchash);
        fprintf(fdlog, "Encoding string:   %32s\n", gchash);
        return;
    }

    output = gamecenter_dec(gchash, data, &len);
    output[len] = 0;
    fprintf(stdout, "%s\n", output);
    fprintf(fdlog, "%s\n", output);
    fflush(fdlog);
}



u_char *gamecenter_dec(u_char *encdata, u_char *data, int *size) {
    static u_char   dec[8192];
    u_char          *p1,
                    *p2;
    int             len;

    p2 = encdata;
    p1 = data = data + 10;
    len = *size = *size - 10;
    while(len--) {
        *p1++ ^= *p2++;
        if(!*p2) p2 = encdata;
    }

    *size = do_inflate(data, *size, dec, sizeof(dec));
    return(dec);
}



int do_inflate(u_char *in, int inlen, u_char *out, int outlen) {
    z_stream    z;

    *out       = 0;
    z.zalloc   = (alloc_func)0;
    z.zfree    = (free_func)0;
    z.opaque   = (voidpf)0;
    z.next_in  = in;
    z.avail_in = 0;
    z.next_out = out;

    if(inflateInit(&z) != Z_OK) return(0);

    while((z.total_out < outlen) && (z.total_in < inlen)) {
        z.avail_in = z.avail_out = 1;
        if(inflate(&z, Z_NO_FLUSH) != Z_OK) break;
    }

    if(inflateEnd(&z) != Z_OK) return(0);
    return(z.total_out);
}



u_int resolv(char *host) {

WSADATA wsadata;
WSAStartup(MAKEWORD(1,0), &wsadata);

    struct  hostent *hp;
    u_int  host_ip;

    host_ip = inet_addr(host);
    if(host_ip == INADDR_NONE) {
        hp = gethostbyname(host);
        if(!hp) {
            printf("\nError: Unable to resolve hostname (%s)\n", host);
            exit(1);
        } else host_ip = *(u_int *)hp->h_addr;
    }
    return(host_ip);
}



void std_err(void) {
    perror("\nError");
    exit(1);
}

