/*

by Luigi Auriemma

*/

#include <stdio.h>
#include <stdlib.h>
#include <winsock.h>
#include <pcap.h>
#include "ether_hdrlen.h"



#define VER     "0.1"
#define PORT    29920



void dispatcher_handler(u_char *temp1, const struct pcap_pkthdr *header, const u_char *pkt_data);
void gamespy3dxor(u_char *string, int len);
u_int resolv(char *host);



struct iphdr {
    u_int   ihl:4;
    u_int   version:4;
    u_char  tos;
    u_short tot_len;
    u_short id;
    u_short frag_off;
    u_char  ttl;
    u_char  protocol;
    u_short check;
    u_int   saddr;
    u_int   daddr;
} *iphdr;

struct tcphdr {
    u_short source;
    u_short dest;
    u_int   seq;
    u_int   ack_seq;
    u_short res1:4;
    u_short doff:4;
    u_short fin:1;
    u_short syn:1;
    u_short rst:1;
    u_short psh:1;
    u_short ack:1;
    u_short urg:1;
    u_short res2:2;
    u_short window;
    u_short check;
    u_short urg_ptr;
} *tcphdr;

u_int  sniffhost;
u_short port;



int main(int argc, char *argv[]) {
    char        errbuf[PCAP_ERRBUF_SIZE];
    pcap_t      *fp;
    int         i,
                inum;
    pcap_if_t   *alldevs,
                *d;

    setbuf(stdout, NULL);
    setbuf(stderr, NULL);

    fprintf(stderr,
        "\n"
        "Gs auth 29920 sniff "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n"
        "Usage: %s   [host_to_monitor(default ANY)]\n"
        "\n",
        argv[0]);

    if(pcap_findalldevs(&alldevs, errbuf) < 0) {
        fprintf(stderr,"Error in pcap_findalldevs: %s\n", errbuf);
        exit(1);
    }

    fprintf(stderr, "Interfaces list:\n\n");
    for(i = 0, d = alldevs; d; d = d->next) {
        fprintf(stderr, "%d. %s", ++i, d->name);
        if(d->description) fprintf(stderr, " (%s)\n", d->description);
            else fprintf(stderr, " (No description available)\n");
    }

    if(!i) {
        fprintf(stderr, "\nNo interfaces found! Make sure WinPcap is installed.\n");
        pcap_freealldevs(alldevs);
        exit(1);
    } else if(i > 1) {
        fprintf(stderr, "\nEnter the interface number (1 - %d): ",i);
        scanf("%d", &inum);

        if(inum < 1 || inum > i) {
            fprintf(stderr, "\nInterface number out of range.\n");
            pcap_freealldevs(alldevs);
            exit(1);
        }
    } else inum = 1;

    for(inum--, d = alldevs, i = 0; i < inum; d = d->next, i++);
    fprintf(stderr, "\nAdapter used: %s\n\n", d->name);

    fp = pcap_open_live(d->name, 65535, 1, 1000, errbuf);
    if(!fp) {
        printf("\nError: %s\n", errbuf);
        pcap_freealldevs(alldevs);
        exit(1);
    }

    pcap_freealldevs(alldevs);

    if(argc > 1) {
        sniffhost = resolv(argv[1]);
    } else {
        sniffhost = 0;
    }
    port = htons(PORT);

    fprintf(stderr, "Monitoring host %s\n",
        sniffhost ? inet_ntoa(*(struct in_addr *)&sniffhost) : "ANY");

    fprintf(stderr, "Monitoring port %hu\n",
        PORT);

    pcap_loop(fp, 0, dispatcher_handler, NULL);

    return(0);
}



void dispatcher_handler(u_char *temp1, const struct pcap_pkthdr *header, const u_char *pkt_data) {
    struct  iphdr   *ip;
    struct  tcphdr  *tcp;
    u_char  *data;
    int     off,
            len;

    off = 14 + ether_hdrlen(ntohs(*(u_short *)(pkt_data + 12)));

    ip   = (struct iphdr *)(pkt_data + off);
    tcp  = (struct tcphdr *)((u_char *)ip + sizeof(struct iphdr));
    data = (u_char *)((u_char *)tcp + sizeof(struct tcphdr));

    if(ip->protocol != IPPROTO_TCP) return;

    if(!tcp->psh) return;

    if((tcp->source != port) && (tcp->dest != port)) return;

    if(sniffhost) {
        if((ip->saddr != sniffhost) && (ip->daddr != sniffhost)) return;
    }

    len =
        header->len -           // full packet length
        off -                   // Ethernet offset
        sizeof(struct iphdr) -  // IP
        sizeof(struct tcphdr);  // TCP

    gamespy3dxor(data, len);
    data[len] = 0;
    printf("\n%s:  %s\n",
        (tcp->source == port) ? "SERVER" : "CLIENT",
        data);
}



void gamespy3dxor(u_char *string, int len) {
    u_char  gamespy[] = "GameSpy3D",
            *gs;

    if(!memcmp(string + len - 7, "\\final\\", 7)) len -= 7;

    for(gs = gamespy; len; len--, gs++, string++) {
        if(!*gs) gs = gamespy;
        *string ^= *gs;
    }
}



u_int resolv(char *host) {
    struct  hostent *hp;
    u_int  host_ip;

    WSADATA wsadata;
    WSAStartup(MAKEWORD(1,0), &wsadata);

    host_ip = inet_addr(host);
    if(host_ip == INADDR_NONE) {
        hp = gethostbyname(host);
        if(!hp) {
            printf("\nError: Unable to resolv hostname (%s)\n", host);
            exit(1);
        } else host_ip = *(u_int *)(hp->h_addr);
    }
    return(host_ip);
}


