/*

by Luigi Auriemma

*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <pcap.h>
#include "ether_hdrlen.h"
#include "show_dump.h"

#ifdef WIN32
    #include <winsock.h>
#else
    #include <unistd.h>
    #include <sys/socket.h>
    #include <arpa/inet.h>
    #include <netinet/in.h>
    #include <netdb.h>
#endif



#define VER     "0.3"
#define HOST    "master.gamespy.com"

#define SEEK_AND_DESTROY(PATTERN, EXEC) \
                p = (u_char *)strstr((char *)data, PATTERN); \
                if(p) { \
                    p += sizeof(PATTERN) - 1; \
                    EXEC; \
                }



void dispatcher_handler(u_char *temp1, const struct pcap_pkthdr *header, const u_char *pkt_data);
void gamespyxor(u_char *data, int len);
u_int resolv(char *host);



struct iphdr {
    u_int   ihl:4;
    u_int   version:4;
    u_char  tos;
    u_short tot_len;
    u_short id;
    u_short frag_off;
    u_char  ttl;
    u_char  protocol;
    u_short check;
    u_int  saddr;
    u_int  daddr;
};

struct udphdr {
    u_short source;
    u_short dest;
    u_short len;
    u_short check;
};

u_int  sniffhost;
int     if_offset,
        any     = 0,
        enconly = 0,
        hex     = 0,
        verbose = 0,
        quiet   = 0;



int main(int argc, char *argv[]) {
    pcap_t      *fp;
    pcap_if_t   *alldevs,
                *d;
    int         i,
                inum;
    char        errbuf[PCAP_ERRBUF_SIZE],
                *host = HOST;

#ifdef WIN32
    WSADATA wsadata;
    WSAStartup(MAKEWORD(1,0), &wsadata);
#endif


    setbuf(stdout, NULL);
    setbuf(stderr, NULL);

    fprintf(stderr,
        "\n"
        "GsHsniff "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n"
        "Usage: %s [options]\n"
        "\n"
        "Options:\n"
        "-s HOST   host to monitor (%s)\n"
        "-a        monitor ANY host (-s ignored)\n"
        "-e        shows only the encoded packets, they starts with char ';' and are\n"
        "          XORed with the text \"gamespy\"\n"
        "-x        hexadecimal visualization of the data in the packet\n"
        "-v        verbose output, very useful to understand some commands and their\n"
        "          parameters (cd-key hash, client token, client IP and more)\n"
        "-q        shows only the data in the packet without time, IPs and ports\n"
        "\n",
        argv[0], host);

    for(i = 1; i < argc; i++) {
        switch(argv[i][1]) {
            case 's': host = argv[++i]; break;
            case 'a': any = 1; break;
            case 'e': enconly = 1; break;
            case 'x': hex = 1; break;
            case 'v': verbose = 1; break;
            case 'q': quiet = 1; break;
            default: {
                fprintf(stderr, "\nError: Wrong command-line argument (%s)\n\n", argv[i]);
                exit(1);
            }
        }
    }

    if(pcap_findalldevs(&alldevs, errbuf) < 0) {
        fprintf(stderr,"Error in pcap_findalldevs: %s\n", errbuf);
        exit(1);
    }

    fprintf(stderr, "- Interfaces list:\n\n");
    for(i = 0, d = alldevs; d; d = d->next) {
        fprintf(stderr, "%d. %s", ++i, d->name);
        if(d->description) fprintf(stderr, " (%s)", d->description);
        fputc('\n', stderr);
    }

    if(i > 1) {
        fprintf(stderr, "\n- Enter the interface number (1 - %d): ",i);
        scanf("%d", &inum);
        if(inum < 1 || inum > i) {
            fprintf(stderr, "\nError: Interface number out of range.\n");
            pcap_freealldevs(alldevs);
            exit(1);
        }
    } else if(!i) {
        fprintf(stderr, "\nError: No interfaces found. Make sure Pcap/WinPcap is installed.\n");
        pcap_freealldevs(alldevs);
        exit(1);
    } else inum = 1;

    for(inum--, d = alldevs, i = 0; i < inum; i++, d = d->next);
    fprintf(stderr, "- Adapter to use: %s\n", d->name);

    fp = pcap_open_live(d->name, 65535, 1, 1000, errbuf);
    if(!fp) {
        printf("\nError: %s\n", errbuf);
        pcap_freealldevs(alldevs);
        exit(1);
    }

    pcap_freealldevs(alldevs);

    if(!any) {
        fprintf(stderr, "- resolv hostname %s --> ", host);
        sniffhost = resolv(host);
        fprintf(stderr, "%s\n", inet_ntoa(*(struct in_addr *)&sniffhost));
    }

    if_offset = pcap_hdrlen(pcap_datalink(fp));
    if(if_offset < 0) {
        fputs("\nError: pcap interface not supported by this tool\n\n", stderr);
        exit(1);
    }

    fputs("- Ready:\n\n", stderr);
    pcap_loop(fp, 0, dispatcher_handler, NULL);

    return(0);
}



void dispatcher_handler(u_char *temp1, const struct pcap_pkthdr *header, const u_char *pkt_data) {
    time_t  datex;
    struct  tm      *tmx;
    struct  iphdr   *ip;
    struct  udphdr  *udp;
    u_int  client_ip;
    int     len,
            enc;
    u_char  *data,
            *p;

    ip   = (struct iphdr *)  (pkt_data   + if_offset +
           ether_hdrlen(ntohs(*(u_short *)(pkt_data + if_offset - 2))));
    udp  = (struct udphdr *)((void *)ip  + sizeof(struct iphdr));
    data = (u_char *)       ((void *)udp + sizeof(struct udphdr));

    if(ip->protocol != IPPROTO_UDP) return;

    if(!any && (ip->saddr != sniffhost) && (ip->daddr != sniffhost)) return;

    len = ntohs(udp->len) - sizeof(struct udphdr);

    if(*data == ';') {
        gamespyxor(data, len);
        enc = 1;
    } else {
        if(enconly) return;
        enc = 0;
    }

    if(!quiet) {
        time(&datex);
        tmx = localtime(&datex);
        printf("         %02d:%02d:%02d   %s:%hu -> ",
            tmx->tm_hour, tmx->tm_min, tmx->tm_sec,
            inet_ntoa(*(struct in_addr *)&ip->saddr),
            ntohs(udp->source));
        printf("%s:%hu   %s\n",
            inet_ntoa(*(struct in_addr *)&ip->daddr),
            ntohs(udp->dest),
            enc ? "[encoded]" : "[plain-text]");
    }

    if(hex) {
        show_dump(data, len, stdout);
        fputc('\n', stdout);
    } else {
        data[len] = 0;
        printf("%s\n\n", data);
    }

    if(!verbose) return;

    SEEK_AND_DESTROY("\\cd\\",
        printf("      -> Cd-key MD5 hash:   %.32s\n", p);
    );

    SEEK_AND_DESTROY("\\resp\\",
        printf(
            "      -> Cd-key MD5 hash:   %.32s\n"
            "      -> Client token:      0x%.8s\n"
            "      -> Verification hash: %.32s\n",
            p,
            p + 32,
            p + 32 + 8);
    );

    SEEK_AND_DESTROY("\\ip\\",
        sscanf((char *)p, "%u", &client_ip);
        printf("      -> Client IP address: %s\n",
            inet_ntoa(*(struct in_addr *)&client_ip));
    );

    fputc('\n', stdout);
}



void gamespyxor(u_char *data, int len) {
    u_char  gamespy[] = "gamespy",
            *gs;

    for(gs = gamespy; len; len--, gs++, data++) {
        if(!*gs) gs = gamespy;
        *data ^= *gs;
    }
}



u_int resolv(char *host) {
    struct  hostent *hp;
    u_int  host_ip;

    host_ip = inet_addr(host);
    if(host_ip == INADDR_NONE) {
        hp = gethostbyname(host);
        if(!hp) {
            printf("\nError: Unable to resolv hostname (%s)\n\n", host);
            exit(1);
        } else host_ip = *(u_int *)(hp->h_addr);
    }
    return(host_ip);
}


