/*
    Copyright 2005,2006,2007 Luigi Auriemma

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    http://www.gnu.org/licenses/gpl.txt
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <time.h>
#include "md5.h"

#ifdef WIN32
    #include <winsock.h>
    #include "winerr.h"

    #define RAND_SEED   (uint32_t)GetTickCount()    // 1000/s resolution
    #define close   closesocket
    #define sleep   Sleep
    #define ONESEC  1000
#else
    #include <unistd.h>
    #include <sys/socket.h>
    #include <sys/types.h>
    #include <arpa/inet.h>
    #include <netinet/in.h>
    #include <netdb.h>

    #define RAND_SEED   (uint32_t)times(0)          // 100/s resolution
    #define ONESEC  1
#endif

typedef uint8_t     u8;
typedef uint16_t    u16;
typedef uint32_t    u32;



#define VER         "0.1.2"
#define BUFFSZ      2048
#define TIMEOUT     5
#define MS          "master.gamespy.com"
#define MSPORT      29910
#define WAITSEC     15
#define NEXTRAND(x) x = ((x * 0x343FD) + 0x269EC3)
#define DBKEYMAX    64
#define WAIT        printf("\nPress RETURN to exit\n"); \
                    fgetc(stdin);



void hash2byte(u8 *in, u8 *out);
int fgetz(u8 *data, int size, FILE *fd);
u8 *do_md5(u8 *data, int len, u8 *hexout);
void create_randchall(u32 rnd, u8 *out, int len);
void gamespyxor(u8 *string, int len);
int timeout(int sock);
u32 resolv(char *host);
void std_err(void);



typedef struct {
    u8  cdkey[64];
    u8  hash[16];
} dbkey_t;

dbkey_t *dbkey  = NULL;
int     dbkey_idx;



int main(int argc, char *argv[]) {
    struct sockaddr_in  peer;
    FILE    *fdkeys = NULL,
            *fdres  = NULL;
    int     sd,
            i,
            len,
            respoff;
    u32     rnd,
            client_token,
            pid     = 0,
            ip;
    int     skey,
            valid,
            invalid;
    u8      buff[BUFFSZ + 1],
            tmp[128],
            cdkey[64],
            challenge[9],
            *p;

#ifdef WIN32
    WSADATA    wsadata;
    WSAStartup(MAKEWORD(1,0), &wsadata);
#endif

    setbuf(stdout, NULL);

    fputs("\n"
        "Online cd-key verifier for games that use the Gamespy cd-key SDK "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", stdout);

    printf("\n"
        "Usage: %s [game_PID] [cd-key/keys_file] [output_result_file]\n"
        "\n", argv[0]);

    if(argc > 1) {
        sscanf(argv[1], "%u", &pid);
    }
    while(!pid) {
        printf(
            "  - partial list of games that use this SDK:\n"
            "    http://aluigi.org/papers/gshlist.txt\n"
            "  - list of games and PIDs:\n"
            "    http://aluigi.org/papers/gspids.txt\n"
            "\n"
            "- insert the PID of the game that uses the Gamespy cd-key SDK:\n  ");
        fgetz(tmp, sizeof(tmp), stdin);
        sscanf(tmp, "%u", &pid);
    }

    if(argc > 2) {
        sprintf(cdkey, "%.*s", sizeof(cdkey), argv[2]);
    } else {
        printf(
            "- insert the cd-key to verify as used by the game or the name of the file which\n"
            "  contains the list of keys to check:\n"
            "  (in some games are used the separators '-' while in others not)\n  ");
        fgetz(cdkey, sizeof(cdkey), stdin);
    }
    fdkeys = fopen(cdkey, "rb");
    if(fdkeys) {
        dbkey = calloc(DBKEYMAX, sizeof(dbkey_t));
        if(!dbkey) std_err();
        dbkey_idx = 0;
    }

    if(argc > 3) {
        fdres = fopen(argv[3], "ab");
        if(!fdres) std_err();
        fprintf(fdres, "PID %u\n", pid);
    }

    printf("- resolv master server hostname "MS"\n");
    peer.sin_addr.s_addr = resolv(MS);
    peer.sin_port        = htons(MSPORT);
    peer.sin_family      = AF_INET;
    printf("- contact   %s : %hu\n",
        inet_ntoa(peer.sin_addr), MSPORT);

    sd = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
    if(sd < 0) std_err();

    valid   = 0;
    invalid = 0;

redo:
    if(fdkeys) {
        if(fgetz(cdkey, sizeof(cdkey), fdkeys) < 0) {
            fclose(fdkeys);
            goto quit;
        }
        if(!cdkey[0]) goto redo;
    }

    printf("\n- cdkey %s\n", cdkey);

    rnd = RAND_SEED;

    NEXTRAND(rnd);
    create_randchall(rnd, challenge, 8);
    printf("- server challenge (SERVER_TOKEN) generated: %s\n", challenge);

    NEXTRAND(rnd);
    client_token = rnd;
    printf("- client challenge (CLIENT_TOKEN) generated: 0x%.8x\n", client_token);

    NEXTRAND(rnd);
    ip   = ~rnd;

    NEXTRAND(rnd);
    skey = rnd & 0xfff;

    len = sprintf(  // do not touch len because it's used later!
        buff,
        "\\auth\\\\pid\\%u\\ch\\%s\\resp\\%n%s%.8x%s\\ip\\%u\\skey\\%d",
        pid,
        challenge,
        &respoff,                           // I don't like to waste space and instructions
        "                                ", // space for cd-key MD5
        client_token,
        "                                ", // space for third string MD5
        ip,
        skey);

    p = do_md5(cdkey, strlen(cdkey), buff + respoff);
    printf("- MD5 hash of the cd-key: %.32s\n", p - 32);

    if(dbkey) {
        hash2byte(p - 32, tmp);
        memcpy(dbkey[dbkey_idx % DBKEYMAX].hash, tmp, 16);
        strncpy(dbkey[dbkey_idx % DBKEYMAX].cdkey, cdkey, 63);
        dbkey_idx++;
    }

    i = sprintf(
        tmp,
        "%s%u%s",
        cdkey,
        client_token % 0xffff,
        challenge);
    do_md5(tmp, i, p + 8);

    printf(
        "- the query that will be sent to the master server is:\n"
        "\n"
        "  %s\n"
        "\n", buff);

    printf("- encode query\n");
    gamespyxor(buff, len);

    for(;;) {
        printf("- send query\n");
        for(i = 0; i < 2; i++) {
            sendto(sd, buff, len, 0, (struct sockaddr *)&peer, sizeof(peer));
            if(!timeout(sd)) break;
        }
        if(i < 2) {
            len = recvfrom(sd, buff, BUFFSZ, 0, NULL, NULL);
            if(len > 0) break;
        }
        printf("- error or no reply received, now I will wait %d seconds and resend the query\n", WAITSEC);
        for(i = WAITSEC; i; i--) {
            printf("  %3d\r", i);
            sleep(ONESEC);
        }
        printf("\n");
    }
    buff[len] = 0;

    if(*buff == ';') {
        printf("- decode reply\n");
        gamespyxor(buff, len);
    }

    printf(
        "- reply:\n"
        "\n"
        "  %s\n"
        "\n"
        "- explanation:\n"
        "\n", buff);

    if(!memcmp(buff, "\\uok\\", 5) || !memcmp(buff, "\\ison\\", 6)) {
        p = strstr(buff, "\\cd\\");
        if(p && dbkey) {
            hash2byte(p + 4, tmp);
            for(i = 0; i < DBKEYMAX; i++) {
                if(!memcmp(dbkey[i].hash, tmp, 16)) break;
            }
            if(i < DBKEYMAX) {
                memset(dbkey[i].hash, 0, 16);   // remove valid key
                valid++;
                printf("  the cdkey %s is VALID!!!\n", dbkey[i].cdkey);
                if(fdres) {
                    fprintf(fdres, "%s VALID!!!\n", cdkey);
                    fflush(fdres);
                }
            } else {
                printf("  the cdkey is VALID but I don't know when we have queried it\n");
            }
        } else {
            valid++;
            printf("  your cd-key is VALID!!!\n");
            if(fdres) {
                fprintf(fdres, "%s VALID!!!\n", cdkey);
                fflush(fdres);
            }
        }

        /* \\disc\\\\pid\\%u\\cd\\%s\\ip\\%u */

    } else if(!memcmp(buff, "\\unok\\", 6)) {
        invalid++;
        p = strstr(buff, "\\errmsg\\");
        if(p) {
            p += 8;
        } else {
            p = buff;
        }
        printf("  your cd-key is NOT VALID, the following is the error received:\n"
            "\n"
            "  %s\n", p);
        // if(fdres) fprintf(fdres, "%s NOT VALID\n", cdkey);

    } else {
        invalid++;
        printf("  the server has sent a strange reply, check it:\n\n  %s\n", buff);
        if(fdres) fprintf(fdres, "%s wrong reply from server\n", cdkey);
    }

    if(fdkeys) goto redo;

quit:
    close(sd);
    if(fdkeys) {
        printf("\n- %u valid cdkeys\n- %u invalid cdkeys\n", valid, invalid);
    }
    WAIT;
    return(0);
}



void hash2byte(u8 *in, u8 *out) {
    int     i,
            n;

    for(i = 0; i < 16; i++) {
        sscanf(in + (i << 1), "%02x", &n);
        out[i] = n;
    }}



int fgetz(u8 *data, int size, FILE *fd) {
    int     len;
    u8      *p,
            *s;

    if(!fgets(data, size, fd)) return(-1);
    for(p = data; *p && (*p != '\n') && (*p != '\r'); p++);
    *p = 0;
    for(s = data; *s && ((*s == ' ') || (*s == '\t')); s++);
    for(p = s; *p && (*p != ' ') && (*p != '\t'); p++);
    *p = 0;
    len = strlen(s);
    if(s != data) memmove(data, s, len + 1);
    return(len);
}



u8 *do_md5(u8 *data, int len, u8 *hexout) {
    md5_context md5t;
    int         i;
    u8          md5h[16];
    static const u8 hex[16] = "0123456789abcdef";

    md5_starts(&md5t);
    md5_update(&md5t, data, len);
    md5_finish(&md5t, md5h);

    for(i = 0; i < 16; i++) {
        *hexout++ = hex[md5h[i] >> 4];
        *hexout++ = hex[md5h[i] & 0xf];
    }
    // no final NULL
    return(hexout);
}



void create_randchall(u32 rnd, u8 *out, int len) {
    const static u8 table[] =
                    "0123456789"
                    "abcdefghijklmnopqrstuvwxyz"
                    "ABCDEFGHIJKLMNOPQRSTUVWXYZ";

    while(len--) {
        rnd = (rnd * 0x343FD) + 0x269EC3;
        *out++ = table[(rnd >> 16) % (sizeof(table) - 1)];
    }
    *out = 0;
}



void gamespyxor(u8 *data, int len) {
    u8      gamespy[] = "gamespy",
            *gs;

    for(gs = gamespy; len; len--, gs++, data++) {
        if(!*gs) gs = gamespy;
        *data ^= *gs;
    }
}



int timeout(int sock) {
    struct  timeval tout;
    fd_set  fdr;
    int     err;

    tout.tv_sec  = TIMEOUT;
    tout.tv_usec = 0;
    FD_ZERO(&fdr);
    FD_SET(sock, &fdr);
    err = select(sock + 1, &fdr, NULL, NULL, &tout);
    if(err < 0) std_err();
    if(!err) return(-1);
    return(0);
}



u32 resolv(char *host) {
    struct  hostent *hp;
    u32     host_ip;

    host_ip = inet_addr(host);
    if(host_ip == INADDR_NONE) {
        hp = gethostbyname(host);
        if(!hp) {
            printf("\nError: Unable to resolv hostname (%s)\n", host);
            WAIT;
            exit(1);
        } else host_ip = *(u32 *)hp->h_addr;
    }
    return(host_ip);
}



#ifndef WIN32
    void std_err(void) {
        perror("\nError");
        exit(1);
    }
#endif



