/*
    Copyright 2005,2006,2007,2008,2009 Luigi Auriemma

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    http://www.gnu.org/licenses/gpl-2.0.txt
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifdef WIN32
    #include <winsock.h>
    #include "winerr.h"

    #define close   closesocket
    #define sleep   Sleep
#else
    #include <unistd.h>
    #include <sys/socket.h>
    #include <sys/types.h>
    #include <arpa/inet.h>
    #include <netinet/in.h>
    #include <netdb.h>

    #define stristr     strcasestr
    #define strnicmp    strncasecmp
#endif



#define VER         "0.1.1a"
#define HOST        "gpsp.gamespy.com"
#define PORT        29901
#define DETECTION   "detection.cfg"
#define NODETECTION "Note: the file "DETECTION" is the same file available in the Services folder\n" \
                    "      of Gamespy Arcade or on http://aluigi.org/papers/"DETECTION"\n"

#define SESSKEY     0   // if this value is an existing sesskey the returned data will be complete without [hidden] fields
#define PROFILEID   0   // also random is ok
#define NAMESPACEID 1

// some available types: search (x), searchunique (uniquenick), valid (email), nicks (email,passenc), pmatch,
//                       check (nick,email), newuser, others, otherslist (numopids,opids|), profilelist

#define SEARCH      "\\search\\" \
                    "\\sesskey\\%u" \
                    "\\profileid\\%u" \
                    "\\namespaceid\\%u" \
                    "\\partnerid\\0"
#define OTHERS      "\\others\\" \
                    "\\sesskey\\%u" \
                    "\\profileid\\%u" \
                    "\\namespaceid\\%u" \
                    "\\partnerid\\0"
#define PMATCH      "\\pmatch\\" \
                    "\\sesskey\\%u" \
                    "\\profileid\\%u" \
                    "\\productid\\%u" \
                    "\\partnerid\\0"
#define END         "\\gamename\\%s" \
                    "\\final\\"
#define CMP(X)      !strcmp(p, X)



void show_list(char *pattern);
u_int resolv(char *host);
void std_err(void);



int main(int argc, char *argv[]) {
    struct  sockaddr_in peer;
    int     sd,
            i,
            len,
            t,
            total,
            buffsz;
    u_char  *buff,
            *gamename   = "",
            *option,
            *fmt,
            *p,
            *l;

#ifdef WIN32
    WSADATA    wsadata;
    WSAStartup(MAKEWORD(1,0), &wsadata);
#endif

    setbuf(stdout, NULL);

    fputs("\n"
        "GSPlayers "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", stdout);

    if(argc < 2) {
        printf("\n"
            "Usage: %s <command>\n"
            "\n"
            "Commands:\n"
            "-s OPT   search an user, OPT is one or more of the following options:\n"
            "         n NICKNAME    all the users who have this nickname (could not work)\n"
            "         e EMAIL       all the users who have this email\n"
            "         u NICKNAME    the user that has this unique nickname\n"
            "         f FIRSTNAME   all the users who have this first name\n"
            "         l LASTNAME    all the users who have this last name\n"
            "         i ICQUIN      all the users who have this ICQ UIN\n"
            "         ? ?           anything else is a custom query like: -s myquery myvalue\n"
            "         usually search patterns are case insensitive (mynick = MyNicK)\n"
            "-o PID   search what people are in the buddy list of the user identified by his\n"
            "         profile ID (PID). The profile ID is visible when you search the user\n"
            "-p PROD  search the current online users that have the game identified by PROD\n"
            "         (product ID) installed on their computers, not so useful\n"
            "-l [P]   option needed to see the full list of available games and their\n"
            "         product IDs. If you use only -l you will see all the list while if you\n"
            "         add a second parameter (P) it will be used to search a specific string\n"
            "-e       some usage examples in case the commands are not clear\n"
            "\n"
            "Note: this tool finds all the people that have a Gamespy account so the users\n"
            "      on Gamespy Arcade and any other game that requires this type of account\n"
            "      like Ground Control 2 for example\n"
            NODETECTION
            "\n", argv[0]);
        exit(1);
    }

    option = argv[1];
    if(option[0] != '-') {
        printf("\nError: you must specify an option (%s)\n", option);
        exit(1);
    }

    buffsz = 8192;
    buff = malloc(buffsz + 1);
    if(!buff) std_err();

    if(option[1] == 's') {
        if(argc < 4) {
            printf("\nError: recheck your command line because you have missed some arguments\n\n");
            exit(1);
        }

        len = sprintf(
            buff,
            SEARCH,
            SESSKEY,
            PROFILEID,
            NAMESPACEID);

        for(i = 2; i < argc; i++) {
            switch(argv[i][0]) {
                case 'n': fmt = "nick";         break;
                case 'e': fmt = "email";        break;
                case 'u': fmt = "uniquenick";   break;
                case 'f': fmt = "firstname";    break;
                case 'l': fmt = "lastname";     break;
                case 'i': fmt = "icquin";       break;
                default:  fmt = NULL;           break;
            }
            if(!fmt || (strlen(argv[i]) != 1)) {
                fmt = argv[i];
            }
            if(fmt[0] == '-') {
                printf("\nError: only one option/command is supported\n");
                exit(1);
            }
            len += sprintf(buff + len, "\\%s\\%s", fmt, argv[++i]);
        }

    } else if(option[1] == 'o') {
        if(argc < 3) {
            printf("\nError: recheck your command line because you have missed some arguments\n\n");
            exit(1);
        }

        len = sprintf(
            buff,
            OTHERS,
            SESSKEY,
            atoi(argv[2]),
            NAMESPACEID);

    } else if(option[1] == 'p') {
        if(argc < 3) {
            printf("\nError: recheck your command line because you have missed some arguments\n\n");
            exit(1);
        }

        len = sprintf(
            buff,
            PMATCH,
            SESSKEY,
            PROFILEID,
            atoi(argv[2]));

    } else if(option[1] == 'l') {
        p = "";
        if(argc > 2) p = argv[2];
        show_list(p);
        return(0);

    } else if(option[1] == 'e') {
        printf("\n"
            "Usage examples:\n"
            "-s n mynick                            search all the users with nick mynick\n"
            "-s e mail@mail.ext                     what accounts are own by mail@mail.ext?\n"
            "-s n mynick e mail@mail.ext f myname   to restrict the search\n"
            "-s u mynick                            finds the user with mynick registered\n"
            "-s i 12345678                          who has the ICQ UIN 12345678?\n"
            "-s myquery myvalue                     custom query\n"
            "\n"
            "-o 123456                              123456 is the profile ID of the user\n"
            "                                       you find it with the search command\n"
            "\n"
            "-p 10298                               10298 is the productID of Battlefield 1942\n"
            "                                       we will know what Gamespy Arcade users\n"
            "                                       are playing it in this moment\n"
            "\n"
            "-l                                     shows the full list of available games\n"
            "-l quake                               shows the list of games that contain the\n"
            "                                       pattern quake in them\n"
            "-l 10298                               as above\n"
            "\n"
            "If you have doubts, contact me\n"
            "\n");
        return(0);

    } else {
        printf("\nError: wrong command specified (%s)\n\n", option);
        exit(1);
    }

    len += sprintf(buff + len, END, gamename);

    printf("\n  QUERY to send:\n  %.*s\n\n", len, buff);

    printf("- resolv host %s:   ", HOST);
    peer.sin_addr.s_addr = resolv(HOST);
    peer.sin_port        = htons(PORT);
    peer.sin_family      = AF_INET;

    printf("%s : %hu\n", inet_ntoa(peer.sin_addr), ntohs(peer.sin_port));

    sd = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
    if(sd < 0) std_err();

    printf("- connecting... ");
    if(connect(sd, (struct sockaddr *)&peer, sizeof(peer))
      < 0) std_err();
    printf("ok\n");

    printf("- send request\n");
    if(send(sd, buff, len, 0)
      < 0) std_err();

    printf("- receive informations:\n");

    for(len = 0;; len += t) {
        if(len >= buffsz) {
            buffsz += 4096;
            buff = realloc(buff, buffsz + 1);
            if(!buff) std_err();
        }
        t = recv(sd, buff + len, buffsz - len, 0);
        if(t <= 0) break;
    }
    buff[len] = 0;

    total = 0;
    t = 1;
    for(p = buff; (l = strchr(p, '\\')); p = l + 1, t++) {
        *l = 0;
        if(t & 1) {
            printf("%s\n", p);
        } else {
            if(CMP("bsr") || CMP("o") || CMP("psr")) {
                total++;
                p = "profile_id";
                fputc('\n', stdout);
            } else if(CMP("bsrdone") || CMP("odone") || CMP("psrdone") || CMP("final")) {
                break;
            }
            printf("%20s: ", p);
        }
    }

    close(sd);
    printf("\n- found %u players\n\n", total);
    return(0);
}



void show_list(char *pattern) {
    FILE    *fd;
    int     ready;
    char    buff[1000],
            out[81]     = "",
            *p;

    printf("- open file "DETECTION"\n");
    fd = fopen(DETECTION, "rb");
    if(!fd) {
        printf(NODETECTION);
        std_err();
    }

    printf("\n"
        "GAMENAME        DESCRIPTION                                           PRODUCTID\n"
        "-------------------------------------------------------------------------------\n");

    *out = 0;
    ready = 3;
    while(fgets(buff, sizeof(buff), fd)) {
        for(p = buff; *p && (*p != '\n') && (*p != '\r'); p++);
        *p = 0;

        if(*buff == '[') {
            if(!ready) {
                out[80] = 0;
                if(!pattern[0] || stristr(out, pattern)) printf("%s\n", out);
                memset(out, 0, sizeof(out));
            }
            ready = 3;

            *(p - 1) = 0;
            sprintf(out, "%-15.15s", buff + 1);
            out[15] = ' ';
            ready--;

        } else if(!strnicmp(buff, "productid=", 10)) {
            sprintf(out + 16 + 54, "%.10s", buff + 10);
            ready--;

        } else if(!strnicmp(buff, "commonname=", 11)) {
            sprintf(out + 16, "%-53.53s", buff + 11);
            out[16 + 53] = ' ';
            ready--;
        }
    }

    fclose(fd);
    fputc('\n', stdout);
}



u_int resolv(char *host) {
    struct  hostent *hp;
    u_int   host_ip;

    host_ip = inet_addr(host);
    if(host_ip == INADDR_NONE) {
        hp = gethostbyname(host);
        if(!hp) {
            printf("\nError: Unable to resolv hostname (%s)\n", host);
            exit(1);
        } else host_ip = *(u_int *)hp->h_addr;
    }
    return(host_ip);
}



#ifndef WIN32
    void std_err(void) {
        perror("\nError");
        exit(1);
    }
#endif


