/*
    Copyright 2007 Luigi Auriemma

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    http://www.gnu.org/licenses/gpl.txt
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <sys/stat.h>

#ifdef WIN32
#else
    #define strnicmp    strncasecmp
    #define stristr     strcasestr
#endif

typedef uint8_t     u8;
typedef uint16_t    u16;
typedef uint32_t    u32;



#define VER         "0.1.1"
#define STARTDATA   "---START-DATA---"
#define STARTDATASZ (sizeof(STARTDATA) - 1)



u8 *gamezone_cr(u8 *data);
u8 *gamezone_cb(u8 *data);
u8 *gamezone_cx(u8 *data);
u8 *gamezone_dr(u8 *data);
u8 *gamezone_cm(u8 *data);
u8 *base64_decode(u8 *data, int *size, u8 *key);
void base64_decode_init(u8 *base, u8 *key);
u8 *base64_decode_next(u8 **data, u8 *limit, int *ret_c);
u8 *find_gamezone_url(u8 *fname);
u8 *fd_read(u8 *name, int *fdlen);
u8 *delimit(u8 *data);



int main(int argc, char *argv[]) {
    int     i;
    u8      *data;

    fputs("\n"
        "GameZone.com URL decoder "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", stdout);

    if(argc < 2) {
        printf("\n"
            "Usage: %s <URL/file.EXE>\n"
            "\n"
            "This tool decodes any URL used on GameZone.com (for example the URL for\n"
            "downloading the demos and so on) and can also try to find and decode the URL\n"
            "located in the executables (GameZone Downloader)\n"
            "An example of URL is DR:687576733A303166646F3076696E76686C2F706874\n"
            "Other info: http://downloads.gamezone.com/support/FAQ_GZDL.htm\n"
            "\n", argv[0]);
        exit(1);
    }

    data = find_gamezone_url(argv[1]);
    if(!data) data = strdup(argv[1]);   // probably is better to dup argv[1]

    printf("- encoded URL:\n  %s\n", data);

    if(!strnicmp(data, "URL=", 4)) data += 4;
    if(!strnicmp(data, "http://", 7)) data += 7;

    for(i = 2; data[i]; i++) {
        if(data[i] == ':') {
            data += i - 2;
            break;
        }
    }

    if(data[2] != ':') {
        printf("- no method in the URL (like DR:, CR: and so on), I try with \"DR:\"\n");
        data = gamezone_dr(data);

    } else if(!strnicmp(data, "CR:", 3)) {
        printf("- encoding type: CR\n");
        data = gamezone_cr(data + 3);

    } else if(!strnicmp(data, "CB:", 3)) {
        printf("- encoding type: CB\n");
        data = gamezone_cb(data + 3);

    } else if(!strnicmp(data, "CX:", 3)) {
        printf("- encoding type: CX\n");
        data = gamezone_cx(data + 3);

    } else if(!strnicmp(data, "DR:", 3)) {
        printf("- encoding type: DR\n");
        data = gamezone_dr(data + 3);

    } else if(!strnicmp(data, "CM:", 3)) {
        printf("- encoding type: CM\n");
        data = gamezone_cm(data + 3);

    } else {
        printf("\nError: unknown URL\n");
        exit(1);
    }

    printf("- decoded URL:\n  %s\n", data);
    return(0);
}



u8 *gamezone_cr(u8 *data) {
    u32     rnd;
    int     i,
            j,
            t,
            len,
            plen;
    u8      key[64],
            *buff,
            *tmp,
            *p,
            *o;
    static const u8 defkey[64] = 
            "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";

    len = ((strlen(data) >> 2) * 3) + 1;
    buff = malloc(len);
    if(!buff) return(NULL);
    o = buff;

    for(p = data; *p; p += plen) {
        memcpy(key, defkey, 64);
        rnd = p - data;
        for(i = 0; i < 64; i++) {
            rnd = (rnd * 0x343fd) + 0x269ec3;
            j = (rnd >> 16) & 63;
            t      = key[i];
            key[i] = key[j];
            key[j] = t;
        }

        len = strlen(p);
        if(len > 4) len = 4;
        plen = len;

        tmp = base64_decode(p, &len, key);
        memcpy(o, tmp, len);
        free(tmp);
        o += len;
    }

    return(buff);
}



u8 *gamezone_cb(u8 *data) {
    data = base64_decode(data, NULL, NULL);

    return(data);
}



u8 *gamezone_cx(u8 *data) {
    int     i,
            c;

    for(i = 0; ; i++) {
        if(sscanf(data + (i << 1), "%02X", &c) != 1) break;
        data[i] = c;
    }
    data[i] = 0;

    return(data);
}



u8 *gamezone_dr(u8 *data) {
    int     i,
            c;

    for(i = 0; ; i++) {
        if(sscanf(data + (i << 1), "%02X", &c) != 1) break;
        data[i] = c - (i & 3);
    }
    data[i] = 0;

    return(data);
}



u8 *gamezone_cm(u8 *data) {
    int     i;

    for(i = 0; data[i]; i++) {
        data[i]--;
    }

    return(data);
}



u8 *base64_decode(u8 *data, int *size, u8 *key) {
    int     len,
            xlen,
            a,
            b,
            c;
    u8      base[128],
            *buff,
            *limit,
            *p;
    static const u8 defkey[64] =
            "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";

    if(!key) key = (u8 *)defkey;
    base64_decode_init(base, key);

    if(!size || (*size < 0)) {      // use size -1 for auto size!
        len = strlen(data);
    } else {
        len = *size;
    }
    xlen = ((len >> 2) * 3) + 1;    // NULL included in output for text
    buff = malloc(xlen);
    if(!buff) return(NULL);

    p = buff;
    limit = data + len;
    for(;;) {
        if(!base64_decode_next(&data, limit, &c)) break;
        a = base[c];

        if(!base64_decode_next(&data, limit, &c)) break;
        b = base[c];
        *p++ = (a << 2)        | (b >> 4);

        if(!base64_decode_next(&data, limit, &c)) break;
        a = base[c];
        *p++ = ((b & 15) << 4) | (a >> 2);

        if(!base64_decode_next(&data, limit, &c)) break;
        *p++ = ((a & 3) << 6)  | base[c];
    }
    *p = 0;

    len = p - buff;
    if(size) *size = len;
    if((len + 1) != xlen) {             // save memory
        buff = realloc(buff, len + 1);  // NULL included!
        if(!buff) return(NULL);
    }
    return(buff);
}



void base64_decode_init(u8 *base, u8 *key) {
    int     i;

    memset(base, 128, 0);
    for(i = 0; i < 64; i++) {
        base[key[i] & 127] = i;
    }
}



u8 *base64_decode_next(u8 **data, u8 *limit, int *ret_c) {
    int     c;
    u8      *p;

    c = *ret_c;
    p = *data;

    do {            /* auto chars cleaning */
        c = *p;
        if((p == limit) || (c == '=')) {
            c = 0;
            break;
        }
        p++;
    } while(c && ((c <= ' ') || (c > 0x7f)));
    if(!c) return(NULL);

    *ret_c = c;
    *data  = p;
    return(p);
}



u8 *find_gamezone_url(u8 *fname) {
    int     size;
    u8      *buff,
            *p,
            *limit;

    buff = fd_read(fname, &size);
    if(!buff) return(NULL);

    limit = buff + size - STARTDATASZ;

    for(p = buff; p < limit; p++) {
        if(!memcmp(p, STARTDATA, STARTDATASZ)) break;
    }
    if(p == limit) goto quit;

    p = (u8 *)stristr(p, "\nURL=");
    if(!p) goto quit;
    p++;

    p = strdup(delimit(p));
    free(buff);
    return(p);

quit:
    printf("\n"
        "Error: no URL found in the input executable\n"
        "       try to unpack it with UPX and recheck it again\n");
    free(buff);
    exit(1);;
}



u8 *fd_read(u8 *name, int *fdlen) {
    struct stat xstat;
    FILE    *fd;
    u8      *buff;

    printf("- check if input is a file\n");
    fd = fopen(name, "rb");
    if(!fd) return(NULL);
    fstat(fileno(fd), &xstat);
    buff = malloc(xstat.st_size);
    fread(buff, xstat.st_size, 1, fd);
    fclose(fd);
    *fdlen = xstat.st_size;
    return(buff);
}



u8 *delimit(u8 *data) {
    u8      *p;

    for(p = data; *p && (*p != '\n') && (*p != '\r'); p++);
    *p = 0;
    return(data);
}


