/*

by Luigi Auriemma


Windows Winpcap version

This software is covered by GNU/GPL
*/

#include <stdio.h>
#include <stdlib.h>
#include <pcap.h>
#include <string.h>
#include <winsock.h>
#include "ether_hdrlen.h"
#include "show_dump.h"
#include "hldec.h"
#include "winerr.h"



#define VER     "0.1.3"




void dispatcher_handler(u_char *temp1, const struct pcap_pkthdr *header, const u_char *pkt_data);
u_int resolv(char *host);
void std_err(void);





struct iphdr {
        u_int   ihl:4;
        u_int   version:4;
        u_char  tos;
        u_short tot_len;
        u_short id;
        u_short frag_off;
        u_char  ttl;
        u_char  protocol;
        u_short check;
        u_int   saddr;
        u_int   daddr;
} *iphdr;


struct udphdr {
        u_short source;
        u_short dest;
        u_short len;
        u_short check;
} *udphdr;




unsigned char   dropff = 1;
unsigned int   sniffhost = 0;
unsigned short  port = 0;




int main(int argc, char *argv[]) {
    char        errbuf[PCAP_ERRBUF_SIZE];
    pcap_t      *fp;
    int         i,
                inum;
    pcap_if_t   *alldevs,
                *d;


    setbuf(stdout, NULL);

    fputs("\n"
        "Half-Life packets decoder sniffer "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n"
        "Command-line options:\n"
        "---------------------\n"
        "-p PORT    gets only the packets with this source/dest port (useful)\n"
        "-i HOST    sniff only from a specific host\n"
        "-f         DOESN'T drop the packets that start with 0xffffffff\n"
        "\n", stderr);

    for(i = 1; i < argc; i++) {
        switch(argv[i][1]) {
            case 'p': { port = atoi(argv[++i]); port = htons(port); } break;
            case 'i': sniffhost = resolv(argv[++i]); break;
            case 'f': dropff = 0; break;
            default: {
                printf("\nError: Wrong argument (%s)\n", argv[i]);
                exit(1);
                } break;
        }
    }


        /* Retrieve the device list */
    if(pcap_findalldevs(&alldevs, errbuf) < 0) {
        fprintf(stderr,"Error in pcap_findalldevs: %s\n", errbuf);
        exit(1);
    }

        /* Print the list */
    i = 0;
    fprintf(stderr, "\nInterfaces list:\n\n");
    for(d = alldevs; d; d = d->next) {
        fprintf(stderr, "%d. %s", ++i, d->name);
        if(d->description) fprintf(stderr, " (%s)\n", d->description);
            else fprintf(stderr, " (No description available)\n");
    }

    if(!i) {
        fprintf(stderr, "\nNo interfaces found! Make sure WinPcap is installed.\n");
        pcap_freealldevs(alldevs);
        exit(1);
    } else if(i > 1) {
        fprintf(stderr, "\n\nEnter the interface number (1 - %d): ",i);
        scanf("%d", &inum);

        if(inum < 1 || inum > i) {
            fprintf(stderr, "\nInterface number out of range.\n");
            pcap_freealldevs(alldevs);
            exit(1);
        }
    } else inum = 1;

        /* Jump to the selected adapter */
    inum--;
    for(d = alldevs, i = 0; i < inum; d = d->next, i++);

    fprintf(stderr, "\n\nAdapter used: %s\n\n", d->name);

    fp = pcap_open_live(d->name, 65535, 1, 1000, errbuf);
    if(!fp) {
        printf("\nError: %s\n", errbuf);
        pcap_freealldevs(alldevs);
        exit(1);
    }

        /* Check the link layer. We support only Ethernet for simplicity. */
    if(pcap_datalink(fp) != DLT_EN10MB) {
        fprintf(stderr,"\nThis program works only on Ethernet networks.\n");
        pcap_freealldevs(alldevs);
        exit(1);
    }

    pcap_freealldevs(alldevs);

    pcap_loop(fp, 0, dispatcher_handler, NULL);

    return(0);
}





void dispatcher_handler(u_char *temp1, const struct pcap_pkthdr *header, const u_char *pkt_data) {
    struct  iphdr   *ip;
    struct  udphdr  *udp;
    u_char  *data;
    u_short len;
    int     off;

    off = 14 + ether_hdrlen(ntohs(*(u_short *)(pkt_data + 12)));

    ip   = (struct iphdr *)(pkt_data + off);
    udp  = (struct udphdr *)((u_char *)ip + sizeof(struct iphdr));
    data = (u_char *)((u_char *)udp + sizeof(struct udphdr));

    if(sniffhost) {
        if((ip->saddr != sniffhost) && (ip->daddr != sniffhost)) return;
    }

    if(ip->protocol != IPPROTO_UDP) return;

    if(dropff) {
        if(!memcmp(data + 1, "\xff\xff\xff", 3)) return;
    }
    if(port) {
        if((udp->source != port) && (udp->dest != port)) return;
    }

    len = htons(udp->len) - sizeof(struct udphdr);
    hldec(data, len);

    printf("\n%s:%u --> ",
        inet_ntoa(*(struct in_addr *)&ip->saddr),
        htons(udp->source));
    printf("%s:%u\n",
        inet_ntoa(*(struct in_addr *)&ip->daddr),
        htons(udp->dest));

    show_dump(data, len, stdout);
}





u_int resolv(char *host) {
WSADATA wsadata;
WSAStartup(MAKEWORD(1,0), &wsadata);

    struct  hostent *hp;
    u_int  host_ip;

    host_ip = inet_addr(host);
    if(host_ip == INADDR_NONE) {
        hp = gethostbyname(host);
        if(!hp) {
            printf("\nError: Unable to resolve hostname (%s)\n", host);
            exit(1);
        } else host_ip = *(u_int *)(hp->h_addr);
    }

    return(host_ip);
}



