/*
    Copyright 2009,2010 Luigi Auriemma

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    http://www.gnu.org/licenses/gpl-2.0.txt
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <ctype.h>
#include <time.h>
#include <bzlib.h>
#include "show_dump.h"

#ifdef WIN32
    #include <winsock.h>
    #include "winerr.h"

    #define close   closesocket
    #define sleep   Sleep
    #define ONESEC  1000
#else   // gcc -o hlswlist hlswlist.c -lbz2
    #include <unistd.h>
    #include <sys/socket.h>
    #include <sys/types.h>
    #include <arpa/inet.h>
    #include <netinet/in.h>
    #include <netdb.h>

    #define ONESEC  1
    #define stristr strcasestr
    #define stricmp strcasecmp
#endif

typedef uint8_t     u8;
typedef uint16_t    u16;
typedef uint32_t    u32;



#define VER         "0.1.1c"

#define FILEOUT         "hlswlist-out.gsl"
#define DOITLATER_LIST  1
#define DOITLATER_UPD   2
#define DOITLATER_CFG   3
#define DOITLATER_CFG0  4
#define ARGHELP         !argv[i] || !strcmp(argv[i], "?") || !stricmp(argv[i], "help")
#define ARGHELP1        !argv[i] || !argv[i + 1] || !strcmp(argv[i + 1], "?") || !stricmp(argv[i + 1], "help")
#define CHECKARG        if(!argv[i]) { \
                            fprintf(stderr, "\n" \
                                "Error: you missed to pass the parameter to the option\n" \
                                "\n"); \
                            exit(1); \
                        }
#define FREEX(X)    freex((void *)&X)
void freex(void **buff) {
    if(!buff || !*buff) return;
    free(*buff);
    *buff = NULL;
}



typedef struct {
    //u8      *game;
    u8      *mod;
    u8      *map;
    u8      *country;
    u8      *version;
    int     players_min;
    int     players_max;
    int     slots_min;
    int     slots_max;
    int     freeslots_min;
    int     freeslots_max;
    int     password;
    int     tv_proxy;
} list_opt_t;



void hlsw_filter(list_opt_t *list_opt, u8 *filter);
void show_countries(void);
void show_filter_help(void);
void show_list(u8 *name);
u8 *handle_server_info(u8 *p, u8 *data, int datalen, int show_server_info, u32 *ret_ip, u16 *ret_port);
void show_help(void);
void show_output_help(void);
u8 *ip2str(u32 ip);
int bunzip2(u8 *in, int size, u8 *out, int maxsz);
int hlsw_send(int sd, int e1, u8 *data, int len);
u8 *hlsw_recv(int sd, int *ret_len);
int hlsw_enc(int e1, u8 *data, int len);
int hlsw_dec(u8 *data, int len);
int recv_tcp(int sd, u8 *data, int datalen);
void myalloc(u8 **data, int wantsize, int *currsize);
int putsx(u8 *data, u8 *str);
int putmm(u8 *data, u8 *str, int len);
int putcc(u8 *data, int chr, int len);
int putxx(u8 *data, u32 num, int bytes);
u32 getxx(u8 *data, int bytes);
int timeout(int sock, int secs);
u32 resolv(char *host);
void std_err(void);



FILE    *fdout  = NULL;
int     quiet   = 0;
u16     msport  = 12451;
u8      *mshost = "multimaster.hlsw.org";

static int enctypex_data_cleaner_level = 2; // 0 = do nothing
                                            // 1 = colors
                                            // 2 = colors + strange chars
                                            // 3 = colors + strange chars + sql



int main(int argc, char *argv[]) {
    list_opt_t  list_opt;
    struct  sockaddr_in peer;
    u32     ip;
    int     i,
            len,
            datalen,
            datasz           = 0,
            sd               = 0,
            execlen          = 0,
            servers          = 0,
            show_server_info = 0,
            iwannaloop       = 0,
            doitlater_type   = 0;
    u16     port;
    u8      tmp[256],
            outtype          = 0,
            *execptr         = NULL,
            *tmpexec         = NULL,
            *execstring      = NULL,
            *execstring_ip   = NULL,
            *execstring_port = NULL,
            *doitlater_str   = NULL,
            *multigamename   = NULL,
            *multigamenamep  = NULL,
            *gamestr         = NULL,
            *filter          = NULL,
            *fname           = NULL,
            *buff            = NULL,
            *data            = NULL,
            *datap           = NULL,
            *ipc,
            *p;

#ifdef WIN32
    WSADATA    wsadata;
    WSAStartup(MAKEWORD(1,0), &wsadata);
#endif

    setbuf(stdout, NULL);
    setbuf(stderr, NULL);

    fputs("\n"
        "HLSWlist "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", stderr);

    fdout = stdout;

    if(argc < 2) {
        show_help();
        exit(1);
    }

    for(i = 1; i < argc; i++) {
        if(stristr(argv[i], "--help")) {
            show_help();
            return(0);
        }
        if(((argv[i][0] != '-') && (argv[i][0] != '/')) || (strlen(argv[i]) != 2)) {
            fprintf(stderr, "\n"
                "Error: recheck your options (%s is not valid)\n"
                "\n", argv[i]);
            exit(1);
        }

        switch(argv[i][1]) {
            case '-':
            case '/':
            case '?':
            case 'h': {
                show_help();
                return(0);
                } break;
            case 'n':
            case 'N': {
                i++;
                if(!argv[i]) {
                    fprintf(stderr, "\n"
                        "Error: you must select a gamename\n"
                        "Use -l for the full list or -s for searching a specific game\n"
                        "\n");
                    exit(1);
                }
                gamestr = argv[i];
                } break;
            case 'l': {
                doitlater_type = DOITLATER_LIST;
                } break;
            case 's': {
                i++;
                if(!argv[i]) {
                    fprintf(stderr, "\n"
                        "Error: you must specify a text pattern to search in the game database\n"
                        "\n");
                    exit(1);
                }
                doitlater_type = DOITLATER_LIST;
                doitlater_str  = argv[i];
                } break;
            case 'f': {
                i++;
                if(ARGHELP) {
                    show_filter_help();
                    return(0);
                }
                filter = argv[i];
                } break;
            case 'r': {
                i++;
                CHECKARG

                execstring = argv[i];
                execlen = strlen(execstring) + 23;

                tmpexec = malloc(execlen);
                if(!tmpexec) std_err();

                execstring_ip = strstr(execstring, "#IP");
                execstring_port = strstr(execstring, "#PORT");
                if(execstring_ip) *execstring_ip = 0;
                if(execstring_port) *execstring_port = 0;

                execlen = strlen(execstring);
                memcpy(tmpexec, execstring, execlen);
                } break;
            case 'o': {
                i++;
                if(ARGHELP) {
                    show_output_help();
                    return(0);
                }
                outtype = atoi(argv[i]);
                if(outtype > 6) outtype = 0;
                if(!outtype) {
                    fdout = fopen(argv[i], "wb");
                    if(!fdout) std_err();
                }
                } break;
            case 'q': {
                quiet = 1;
                } break;
            case 'x': {
                i++;
                if(!argv[i]) {
                    fprintf(stderr, "\n"
                        "Error: you must specify the master server and optionally its port\n"
                        "\n");
                    exit(1);
                }
                mshost = strchr(argv[i], ':');
                if(mshost) {
                    msport = atoi(mshost + 1);
                    *mshost = 0;
                }
                mshost = argv[i];
                //mymshost = mshost;
                } break;
            case 'L': {
                i++;
                if(!argv[i]) {
                    fprintf(stderr, "\n"
                        "Error: you must specify the amount of seconds for the loop\n"
                        "\n");
                    exit(1);
                }
                iwannaloop = atoi(argv[i]);
                } break;
            case 'c': {
                show_countries();
                return(0);
                } break;
            case 'u':
            case 'U': {
                doitlater_type = DOITLATER_UPD;
                } break;
            case 'X': {
                i++;
                if(!argv[i]) {
                    fprintf(stderr, "\n"
                        "Error: you must specify the informations you want to return from enctypex\n"
                        "       for example: -t -1 -X \\hostname\\gamever\\gametype\\gamemode\\numplayers\n"
                        "\n");
                    exit(1);
                }
                show_server_info = 1;
                } break;
            case 'C': {
                enctypex_data_cleaner_level = 0;
                } break;
            default: {
                fprintf(stderr, "\n"
                    "Error: wrong argument (%s)\n"
                    "\n", argv[i]);
                exit(1);
                } break;
        }
    }

    srand(time(NULL));

    if(doitlater_type) {
        switch(doitlater_type) {
            case DOITLATER_LIST:    show_list(doitlater_str);   break;
            case DOITLATER_UPD:     gamestr = "";               break;
            default: break;
        }
        if(doitlater_type != DOITLATER_UPD) return(0);  // temporary (debug only)
    }

    if(!gamestr) {
        fprintf(stderr, "\n"
            "Error: The game is not available or has not been specified.\n"
            "       Use -n to specify a gamename\n"
            "\n");
        exit(1);
    }

    hlsw_filter(&list_opt, filter);

    peer.sin_addr.s_addr = resolv(mshost);
    peer.sin_port        = htons(msport);
    peer.sin_family      = AF_INET;

    fprintf(stderr, "- target   %s : %hu\n", inet_ntoa(peer.sin_addr), ntohs(peer.sin_port));

    multigamename = gamestr;

get_list:
    multigamenamep = strchr(gamestr, ',');
    if(multigamenamep) *multigamenamep = 0;
    if(!quiet) fprintf(stderr, "Gamename:    %s\n", gamestr);

    if(!sd) {
        sd = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
        if(sd < 0) std_err();
        if(connect(sd, (struct sockaddr *)&peer, sizeof(peer))
          < 0) std_err();
    }

    switch(outtype) {
        case 0: break;
        case 1:
        case 3: {
            if(!fname) fname = malloc(strlen(gamestr) + 10);
            sprintf(fname, "%s.gsl", gamestr);
            } break;
        case 2:
        case 4: {
            if(!fname) fname = malloc(strlen(FILEOUT) + 10);
            sprintf(fname, "%s",     FILEOUT);
            } break;
        case 5:
        case 6: {
            fdout = stdout;
            } break;
        default: break;
    }

    if(fname) {
        if(!quiet) fprintf(stderr, "- output file: %s\n", fname);
        fdout = fopen(fname, "wb");
        if(!fdout) std_err();
        FREEX(fname);
    }

    p = tmp;
    if(doitlater_type == DOITLATER_UPD) {
        p += putxx(p, 0x80c1, 2);
        p += putxx(p, 2,    1);         // 1 for the list, 2 for the list of supported games
    } else {
        p += putxx(p, 0x8061, 2);
        p += putxx(p, 1,    1);         // 1 for the list, 2 for the list of supported games
    }
    p += putxx(p, 2,    1);
    p += putxx(p, 4,    4);             // any number is ok, looks like a tracker
    p += putsx(p, gamestr);
    p += putsx(p, list_opt.mod);
    p += putsx(p, list_opt.map);
    p += putsx(p, list_opt.country);
    p += putxx(p, list_opt.players_min,   4);
    p += putxx(p, list_opt.players_max,   4);
    p += putxx(p, list_opt.slots_min,     4);
    p += putxx(p, list_opt.slots_max,     4);
    p += putxx(p, list_opt.freeslots_min, 4);
    p += putxx(p, list_opt.freeslots_max, 4);
    p += putxx(p, list_opt.password,      1);
    p += putxx(p, list_opt.tv_proxy,      1);
    p += putsx(p, list_opt.version);
    p += putxx(p, 0,    2);             // end of the packet (protocol)
    if(hlsw_send(sd, 2, tmp, p - tmp) < 0) {
        fprintf(stderr, "\nError: connection closed during the sending of the data\n");
        exit(1);
    }
    buff = hlsw_recv(sd, &len);
    if(!buff || (len < 0)) {
        fprintf(stderr, "\nError: connection lost, no reply received\n");
        exit(1);
    }

    if(!quiet) {
        fprintf(stderr,
            "Receiving:   %d bytes\n"
            "-----------------------\n",
            len);
    }

    if(outtype == 6) {
        fprintf(stderr, "\n\n");
        show_dump(buff, len, fdout);
        fputc('\n', stderr);
        //goto hlswlist_exit;
        outtype = 0;
    }

    if(len < 12) {
        //fprintf(stderr, "- no servers or connection closed or too short reply\n");
        //exit(1);
        datalen = 0;
    } else if((buff[1] & 0x40) || !memcmp(buff + 12, "BZh", 4)) {
        datalen = getxx(buff + 8, 4);
        //data = realloc(data, datalen);
        myalloc(&data, datalen, &datasz);
        if(!data) std_err();
        datalen = bunzip2(buff + 12, len - 12, data, datalen);
    } else {
        datalen = len - 8;
        //data = realloc(data, datalen);
        myalloc(&data, datalen, &datasz);
        if(!data) std_err();
        memcpy(data, buff + 8, len - 8);
    }

    if(doitlater_type == DOITLATER_UPD) {
        fprintf(stderr, "\n- update not yet implemented (only for debug), hex dump follows:\n");
        show_dump(data, datalen, stdout);
        exit(1);
    }

//handle_servers:
    servers = 0;
    datap = data;
    if(datalen >= 1) {
        servers = *datap++;
        if(servers == 0xff) {
            servers = getxx(datap, 2);  datap += 2;
        } else if(servers == 0xfe) {
            servers = getxx(datap, 4);  datap += 4;
        }
    }

    for(i = 0; i < servers; i++) {
        if(datap > (data + datalen)) break;
        datap = handle_server_info(datap, data, datalen, show_server_info, &ip, &port);
        if(!datap) break;

        ipc = ip2str(ip);
        if(!show_server_info) {
            switch(outtype) {
                case 0: {
                    fprintf(fdout, "%15s   %hu\n", ipc, port);
                    } break;
                case 5:
                case 1:
                case 2: {
                    fprintf(fdout, "%s:%hu\n", ipc, port);
                    } break;
                case 3:
                case 4: {
                    fwrite((u8 *)&ip, 1, 4, fdout);
                    fputc((port >> 8) & 0xff, fdout);
                    fputc(port & 0xff, fdout);
                    } break;
                default: break;
            }
        }

        if(execstring) {
            execptr = tmpexec + execlen;
            if(execstring_ip && !execstring_port) {
                execptr += sprintf(execptr, "%s", ipc);
                strcpy(execptr, execstring_ip + 3);

            } else if(execstring_port && !execstring_ip) {
                execptr += sprintf(execptr, "%hu", port);
                strcpy(execptr, execstring_port + 5);

            } else if(execstring_ip < execstring_port) {
                execptr += sprintf(execptr, "%s", ipc);
                execptr += sprintf(execptr, "%s", execstring_ip + 3);
                execptr += sprintf(execptr, "%hu", port);
                strcpy(execptr, execstring_port + 5);

            } else if(execstring_port < execstring_ip) {
                execptr += sprintf(execptr, "%hu", port);
                execptr += sprintf(execptr, "%s", execstring_port + 5);
                execptr += sprintf(execptr, "%s", ipc);
                strcpy(execptr, execstring_ip + 3);
            }

            fprintf(stderr, "   Execute: \"%s\"\n", tmpexec);
            system(tmpexec);
        }
    }
    servers = i;    // not needed, compatible with gslist

    if(!quiet) fprintf(stderr, "\n%u servers found\n\n", servers);

    fflush(fdout);
    if(outtype) fclose(fdout);
        // -o filename will be closed when the program is terminated

    if(multigamenamep) {
        *multigamenamep = ',';
        gamestr = multigamenamep + 1;
        goto get_list;
    } else {
        gamestr = multigamename;
    }

    if(iwannaloop) {
        close(sd);  // better to close it
        sd = 0;

        for(i = 0; i < iwannaloop; i++) {
            sleep(ONESEC);
        }
        goto get_list;
    }

//hlswlist_exit:
    if(sd) close(sd);
    FREEX(tmpexec);
    FREEX(buff);
    FREEX(data);
    return(0);
}



void lame_min_max(int op, int *min, int *max, u8 *val) {
    int     eq  = 1;    // for handling <= and >= because op covers only 1 byte, lame but fast solution

    if(op < 0) {
        op = -op;
        eq = 0;
    }
    switch(op) {
        case '>': *min = atoi(val) + eq;    break;
        case '<': *max = atoi(val) - eq;    break;
        case '=': *min = *max = atoi(val);  break;
        default: break;
    }
}



void hlsw_filter(list_opt_t *list_opt, u8 *filter) {
    int     op,
            end;
    u8      *par    = NULL,
            *val    = NULL,
            *p;

    //list_opt->game          = "";
    list_opt->mod           = "";
    list_opt->map           = "";
    list_opt->country       = "";
    list_opt->version       = "";
    list_opt->players_min   = -1;
    list_opt->players_max   = -1;
    list_opt->slots_min     = -1;
    list_opt->slots_max     = -1;
    list_opt->freeslots_min = -1;
    list_opt->freeslots_max = -1;
    list_opt->password      = 0;
    list_opt->tv_proxy      = 0;

    if(!filter) return;

    p = filter;
    while(*p) {
        par = NULL;
        val = NULL;
        for(op = end = 0; *p; p++) {
            if((*p <= ' ') || strchr(",;|()\"\'", *p)) {
                *p = 0;
                if(end) {
                    p++;
                    break;
                }
                if(op) val = p + 1;
            } else {
                if(!par) {
                    par = p;
                } else {
                    if(op && !end) end = 1;
                    if(strchr(":=<>", *p)) {
                        if(!op) {
                            op = *p;
                        } else if(op && (*p == '=')) {
                            op = -op;
                        }
                        *p = 0;
                        val = p + 1;
                    }
                }
            }
        }
        if(!par || !val) continue;

        if(!stricmp(par, "mod")) {
            list_opt->mod           = val;
        } else if(!stricmp(par, "map") || !stricmp(par, "mapname")) {
            list_opt->map           = val;
        } else if(!stricmp(par, "country")) {
            list_opt->country       = val;
        } else if(!stricmp(par, "version")) {
            list_opt->version       = val;
        } else if(!stricmp(par, "players")) {
            lame_min_max(op, &list_opt->players_min, &list_opt->players_max, val);
        } else if(!stricmp(par, "slots")) {
            lame_min_max(op, &list_opt->slots_min, &list_opt->slots_max, val);
        } else if(!stricmp(par, "freeslots") || !stricmp(par, "free_slots")) {
            lame_min_max(op, &list_opt->freeslots_min, &list_opt->freeslots_max, val);
        } else if(!stricmp(par, "password")) {
            list_opt->password      = atoi(val) ? 3 : 2;
        } else if(!stricmp(par, "tv") || !stricmp(par, "tvproxy") || !stricmp(par, "tv_proxy")) {
            list_opt->tv_proxy      = atoi(val) ? 3 : 2;
        }
    }
}



static const char   *countries[][2] = {
    { "[Codes]", "" },
    { "US", "United States" },
    { "ZZ", "Other" },
    { "AF", "Afghanistan" },
    { "AL", "Albania" },
    { "DZ", "Algeria" },
    { "AS", "American Samoa" },
    { "AD", "Andorra" },
    { "AO", "Angola" },
    { "AI", "Anguilla" },
    { "AQ", "Antarctica" },
    { "AG", "Antigua and Barbuda" },
    { "AR", "Argentina" },
    { "AM", "Armenia" },
    { "AW", "Aruba" },
    { "AU", "Australia" },
    { "AT", "Austria" },
    { "AZ", "Azerbaijan" },
    { "BS", "Bahamas" },
    { "BH", "Bahrain" },
    { "BD", "Bangladesh" },
    { "BB", "Barbados" },
    { "BY", "Belarus" },
    { "BE", "Belgium" },
    { "BZ", "Belize" },
    { "BJ", "Benin" },
    { "BM", "Bermuda" },
    { "BT", "Bhutan" },
    { "BO", "Bolivia" },
    { "BA", "Bosnia and Herzegovina" },
    { "BW", "Botswana" },
    { "BV", "Bouvet Island" },
    { "BR", "Brazil" },
    { "IO", "British Indian Ocean Territory" },
    { "BN", "Brunei Darussalam" },
    { "BG", "Bulgaria" },
    { "BF", "Burkina Faso" },
    { "BI", "Burundi" },
    { "KH", "Cambodia" },
    { "CM", "Cameroon" },
    { "CA", "Canada" },
    { "CV", "Cape Verde" },
    { "CT", "Catalonia" },
    { "KY", "Cayman Islands" },
    { "CF", "Central African Republic" },
    { "TD", "Chad" },
    { "CL", "Chile" },
    { "CN", "China" },
    { "CX", "Christmas Island" },
    { "CC", "Cocos (Keeling Islands)" },
    { "CO", "Colombia" },
    { "KM", "Comoros" },
    { "CG", "Congo" },
    { "CK", "Cook Islands" },
    { "CR", "Costa Rica" },
    { "CI", "Cote D'Ivoire (Ivory Coast)" },
    { "HR", "Croatia (Hrvatska)" },
    { "CU", "Cuba" },
    { "CY", "Cyprus" },
    { "CZ", "Czech Republic" },
    { "DK", "Denmark" },
    { "DJ", "Djibouti" },
    { "DM", "Dominica" },
    { "DO", "Dominican Republic" },
    { "TP", "East Timor" },
    { "EC", "Ecuador" },
    { "EG", "Egypt" },
    { "SV", "El Salvador" },
    { "GQ", "Equatorial Guinea" },
    { "ER", "Eritrea" },
    { "EE", "Estonia" },
    { "ET", "Ethiopia" },
    { "FK", "Falkland Islands (Malvinas)" },
    { "FO", "Faroe Islands" },
    { "FJ", "Fiji" },
    { "FI", "Finland" },
    { "FR", "France" },
    { "FX", "France Metropolitan" },
    { "GF", "French Guiana" },
    { "PF", "French Polynesia" },
    { "TF", "French Southern Territories" },
    { "GA", "Gabon" },
    { "GM", "Gambia" },
    { "GE", "Georgia" },
    { "DE", "Germany" },
    { "GH", "Ghana" },
    { "GI", "Gibraltar" },
    { "GR", "Greece" },
    { "GL", "Greenland" },
    { "GD", "Grenada" },
    { "GP", "Guadeloupe" },
    { "GU", "Guam" },
    { "GT", "Guatemala" },
    { "GN", "Guinea" },
    { "GW", "Guinea-Bissau" },
    { "GY", "Guyana" },
    { "HT", "Haiti" },
    { "HM", "Heard and McDonald Islands" },
    { "HN", "Honduras" },
    { "HK", "Hong Kong" },
    { "HU", "Hungary" },
    { "IS", "Iceland" },
    { "IN", "India" },
    { "ID", "Indonesia" },
    { "IR", "Iran" },
    { "IQ", "Iraq" },
    { "IE", "Ireland" },
    { "IL", "Israel" },
    { "IT", "Italy" },
    { "JM", "Jamaica" },
    { "JP", "Japan" },
    { "JO", "Jordan" },
    { "KZ", "Kazakhstan" },
    { "KE", "Kenya" },
    { "KI", "Kiribati" },
    { "KP", "North Korea" },
    { "KR", "South Korea" },
    { "KW", "Kuwait" },
    { "KG", "Kyrgyzstan" },
    { "LA", "Laos" },
    { "LV", "Latvia" },
    { "LB", "Lebanon" },
    { "LS", "Lesotho" },
    { "LR", "Liberia" },
    { "LY", "Libya" },
    { "LI", "Liechtenstein" },
    { "LT", "Lithuania" },
    { "LU", "Luxembourg" },
    { "MO", "Macau" },
    { "MK", "Macedonia" },
    { "MG", "Madagascar" },
    { "MW", "Malawi" },
    { "MY", "Malaysia" },
    { "MV", "Maldives" },
    { "ML", "Mali" },
    { "MT", "Malta" },
    { "MH", "Marshall Islands" },
    { "MQ", "Martinique" },
    { "MR", "Mauritania" },
    { "MU", "Mauritius" },
    { "YT", "Mayotte" },
    { "MX", "Mexico" },
    { "FM", "Micronesia" },
    { "MD", "Moldova" },
    { "MC", "Monaco" },
    { "MN", "Mongolia" },
    { "MS", "Montserrat" },
    { "MA", "Morocco" },
    { "MZ", "Mozambique" },
    { "MM", "Myanmar (Burma)" },
    { "NA", "Namibia" },
    { "NR", "Nauru" },
    { "NP", "Nepal" },
    { "NL", "Netherlands" },
    { "AN", "Netherlands Antilles" },
    { "NC", "New Caledonia" },
    { "NZ", "New Zealand" },
    { "NI", "Nicaragua" },
    { "NE", "Niger" },
    { "NG", "Nigeria" },
    { "NU", "Niue" },
    { "NF", "Norfolk Island" },
    { "MP", "Northern Mariana Islands" },
    { "NO", "Norway" },
    { "OM", "Oman" },
    { "PK", "Pakistan" },
    { "PW", "Palau" },
    { "PA", "Panama" },
    { "PG", "Papua New Guinea" },
    { "PY", "Paraguay" },
    { "PE", "Peru" },
    { "PH", "Philippines" },
    { "PN", "Pitcairn" },
    { "PL", "Poland" },
    { "PT", "Portugal" },
    { "PR", "Puerto Rico" },
    { "QA", "Qatar" },
    { "RE", "Reunion" },
    { "RO", "Romania" },
    { "RU", "Russian Federation" },
    { "RW", "Rwanda" },
    { "KN", "Saint Kitts and Nevis" },
    { "LC", "Saint Lucia" },
    { "VC", "Saint Vincent and The Grenadines" },
    { "WS", "Samoa" },
    { "SM", "San Marino" },
    { "ST", "Sao Tome and Principe" },
    { "SA", "Saudi Arabia" },
    { "SN", "Senegal" },
    { "SC", "Seychelles" },
    { "SL", "Sierra Leone" },
    { "SG", "Singapore" },
    { "SK", "Slovak Republic" },
    { "SI", "Slovenia" },
    { "SB", "Solomon Islands" },
    { "SO", "Somalia" },
    { "ZA", "South Africa" },
    { "GS", "S. Georgia and S. Sandwich Isls." },
    { "ES", "Spain" },
    { "LK", "Sri Lanka" },
    { "SH", "St. Helena" },
    { "PM", "St. Pierre and Miquelon" },
    { "SD", "Sudan" },
    { "SR", "Suriname" },
    { "SJ", "Svalbard and Jan Mayen Islands" },
    { "SZ", "Swaziland" },
    { "SE", "Sweden" },
    { "CH", "Switzerland" },
    { "SY", "Syria" },
    { "TW", "Taiwan" },
    { "TJ", "Tajikistan" },
    { "TZ", "Tanzania" },
    { "TH", "Thailand" },
    { "TG", "Togo" },
    { "TK", "Tokelau" },
    { "TO", "Tonga" },
    { "TT", "Trinidad and Tobago" },
    { "TN", "Tunisia" },
    { "TR", "Turkey" },
    { "TM", "Turkmenistan" },
    { "TC", "Turks and Caicos Islands" },
    { "TV", "Tuvalu" },
    { "UG", "Uganda" },
    { "UA", "Ukraine" },
    { "AE", "United Arab Emirates" },
    { "UK", "United Kingdom" },
    { "UM", "US Minor Outlying Islands" },
    { "UY", "Uruguay" },
    { "UZ", "Uzbekistan" },
    { "VU", "Vanuatu" },
    { "VA", "Vatican City" },
    { "VE", "Venezuela" },
    { "VN", "VietNam" },
    { "VG", "Virgin Islands (British)" },
    { "VI", "Virgin Islands (US)" },
    { "WF", "Wallis and Futuna Islands" },
    { "EH", "Western Sahara" },
    { "YE", "Yemen" },
    { "YU", "Yugoslavia" },
    { "ZR", "Zaire" },
    { "ZM", "Zambia" },
    { "ZW", "Zimbabwe" },
    { NULL, NULL }
};



void show_countries(void) {
    int     i;

    for(i = 0; countries[i][0]; i++) {
        fprintf(stderr, "%s   %s\n", countries[i][0], countries[i][1]);
    }
}



void show_filter_help(void) {
    fprintf(fdout,
        "  Filters:\n"
        "    mod, map, country, version, players, slots, freeslots, password, tv_proxy\n"
        "\n"
        "  Examples:\n"
        "    only passworded servers: -f password=1\n"
        "    no passworded servers:   -f password=0\n"
        "    italian servers:         -f country=IT\n"
        "    no empty servers:        -f players>0\n"
        "    players > 3 and < 5:     -f \"players>3 players<5\"\n"
        "    CoD Original mod:        -f mod=Original\n"
        "    mix:                     -f \"players>0 country=IT password=0\"\n"
        "    mix:                     -f \"password=1,map=bloodgulch,mod=CTF\"\n"
        "\n");
}



void show_list(u8 *name) {
    static const u8  *games_list[][2] = {
        { "\"\"", "ANY server of ANY game (the first 10000)" },
        { "AvP2", "Alien vs. Predator 2" },
        { "AAO", "America's Army: Operations" },
        { "ARMA2", "ARMA 2" },
        { "ARMA2OA", "ARMA 2: Operation Arrowhead" },
        { "ArmA", "Armed Assault" },
        { "BF1942", "Battlefield 1942" },
        { "BF2", "Battlefield 2" },
        { "BF2142", "Battlefield 2142" },
        { "BFV", "Battlefield Vietnam" },
        { "CCR", "C&C Renegade" },
        { "CoD", "Call of Duty" },
        { "CoD2", "Call of Duty 2" },
        { "CoD4", "Call of Duty 4" },
        { "CoDUO", "Call of Duty: United Offensive" },
        { "CoDWW", "Call of Duty: World at War" },
        { "Crysis", "Crysis" },
        { "CrysisWars", "Crysis Wars" },
        { "DarkMessiah", "Dark Messiah of Might and Magic" },
        { "DVS", "Devastation" },
        { "D3", "Doom 3" },
        { "EF", "Elite Force" },
        { "EF2", "Elite Force II" },
        { "ETQW", "Enemy Territory: Quake Wars" },
        { "FEAR", "F.E.A.R." },
        { "FEARXP2", "F.E.A.R. Perseus Mandate" },
        { "FarCry", "Far Cry" },
        { "FFOW", "Frontlines-Fuel of War" },
        { "GRAW2", "Ghost Recon Advanced Warfighter 2" },
        { "HL", "Half-Life" },
        { "HLWON", "Half-Life (WON)" },
        { "HL2", "Half-Life 2" },
        { "HALO", "HALO" },
        { "ioQ3", "ioQuake3" },
        { "JK2", "Jedi Knight 2" },
        { "JK3", "Jedi Knight 3" },
        { "KF", "Killing Floor" },
        { "L4D", "Left 4 Dead" },
        { "L4D2", "Left 4 Dead 2" },
        { "LOTD", "Land Of The Dead: Road to Fiddler's Green" },
        { "MoHAA", "Medal of Honor Allied Assault" },
        { "MoHPA", "Medal of Honor Pacific Assault" },
        { "MoHAAB", "Medal of Honor: Allied Assault Breakthrough" },
        { "MoHAAS", "Medal of Honor: Allied Assault Spearhead" },
        { "NWN", "Neverwinter Nights" },
        { "Nexuiz", "Nexuiz" },
        { "OPF", "Operation Flashpoint" },
        { "OPFR", "Operation Flashpoint Resistance" },
        { "PK", "Painkiller" },
        { "Pariah", "Pariah" },
        { "Postal2", "Postal 2" },
        { "PREY", "Prey" },
        { "IGI2", "Project IGI2: Covert Strike" },
        { "Q1", "Quake 1" },
        { "Q2", "Quake 2" },
        { "Q3", "Quake 3 Arena" },
        { "Q4", "Quake 4" },
        { "ROO", "Red Orchestra: Ostfront 41-45" },
        { "RtCW", "Return to Castle Wolfenstein" },
        { "Rune", "Rune" },
        { "STALKER", "S.T.A.L.K.E.R. - Shadow of Chernobyl" },
        { "Sav", "Savage: The Battle For Newerth" },
        { "Shockvoice", "Shockvoice (v0.9.5+)" },
        { "SS:TFE", "Serious Sam: The First Encounter" },
        { "SS:TSE", "Serious Sam: The Second Encounter" },
        { "SniperElite", "Sniper Elite PC" },
        { "SoF2", "Soldier of Fortune 2" },
        { "SWBF", "Star Wars: Battlefront" },
        { "SWAT4", "SWAT 4" },
        { "SWAT4X", "SWAT 4: The Stetchkov Syndicate" },
        { "Tremulous", "Tremulous" },
        { "Tribes2", "Tribes 2" },
        { "TribesV", "Tribes Vengeance" },
        { "UT", "Unreal Tournament" },
        { "UT2003", "Unreal Tournament 2003" },
        { "UT2004", "Unreal Tournament 2004" },
        { "UT3", "Unreal Tournament 3" },
        { "UrT", "Urban Terror" },
        { "Ventrilo", "Ventrilo (v2.3.0+)" },
        { "Vietcong", "Vietcong" },
        { "WSW", "Warsow" },
        { "ET", "Wolfenstein - Enemy Territory" },
        { "Wolf09", "Wolfenstein 2009" },
        { NULL, NULL }
    };
    int     i;

    if(!quiet) {
        fprintf(fdout,
            "DESCRIPTION                                           GAMENAME\n"
            "-------------------------------------------------------------------------------\n");
    }

    for(i = 0; games_list[i][0]; i++) {
        if(!name || (stristr(games_list[i][0], name) || stristr(games_list[i][1], name))) {
            fprintf(fdout, "%-53s %s\n", games_list[i][1], games_list[i][0]);
        }
    }
    if(!name) {
        fprintf(fdout, "\n"
            "- like in gslist you need only to know the correct gamename also if it's not\n"
            "  included in the above list (which could be not updated)\n");
    }
}



int enctypex_data_cleaner(unsigned char *dst, unsigned char *src, int max) {
    static const unsigned char strange_chars[] = {
                    ' ','E',' ',',','f',',','.','t',' ','^','%','S','<','E',' ','Z',
                    ' ',' ','`','`','"','"','.','-','-','~','`','S','>','e',' ','Z',
                    'Y','Y','i','c','e','o','Y','I','S','`','c','a','<','-','-','E',
                    '-','`','+','2','3','`','u','P','-',',','1','`','>','%','%','%',
                    '?','A','A','A','A','A','A','A','C','E','E','E','E','I','I','I',
                    'I','D','N','O','O','O','O','O','x','0','U','U','U','U','Y','D',
                    'B','a','a','a','a','a','a','e','c','e','e','e','e','i','i','i',
                    'i','o','n','o','o','o','o','o','+','o','u','u','u','u','y','b',
                    'y' };
    unsigned char   c,
                    *p;

    if(!dst) return(0);
    if(dst != src) dst[0] = 0;
    if(!src) return(0);

    if(max < 0) max = strlen(src);

    for(p = dst; (c = *src) && (max > 0); src++, max--) {
        if(c == '\\') {                     // avoids the backslash delimiter
            *p++ = '/';
            continue;
        }

        if(enctypex_data_cleaner_level >= 1) {
            if(c == '^') {                  // Quake 3 colors
                //if(src[1] == 'x') {         // ^x112233 (I don't remember the game which used this format)
                    //src += 7;
                    //max -= 7;
                //} else
                if(isdigit(src[1]) || islower(src[1])) { // ^0-^9, ^a-^z... a good compromise
                    src++;
                    max--;
                } else {
                    *p++ = c;
                }
                continue;
            }
            if(c == 0x1b) {                 // Unreal colors
                src += 3;
                max -= 3;
                continue;
            }
            if(c < ' ') {                   // other colors
                continue;
            }
        }

        if(enctypex_data_cleaner_level >= 2) {
            if(c >= 0x7f) c = strange_chars[c - 0x7f];
        }

        if(enctypex_data_cleaner_level >= 3) {
            switch(c) {                     // html/SQL injection (paranoid mode)
                case '\'':
                case '\"':
                case '&':
                case '^':
                case '?':
                case '{':
                case '}':
                case '(':
                case ')':
                case '[':
                case ']':
                case '-':
                case ';':
                case '~':
                case '|':
                case '$':
                case '!':
                case '<':
                case '>':
                case '*':
                case '%':
                case ',': c = '.';  break;
                default: break;
            }
        }

        if((c == '\r') || (c == '\n')) {    // no new line
            continue;
        }
        *p++ = c;
    }
    *p = 0;
    return(p - dst);
}



u8 *server_info_string(u8 *p, u8 *limit, int show_server_info, u8 *str) {
    static  int tmpsz   = 0;
    static  u8  *tmp    = NULL;
    int     len,
            tmplen;

    len = *p++;
    if(len == 0xff) {
        len = getxx(p, 2);  p += 2;
    } else if(len == 0xfe) {
        len = getxx(p, 4);  p += 4;
    }
    if((p + len) > limit) return(NULL);
    if(show_server_info && (len > 0)) {
        myalloc(&tmp, len + 1, &tmpsz);
        tmplen = enctypex_data_cleaner(tmp, p, len);
        fprintf(fdout, "\\%s\\%s", str, tmp);
    }
    return(p + len);
}



u8 *handle_server_info(u8 *p, u8 *data, int datalen, int show_server_info, u32 *ret_ip, u16 *ret_port) {
    u32     ip;
    u16     hostport,
            queryport,
            numplayers,
            maxplayers,
            spectators;
    u8      *limit,
            password;

    *ret_ip     = 0;
    *ret_port   = 0;

    limit = data + datalen;

    ip        = p[3] | (p[2] << 8) | (p[1] << 16) | (p[0] << 24);   p += 4;
    hostport  = getxx(p, 2);    p += 2;
    queryport = getxx(p, 2);    p += 2;
    if(p > limit) return(NULL);

    *ret_ip     = ip;
    *ret_port   = queryport;
    if(show_server_info) fprintf(fdout, "%s:%hu \\hostport\\%hu", ip2str(ip), queryport, hostport); // gslist compatible

    p = server_info_string(p, limit, show_server_info, "gamename"); if(!p) return(NULL);
    p = server_info_string(p, limit, show_server_info, "hostname"); if(!p) return(NULL);
    numplayers = getxx(p, 2);   p += 2; if(show_server_info) fprintf(fdout, "\\numplayers\\%u", numplayers);
    maxplayers = getxx(p, 2);   p += 2; if(show_server_info) fprintf(fdout, "\\maxplayers\\%u", maxplayers);
                 getxx(p, 2);   p += 2;                 // ???
    spectators = getxx(p, 2);   p += 2; if(show_server_info && spectators) fprintf(fdout, "\\spectators\\%u", spectators);
    password   = getxx(p, 1);   p += 1; if(show_server_info) fprintf(fdout, "\\password\\%u",   password);
    p = server_info_string(p, limit, show_server_info, "mapname");  if(!p) return(NULL);
    p = server_info_string(p, limit, show_server_info, "gamemode"); if(!p) return(NULL);
    p = server_info_string(p, limit, show_server_info, "gamemod");  if(!p) return(NULL);    // gamevariant
    p = server_info_string(p, limit, show_server_info, "gametype"); if(!p) return(NULL);
    p += 2;                                             // delimiter 0xff 0xff
    if(p > limit) return(NULL);
    if(show_server_info) fprintf(fdout, "\n");
    return(p);
}



void show_help(void) {
    fprintf(stdout, "\n"
        "Usage: %s [options]\n"
        "\n"
        "Options:\n"
        "-n GAMENAME    servers list of the game GAMENAME (use -l or -s to know it)\n"
        "               you can also specify multiple GAMENAMEs: -n GN1,GN2,...,GNN\n"
        "-l             complete list of supported games and their details\n"
        "-s PATTERN     search for games in the database (case insensitve)\n"
        "-u             update the database of supported games (debug, not implemented)\n"
        "-f FILTERS     specify a filter to apply to the servers list. Use -f ? for help\n"
        "-r \"prog...\"   lets you to execute a specific program for each IP found.\n"
        "               there are 2 available parameters: #IP and #PORT automatically\n"
        "               substituited with the IP and port of the each online game server\n"
        "-o OUT         specify a different output for the servers list (default output\n"
        "               is screen). Use -o ? for the list of options\n"
        "-q             quiet output, many informations will be not shown\n"
        "-x S[:P]       specify a different master server (S) and port (P is optional)\n"
        "               default is %s:%d\n"
        "-L SEC         continuous servers list loop each SEC seconds\n"
        "-c             list of country codes to use with the \"country\" filter\n"
        "-X INFO        show the informations for each server instead of the IP list,\n"
        "               INFO is not handled at the moment so use just -X none\n"
        "-C             do not filter colors from the game info replied by the servers\n"
        "\n", "hlswlist", mshost, msport);
}



void show_output_help(void) {
    fputs("\n"
        "  1 = text output to a file for each game (ex: serioussam.gsl).\n"
        "      string: 1.2.3.4:1234 (plus a final line-feed)\n"
        "  2 = text output as above but to only one file ("FILEOUT")\n"
        "  3 = binary output to a file for each game: 4 bytes IP, 2 port\n"
        "      example: (hex) 7F0000011E62 = 127.0.0.1 7778\n"
        "  4 = binary output as above but to only one file ("FILEOUT")\n"
        "  5 = exactly like 1 but to stdout\n"
        "  6 = hexadecimal visualization of the raw servers list as is\n"
        "  FILENAME = if OUT is a filename all the screen output will be\n"
        "             dumped into the file FILENAME\n"
        "\n", stdout);
}



u8 *ip2str(u32 ip) {
    static u8   data[16];

    sprintf(data, "%u.%u.%u.%u",
        (ip & 0xff), ((ip >> 8) & 0xff), ((ip >> 16) & 0xff), ((ip >> 24) & 0xff));
    return(data);
}



int bunzip2(u8 *in, int insz, u8 *out, int outsz) {
    int     err;

    err = BZ2_bzBuffToBuffDecompress(out, &outsz, in, insz, 0, 0);
    if(err != BZ_OK) {
        fprintf(stderr, "\nError: invalid bz2 compressed data (%d)\n", err);
        exit(1);
    }
    return(outsz);
}



int hlsw_send(int sd, int e1, u8 *data, int len) {
    static int  buffsz  = 0;    // fast solution
    static u8   *buff   = NULL;
    int     slen;

    slen = 8 + len;
    if(slen > buffsz) {
        buffsz = slen;
        buff = realloc(buff, slen);
        if(!buff) std_err();
    }
    putxx(buff,     0xcece, 2);
    putxx(buff + 2, 0x0000, 2);
    putxx(buff + 4, len, 4);
    memcpy(buff + 8, data, len);
    hlsw_enc(e1, buff + 8, len);

    if(send(sd, buff, slen, 0) != slen) return(-1);
    return(0);
}



u8 *hlsw_recv(int sd, int *ret_len) {
    static int  buffsz  = 0;
    static u8   *buff   = NULL;
    u32     len;
    u8      header[8];

    *ret_len = -1;
    if(recv_tcp(sd, header, 8) < 0) return(NULL);
    if(header[0] != 0xce) return(NULL);
    if(header[1] != 0xce) return(NULL);

    len = getxx(header + 4, 4);
    if(len > buffsz) {
        buffsz = len;
        buff = realloc(buff, len);
        if(!buff) std_err();
    }

    if(recv_tcp(sd, buff, len) < 0) return(NULL);
    hlsw_dec(buff, len);

    *ret_len = len;
    return(buff);
}



int recv_tcp(int sd, u8 *data, int datalen) {
    int     len,
            t;

    for(len = 0; len < datalen; len += t) {
        if(timeout(sd, 10) < 0) return(-1);
        t = recv(sd, data + len, datalen - len, 0);
        if(t <= 0) return(-1);
    }
    return(len);
}



int hlsw_enc(int e1, u8 *data, int len) {
    int     i,
            t;
    u8      tmp[2],
            *p;

    if(len <= 0) return(len);

    if((e1 != 1) && (e1 != 2)) return(len);

    tmp[0] = data[0];
    tmp[1] = data[1];
    if(e1 == 2) {
        p = data;
        for(t = 0x20; t < (len - 2); t <<= 1) {
            tmp[0] = data[t];
            tmp[1] = data[t + 1];
            data[t]     = p[0];
            data[t + 1] = p[1];
        }
    }
    data[len - 2] = tmp[0];
    data[len - 1] = tmp[1];

    data[0] = e1;
    data[1] = rand();

    for(i = 1; i < (len - 1); i++) {
        data[i + 1] ^= data[i];
    }
    return(len);
}



int hlsw_dec(u8 *data, int len) {
    int     i,
            t;
    u8      tmp[2],
            e1,
            *p;

    if(len <= 0) return(len);

    e1 = data[0];
    if((e1 != 1) && (e1 != 2)) return(len);

    for(i = len - 1; i >= 2; i--) {
        data[i] ^= data[i - 1];
    }

    if(len < 2) return(len);

    tmp[0] = data[len - 2];
    tmp[1] = data[len - 1];
    p = data;
    if(e1 == 2) {
        for(t = 0x20; t < (len - 2); t <<= 1) {
            p[0] = data[t];
            p[1] = data[t + 1];
            p = data + t;
        }
    }
    p[0] = tmp[0];
    p[1] = tmp[1];
    return(len);
}



void myalloc(u8 **data, int wantsize, int *currsize) {
    if(!wantsize) return;
    if(wantsize <= *currsize) {
        if(*currsize > 0) return;
    }
    *data = realloc(*data, wantsize);
    if(!*data) std_err();
    *currsize = wantsize;
}



int putsx(u8 *data, u8 *str) {
    int     t,
            len;
    u8      *p;

    p = data;
    len = strlen(str);
    if(len < 0xfe) {
        t = 1;
    } else if(len < 0xffff) {
        *p++ = 0xff;
        t = 2;
    } else {
        *p++ = 0xfe;
        t = 4;
    }
    p += putxx(p, len, t);
    p += putmm(p, str, len);
    return(p - data);
}



int putmm(u8 *data, u8 *str, int len) {
    memcpy(data, str, len);
    return(len);
}



int putcc(u8 *data, int chr, int len) {
    memset(data, chr, len);
    return(len);
}



int putxx(u8 *data, u32 num, int bytes) {
    int     i;

    for(i = 0; i < bytes; i++) {
        data[i] = num >> (i << 3);
    }
    return(bytes);
}



u32 getxx(u8 *data, int bytes) {
    u32     num;
    int     i;

    for(num = i = 0; i < bytes; i++) {
        num |= (data[i] << (i << 3));
    }
    return(num);
}



int timeout(int sock, int secs) {
    struct  timeval tout;
    fd_set  fd_read;

    tout.tv_sec  = secs;
    tout.tv_usec = 0;
    FD_ZERO(&fd_read);
    FD_SET(sock, &fd_read);
    if(select(sock + 1, &fd_read, NULL, NULL, &tout)
      <= 0) return(-1);
    return(0);
}



u32 resolv(char *host) {
    struct  hostent *hp;
    u32     host_ip;

    host_ip = inet_addr(host);
    if(host_ip == INADDR_NONE) {
        hp = gethostbyname(host);
        if(!hp) {
            fprintf(stderr, "\nError: Unable to resolv hostname (%s)\n", host);
            exit(1);
        } else host_ip = *(u32 *)hp->h_addr;
    }
    return(host_ip);
}



#ifndef WIN32
    void std_err(void) {
        perror("\nError");
        exit(1);
    }
#endif


