/*
    Copyright 2005,2006,2007,2008 Luigi Auriemma

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    http://www.gnu.org/licenses/gpl-2.0.txt
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <time.h>
#include <errno.h>
#include <sys/stat.h>
#include <ctype.h>
#include <zlib.h>
#include "mycrc32.h"
#include "mydownlib.h"
#include "sha1.h"

#ifdef WIN32
    #include <direct.h>
    #include <windows.h>

    #define close       closesocket
    #define sleep       Sleep
    #define usleep      sleep
    #define ONESEC      1000
    #define check_time  clock()
    #define make_dir(x) mkdir(x)
#else
    #include <unistd.h>
    #include <sys/socket.h>
    #include <sys/types.h>
    #include <arpa/inet.h>
    #include <netinet/in.h>
    #include <netdb.h>
    #include <sys/stat.h>
    #include <sys/times.h>
    #include <sys/timeb.h>

    #define stristr     strcasestr
    #define ONESEC      1
    #define check_time  (times(0) * 10)
    #define make_dir(x) mkdir(x, 0755)
#endif

typedef uint8_t     u8;
typedef uint16_t    u16;
typedef uint32_t    u32;
typedef uint64_t    u64;



#define VER             "0.2d"

#define DIRHOST         "directory.jmeeting.com"
#define DIRPORT         80
#define VIDEOHOST       "video%d.jmeeting.com"
#define VIDEOPORT       8080

#define JMEETING_DIR    "http://%s:%hu/data"
#define JMEETING_VIDEO  "http://%s:%hu/%s"

#define JMEETING_FD     "jmeeting.dat"
#define WATCH_THUMB     "watch_thumb.htm"
#define WATCH_EVER      "watch_ever.htm"
#define WATCH_NOW       "watch_now.htm"

#define PWD             "Content-Password: "

#define REFRESH_LIST    180

#define SOCKERR         (-1)
#define TIMEERR         (-2)



typedef struct {
    u8      *name;
    u32     crc;
    u8      *url;
    u8      disabled;
    u8      *password;
} ids_t;



int     record  = 0,
        askpass = 1;



void free_ids(ids_t *ids);
u8 *do_sha1(u8 *data, int len, u8 *out);
u8 *create_ticket(u8 *id);
u8 *build_dir_url(void);
u8 *build_video_url(u8 *nickname, int videoserver);
void check_filter(u8 *p);
u8 *myitoa(int num);
u8 *get_date(void);
void launch_browser(u8 *fname);
ids_t *getids(u8 *buff, u8 *filter);
u8 *fdgetz(u8 **buff);
void create_folder(u8 *folder);
u8 *fdload(u8 *fname);
u8 *unzip(u8 *in, int *insz);
void jmeetrec(ids_t *ids);
int rnd_mem(u8 *data, int len, u32 rnd);
int save_file(u8 *data, int len, u8 *id, u32 frame);
void write_watch_ever(void);
void write_watch_now(u8 *fname);
u32 mysqrt(u32 num);
void write_watch_thumb(ids_t *ids);
void show_help(void);
void myfree(u8 **buff);
int fgetz(u8 *data, int datalen);
void std_err(void);



int main(int argc, char *argv[]) {
    mydown_options  opt;
    ids_t   *ids;
    int     i,
            list            = 0,
            run_browser     = 0;
    u8      wwwdata[256],
            user[32],
            pass[32],
            *jmeetlist      = NULL,
            *jmeeting_dir   = NULL,
            *fname          = JMEETING_FD,
            *filter         = NULL,
            *webcam_id      = NULL;

    setbuf(stdout, NULL);

    fputs("\n"
        "JmeetREC "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", stderr);

    if(argc < 2) {
        printf("\n"
            "Usage %s [options] [nickname]\n"
            "\n", argv[0]);
        show_help();
    }

    for(i = 1; i < argc; i++) {
        if(((argv[i][0] != '-') && (argv[i][0] != '/')) || (strlen(argv[i]) != 2)) {
            if(i == (argc - 1)) {
                webcam_id = argv[i];
                break;
            }
            fprintf(stderr, "\nError: wrong command-line argument (%s)\n", argv[i]);
            exit(1);
        }
        switch(argv[i][1]) {
            case 'r': {
                if(!argv[++i]) show_help();
                webcam_id         = argv[i];
                } break;
            case 'f': {
                if(!argv[++i]) show_help();
                filter            = argv[i];
                check_filter(filter);
                } break;
            case 'c': {
                if(!argv[++i]) show_help();
                fname             = argv[i];
                } break;
            case 'l': list        = 1;                  break;
            case 'i': run_browser = 1;                  break;
            default: {
                printf("\nError: wrong command-line argument (%s)\n", argv[i]);
                exit(1);
            }
        }
    }

    if(webcam_id) {
        printf("- start recording of the webcam %s\n", webcam_id);

        ids = calloc(sizeof(ids_t), 2);
        ids[0].name = webcam_id;
        ids[0].crc  = 0;
        ids[0].url  = build_video_url(webcam_id, 1);

        create_folder(webcam_id);
        create_folder(get_date());
        write_watch_ever();
        if(run_browser) launch_browser(WATCH_EVER);

        record = 1;

        jmeetrec(ids);

        goto quit;
    }

    printf("- start thumbnails collection\n");

    create_folder("thumb");

    jmeeting_dir = build_dir_url();

    memset(&opt, 0, sizeof(opt));
    opt.useragent   = "Mozilla/4.0 (Windows XP 5.1) Java/1.6.0_12";
    opt.verbose     = 1;
    opt.more_http   = wwwdata;
    opt.timeout     = 5;
    opt.onflyunzip  = 1;

    for(;;) {
        if(fname == (u8 *)JMEETING_FD) {
            rnd_mem(user, 1, time(NULL));
            rnd_mem(pass, 1, time(NULL));

            sprintf(wwwdata,
                "Content-type: application/x-www-form-urlencoded\r\n"
                "Credentials: %s@%s",
                user,
                do_sha1(pass, strlen(pass), NULL));

            if(mydown(jmeeting_dir, fname, &opt) == MYDOWN_ERROR) continue;
        }

        myfree((void *)&jmeetlist);
        jmeetlist = fdload(fname);
        if(!jmeetlist) {
            printf("\nError: something wrong during the loading of the JMEETING file\n");
            exit(1);
        }

        ids = getids(jmeetlist, filter);

        if(list) goto quit;

        write_watch_thumb(ids);
        if(run_browser) {
            launch_browser(WATCH_THUMB);
            run_browser = 0;
        }

        jmeetrec(ids);
    }

quit:
    free_ids(ids);
    return(0);
}



void free_ids(ids_t *ids) {
    int     idsnum;

    for(idsnum = 0; ids[idsnum].name; idsnum++) {
        free(ids[idsnum].url);
    }
    free(ids);
}



u8 *do_sha1(u8 *data, int len, u8 *out) {
    sha1_context    sha1t;
    int     i;
    u8      sha1h[20],
            *p;
    static const u8 hex[16] = "0123456789abcdef";
    static u8       tmpout[41];

    sha1_starts(&sha1t);
    sha1_update(&sha1t, data, len);
    sha1_finish(&sha1t, sha1h);

    if(!out) out = tmpout;
    p = out;
    for(i = 0; i < 20; i++) {
        *p++ = hex[sha1h[i] >> 4];
        *p++ = hex[sha1h[i] & 15];
    }
    *p = 0;
    return(out);
}



u8 *create_ticket(u8 *id) {
    u64     timems;
    int     len;
    u8      buff[64];
    static u8  data[41];

    timems = time(NULL);
    timems *= 1000;

    len = sprintf(
        id,
        "%x",
        (u32)(timems >> 32));
    sprintf(
        id + len,
        "%08x",
        (u32)(timems & 0xffffffff));

    len = sprintf(buff, "jsyntax@%s", id);
    do_sha1(buff, len, data);
    return(data);
}



u8 *build_dir_url(void) {
    u8      *url;

    url = malloc(
        strlen(JMEETING_DIR) +
        strlen(DIRHOST) +
        10);

    sprintf(url,
        JMEETING_DIR,
        DIRHOST, DIRPORT);

    return(url);
}



u8 *build_video_url(u8 *nickname, int videoserver) {
    u8      tmp[64],
            *url;

    sprintf(tmp, VIDEOHOST, videoserver);

    url = malloc(
        strlen(JMEETING_VIDEO) +
        strlen(tmp)            +
        strlen(nickname)       +
        10);

    sprintf(url,
        JMEETING_VIDEO,
        tmp, VIDEOPORT, nickname);

    return(url);
}



void check_filter(u8 *p) {
    for(; *p && (*p != ','); p++) {
        *p = tolower(*p);
        switch(*p) {
            case 'f':
            case 'm':
            case 'c':
            case 'u':
            case 'p': break;
            case 'o': askpass = 0;  break;
            default: {
                printf("\nError: wrong filter, recheck it\n");
                exit(1);
                } break;
        }
    }
}



u8 *myitoa(int num) {
    static u8  mini[12];
    sprintf(mini, "%u", num);
    return(mini);
}



u8 *get_date(void) {
    time_t  datex;
    struct  tm  *tmx;
    static u8  folder[32];
    static const u8 *months[12] = {
                "jan","feb","mar","apr","may","jun",
                "jul","aug","sep","oct","nov","dec"
            };

    time(&datex);
    tmx = localtime(&datex);
    sprintf(
        folder,
        "%02hhu.%s.%02hhu-%02hhu.%02hhu",
        tmx->tm_mday, months[tmx->tm_mon], tmx->tm_year % 100,
        tmx->tm_hour, tmx->tm_min);

    return(folder);
}



void launch_browser(u8 *fname) {
    u8      tmp[256];

#ifdef WIN32
    getcwd(tmp, sizeof(tmp) - strlen(fname));
    sprintf(tmp + strlen(tmp), "\\%s", fname);
    printf("- launch browser: %s\n", tmp);

    ShellExecute(
        NULL,
        "open",
        tmp,
        NULL,
        NULL,
        SW_SHOW);
#else
    strcpy(tmp, "$BROWSER ");
    getcwd(tmp + 9, sizeof(tmp) - 9 - strlen(fname) - 3);
    sprintf(tmp + strlen(tmp), "/%s &", fname);
    printf("- launch browser: %s\n", tmp);

    system(tmp);
#endif
}



ids_t *getids(u8 *buff, u8 *filter) {
    ids_t   *ids;
    u32     idsnum;
    int     ok;
    u8      type,
            priv,
            vnum,
            *nick,
            *data,
            *p,
            *l;

    idsnum = 1;                 // full allocation
    for(p = buff; *p; p++) {
        if(*p == '\n') idsnum++;
    }
    ids = calloc(sizeof(ids_t), idsnum + 1);
    if(!ids) std_err();
    idsnum = 0;

    printf("- webcams:\n\n");

    while((data = fdgetz(&buff))) {

#define DOIT(x,y)   p = strchr(data, x);    \
                    if(!p) continue;        \
                    *p = 0;                 \
                    y;                      \
                    data = p + 1;

        DOIT('@', nick = data)
        DOIT('.', type = data[0])
        priv = data[0];
        vnum = data[2];
        if(!vnum) vnum = 1;

#undef DOIT

        if(filter) {
            ok = 0;
            p = strchr(filter, ',');
            if(p) *p = 0;

            switch(type) {
                case 'f': if(strchr(filter, 'f')) ok = 1;   break;
                case 'm': if(strchr(filter, 'm')) ok = 1;   break;
                case 'c': if(strchr(filter, 'c')) ok = 1;   break;
                case 'u': if(strchr(filter, 'u')) ok = 1;   break;
            }

            if(ok) {
                if(
                  ((priv == 'o') && (strchr(filter, 'o'))) ||
                  ((priv == 'p') && (strchr(filter, 'p'))) ||
                  (!strchr(filter, 'o') && !strchr(filter, 'p'))
                ) {
                    ok = 1;
                } else {
                    continue;
                }
            }

            if(p) {
                *p = ',';
                for(p++; !ok; p = l + 1) {
                    l = strchr(p, ',');
                    if(l) *l = 0;
                    if(stristr(nick, p)) ok = 1;
                    if(!l) break;
                    *l = ',';
                }
            }

            if(!ok) continue;
        }

        printf("  %-50s %c%c%c\n", nick, type, priv, vnum);
        ids[idsnum].name = nick;
        ids[idsnum].crc  = 0;
        ids[idsnum].url  = build_video_url(nick, vnum - '0');
        idsnum++;
    }

    if(!idsnum) {
        printf("\n- no webcams in the filter\n");
        free_ids(ids);
        exit(1);
    }

    printf("\n- %u webcams in the filter\n", idsnum);
    return(ids);
}



u8 *fdgetz(u8 **buff) {
    u8      *ret,
            *p;

    for(ret = p = *buff; *p && (*p != '\n') && (*p != '\r'); p++);
    if(!(p - ret)) return(NULL);
    *p++ = 0;
    while((*p == '\n') || (*p == '\r')) p++;
    *buff = p;
    return(ret);
}



void create_folder(u8 *folder) {
    u8      tmp[256];

    printf("- create folder: %s\n", folder);
    make_dir(folder);
    if(chdir(folder) < 0) std_err();
    printf("- folder: %s\n", getcwd(tmp, sizeof(tmp)));
}



u8 *fdload(u8 *fname) {
    struct stat xstat;
    FILE    *fd;
    int     size;
    u8      *buff;

    printf("- open file %s\n", fname);
    fd = fopen(fname, "rb");
    if(!fd) std_err();

    fstat(fileno(fd), &xstat);
    size = xstat.st_size;

    buff = malloc(size + 1);
    if(!buff) std_err();
    fread(buff, 1, size, fd);
    fclose(fd);
    buff[size] = 0;     // delimiter

    return(buff);
}



void jmeetrec(ids_t *ids) {
    mydown_options  opt;
    time_t  refresh_time,
            onesec_time;
    u32     crc,
            frame,
            idsnum;
    int     sd,
            len,
            code,
            waitsec;
    u8      wwwdata[256],
            password[64],
            ticket_id[17],
            *ticket_data,
            *buff;

    printf("\n"
        "frames:    . ok    x duplicated    ? timeout_or_other_network_problems         \n"
        "-------------------------------------------------------------------------------\n");

    srand(time(NULL));
    frame       = 0;
    idsnum      = 0;
    sd          = 0;

    memset(&opt, 0, sizeof(opt));
    opt.verbose     = -1;
    opt.useragent   = "jsyntax";
    opt.more_http   = wwwdata;
    opt.filedata    = &buff;
    opt.keep_alive  = &sd;
    opt.timeout     = 5,
    opt.ret_code    = &code;
    opt.onflyunzip  = 1;

    refresh_time = time(NULL) + REFRESH_LIST;
    onesec_time  = time(NULL) + 1;

    for(;;) {
        for(;; idsnum++) {
            if(!record) {
                if(time(NULL) > refresh_time) goto quit;
            }

            if(!ids[idsnum].name) {
                idsnum = 0;
                if((time(NULL) - onesec_time) < 1) sleep(ONESEC);
                onesec_time = time(NULL) + 1;
            }
            if(ids[idsnum].disabled) continue;

            ticket_data = create_ticket(ticket_id);

            sprintf(wwwdata,
                "Ticket-Data: %s\r\n"
                "Content-Length: 0\r\n"     // remove???
                "Ticket-Id: %s\r\n"
                "%s%s",
                ticket_data,
                ticket_id,
                ids[idsnum].password ? PWD : "", ids[idsnum].password ? (char *)ids[idsnum].password : "");

            len = mydown(ids[idsnum].url, NULL, &opt);
            if(len == MYDOWN_ERROR) {
                if(code == 403) {
                    if(!askpass) {
                        printf("- %s temporary disabled (needs password but you choosed the 'o' filter)\n", ids[idsnum].name);
                        ids[idsnum].disabled = 1;
                        continue;
                    }
                    printf(
                        "- the webcam %s is protected with a password\n"
                        "  insert it or press RETURN for %s:\n  ",
                        ids[idsnum].name,
                        record ? "waiting if it returns free" : "skipping this webcam the next times");
                    fgetz(password, sizeof(password));
                    ids[idsnum].password = strdup(password);
                    if(ids[idsnum].password[0]) {
                        idsnum--;
                        continue;
                    } else {
                        ids[idsnum].disabled = 1;
                    }
                }

                if((code == 404) || (code == 403)) {
                    if(record) {
                        printf("- webcam temporary offline or protected, wait mode... ");
                        for(waitsec = 15; waitsec; waitsec--) {
                            printf("%-3d\b\b\b", waitsec);
                            sleep(ONESEC);
                        }
                        printf("\n");
                    } else {
                        printf("- %s temporary disabled\n", ids[idsnum].name);
                        ids[idsnum].disabled = 1;
                    }
                } else {
                    // timeout webcams are rescanned later
                    fputc('?', stdout);
                }
                continue;
            }
            if(!len) {
                idsnum--;
                sd = 0;
                continue;
            }

            crc = mycrc32(buff, len);
            if(crc == ids[idsnum].crc) {
                fputc('x', stdout);
            } else {
                fputc('.', stdout);
                ids[idsnum].crc = crc;
                save_file(buff, len, record ? NULL : ids[idsnum].name, frame);
                frame++;
            }

            myfree((void *)&buff);
        }

        printf("\n- connection lost or timed out\n");
    }

quit:
    if(sd) close(sd);
    myfree((void *)&buff);
}




int rnd_mem(u8 *data, int len, u32 rnd) {
    u8      *p;
    static const u8 table[] =
                "abcdefghijklmnopqrstuvwxyz";

    for(p = data; len; len--) {
        rnd = (rnd * 0x343FD) + 0x269EC3;
        *p++ = table[rnd % (sizeof(table) - 1)];
    }
    *p++ = 0;

    return(p - data);
}



int save_file(u8 *data, int len, u8 *id, u32 frame) {
    FILE    *fd;
    u8      fname[128];

    if(id) {
        sprintf(fname, "%.*s.jpg", sizeof(fname) - 5, id);
    } else {
        sprintf(fname, "%08x.jpg", frame);
        write_watch_now(fname);
    }
    fd = fopen(fname, "wb");
    if(!fd) std_err();
    fwrite(data, len, 1, fd);
    fclose(fd);
    return(0);
}



void write_watch_ever(void) {
    FILE    *fd;

    fd = fopen(WATCH_EVER, "wb");
    if(!fd) std_err();
    printf(
        "- Note: if you use Firefox remember to set to FALSE the parameter\n"
        "  dom.disable_image_src_set available in about:config or "WATCH_EVER"\n"
        "  will not work\n");

    fprintf(fd,
        "<html>\n"
        "<title>JmeetREC</title>\n"
        "<head>\n"
        "<style type=\"text/css\"><!--*{margin:0;padding:0;}--></style>\n"
        "<script language=\"JavaScript\">\n"
        "<!--\n"
        "    var i = 0;\n"
        "    setTimeout(\"loadimg()\", 1);\n"
        "\n"
        "    function d2h(d) {\n"
        "        var hex = \"0123456789abcdef\";\n"
        "        var h = \"\";\n"
        "        var j;\n"
        "        for(j = 0; j < 8; j++) {\n"
        "            h = hex.substr(d & 15, 1) + h;\n"
        "            d >>= 4;\n"
        "        }\n"
        "        return h;\n"
        "    }\n"
        "\n"
        "    function loadimg() {\n"
        "        document.myimg.src = d2h(i) + \".jpg\";\n"
        "        i++;\n"
        "        setTimeout(\"loadimg()\", 300);\n"
        "    }\n"
        "-->\n"
        "</script>\n"
        "</head><body>\n"
        "<img name=\"myimg\" src=\"00000000.jpg\" width=100%% border=0>\n"
        "</body></html>\n");

    fclose(fd);
}



void write_watch_now(u8 *fname) {
    FILE    *fd;

    fd = fopen(WATCH_NOW, "wb");
    if(!fd) return;

    fprintf(fd,
        "<html>"
        "<title>JMeetREC</title>"
        "<head>"
        "<style type=\"text/css\"><!--*{margin:0;padding:0;}--></style>"
        "<meta http-equiv=\"Refresh\" Content=\"1\">"
        "</head>"
        "<body>"
        "<img src=\"%s\" width=100%%>\n"
        "</body>"
        "</html>",
        fname);

    fclose(fd);
}



u32 mysqrt(u32 num) {
     u32    ret    = 0,
            ret_sq = 0,
            b;
     int    s;

     for(s = (32 >> 1) - 1; s >= 0; s--) {
         b = ret_sq + ((u32)1 << (s << 1)) + ((ret << s) << 1);
         if(b <= num) {
             ret_sq = b;
             ret += (u32)1 << s;
         }
     }

     return(ret);
}



void write_watch_thumb(ids_t *ids) {
    FILE    *fd;
    int     idsnum,
            horiz;

    fd = fopen(WATCH_THUMB, "wb");
    if(!fd) std_err();

    fprintf(fd,
        "<html>"
        "<title>JmeetREC</title>"
        "<head><meta http-equiv=\"Refresh\" Content=\"1\"></head>"
        "<style type=\"text/css\"><!--*{margin:0;padding:0;}--></style>\n"
        "<body>");

    for(idsnum = 0; ids[idsnum].name; idsnum++);
    horiz = 100 / mysqrt(idsnum);

    for(idsnum = 0; ids[idsnum].name; idsnum++) {
        fprintf(fd,
            "<a href=\"%s.jpg\"><img src=\"%s.jpg\" width=%d%% border=0></a>",
            ids[idsnum].name, ids[idsnum].name, horiz);
    }

    fprintf(fd, "</body></html>");
    fclose(fd);
}



void show_help(void) {
    printf(
        "Options:\n"
        "-r ID    record the webcam identified by ID (it's the same of specifying the\n"
        "         webcam_ID as last argument)\n"
        "-f FLT   show the webcams in the FLT filter:\n"
        "         f=female, m=male, c=couple, u=unknown, o=open, p=private, nicknames\n"
        "         example for female and couple and hot or ita in the nick:-f fc,hot,ita\n"
        "         example for the webcams with hot or ita in the nick:     -f ,hot,ita\n"
        "         when you choose the 'o' filter the tool will never ask you to insert\n"
        "         password if the webcam needs it, will simply skip the webcam\n"
        "-c FILE  load the local list of webcams from FILE\n"
        "-l       list the webcams and exit\n"
        "-i       automatically launch the browser for watching the webcams\n"
        "\n"
        "Note: the webcam is recorded ONLY with the -r option or with ID argument!\n"
        "      if you see many 403 errors even using the 'o' filter check your clock!\n"
        "\n");
    exit(1);
}



void myfree(u8 **buff) {
    if(!*buff) return;
    free(*buff);
    *buff = NULL;
}



int fgetz(u8 *data, int datalen) {
    u8     *p;

    fgets(data, datalen, stdin);
    for(p = data; *p && (*p != '\n') && (*p != '\r'); p++);
    *p = 0;
    return(p - data);
}



void std_err(void) {
    perror("\nError");
    exit(1);
}


