/*

    Copyright 2004,2005,2006 Luigi Auriemma

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    http://www.gnu.org/licenses/gpl.txt

*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "blowfish.h"

#ifdef WIN32
    #include <io.h>
    #define ftruncate   chsize
    #define u_char      unsigned char
#else
    #include <unistd.h>
    #define strnicmp    strncasecmp
#endif



#define VER                 "0.2.1"
#define BUFFSZ              32768
#define SIGN                "Lineage2Ver"
#define SIGNSIZE            28
#define LIN2KEY212          "[;'.]94-&@%!^+]-31=="
#define LIN2KEY211          "31==-%&@!^+][;'.]94-"
#define FINAL               "\x00\x00\x00\x00" \
                            "\x01\x00\x00\x00" \
                            "\x01\x00\x00\x00" \
                            "\xFF\xFF\xFF\xFF" \
                            "\x00\x00\x00\x00"

#define ENCBYTESWITCH(filename, errmsg) \
    switch(linver) { \
        case 111: encbyte = 0xAC; break; \
        case 121: encbyte = lin2encbyte(filename); break; \
        case 211: { \
            lin2keyptr = LIN2KEY211; \
            encbyte = 0; \
            } break; \
        case 212: { \
            lin2keyptr = LIN2KEY212; \
            encbyte = 0; \
            } break; \
        default: { \
           printf("\n" \
                "Error: "errmsg"\n", linver); \
           exit(1); \
        } \
    }

#define LIN2ENCDEC(blf_op)  \
    if(encbyte) { \
        /* XOR */ \
        while(1) { \
            len = fread(buff, 1, BUFFSZ, fd); \
            if(!len) break; \
            \
            xordata(buff, len, encbyte); \
            if(fwrite(buff, len, 1, fdout) != 1) std_err(2); \
        } \
    } else { \
        /* BLOWFISH */ \
        InitializeBlowfish(&bfx, lin2keyptr, 21); \
        \
        while(1) { \
            len = fread(buff, 1, BUFFSZ, fd); \
            if(!len) break; \
            \
            blf_op(&bfx, (unsigned int *)buff, len >> 3); \
            if(fwrite(buff, len, 1, fdout) != 1) std_err(2); \
        } \
    }



void lin2decode(FILE *fd, FILE *fdout, char *input);
void lin2encode(FILE *fd, FILE *fdout, char *output, char *arg4);
void xordata(u_char *data, int size, int encbyte);
int lin2encbyte(u_char *fname);
void unicode2char(u_char *data, int size);
void char2unicode(u_char *in, u_char *out, int size);
void std_err(int err);



int main(int argc, char *argv[]) {
    FILE    *fd,
            *fdout;
    char    *input,
            *output;
    int     oper,
            risp;


    setbuf(stdout, NULL);

    fputs("\n"
        "Lineage II files decoder/encoder "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", stdout);

    if(argc < 4) {
        printf("\nUsage: %s <d/e> <file> <output> [format]\n"
            "\n"
            "Examples:\n"
            " to decode: %s d User.ini out.txt\n"
            " to encode: %s e out.txt User.ini 212\n"
            "\n"
            "Important notes:\n"
            "- the fourth argument in the encoding is the format you want to use to\n"
            "  encode your file, it is shown when you decode the original\n"
            "- for the encoding you must use the SAME output filename of the original\n"
            "  because in some cases the algorithm is calculated on the filename\n"
            "- the final part of the Lineage II encoded files is not used in the current\n"
            "  version of the game tested by me so I use a fixed one and works\n"
            "- the structure of the encoded files is not totally clear however those\n"
            "  created by this program seems to work perfectly\n"
            "- contact me if you have problems, informations or suggestions\n"
            "\n", argv[0], argv[0], argv[0]);
        exit(1);
    }

    input  = argv[2];
    output = argv[3];
    oper   = argv[1][0];
    if((oper != 'e') && (oper != 'd')) {
        fputs("\nError: you must choose 'e' to create a Lineage 2 encoded file or 'd' to decode an existent one\n",
            stdout);
        exit(1);
    }

    printf("Input:           %s\n", input);
    fd = fopen(input, "rb");
    if(!fd) std_err(0);

    printf("Output:          %s\n", output);
    fdout = fopen(output, "rb");
    if(fdout) {
        fclose(fdout);
        fputs("\nA file with the same name already exists, do you want to overwrite it? (y/n)\n", stdout);
        fflush(stdin);
        risp = fgetc(stdin);
        if((risp != 'y') && (risp != 'Y')) exit(1);
    }
    fdout = fopen(output, "wb");
    if(!fdout) std_err(0);

    if(oper == 'd') {
        fputs("Operation:       decoding\n", stdout);
        lin2decode(fd, fdout, input);
    } else {
        fputs("Operation:       encoding\n", stdout);
        if(argc < 5) {
            fputs("\n"
                "Error: you must specify the format version.\n"
                "       It is the value shown when you decode a Lineage 2 file.\n"
                "       This value is visible also at the beginning of the original encoded file\n"
                "       and you can see it with a text or hex editor (\"Lineage2Ver\" NUMBER)\n"
                "\n", stdout);
            exit(1);
        }
        lin2encode(fd, fdout, output, argv[4]);
    }

    fclose(fd);
    fclose(fdout);
    fputs("\nFinished\n\n", stdout);
    return(0);
}



void lin2decode(FILE *fd, FILE *fdout, char *input) {
    int     filesize;
    int     encbyte,
            len,
            linver;
    u_char  *buff,
            *lin2keyptr = 0;
    blf_ctx bfx;


    buff = malloc(BUFFSZ);
    if(!buff) std_err(0);

    if(fread(buff, SIGNSIZE, 1, fd) != 1) std_err(1);
    unicode2char(buff, SIGNSIZE);

    if(strnicmp(buff, SIGN, sizeof(SIGN) - 1)) {
        fputs("\nError: the file is not a Lineage2 encoded file\n", stdout);
        exit(1);
    }

    linver = atoi(buff + sizeof(SIGN) - 1);
    ENCBYTESWITCH(
        input,
        "the file is encoded with an unknown file format version (%d)\n"
        "       Contact me\n");

    printf("Format version:  %d   (remember it, you must use it to encode!)\n", linver);
    printf("Decoding byte:   0x%02x\n", encbyte);
    if(!encbyte) printf("Blowfish key:    %s\n", lin2keyptr);

    LIN2ENCDEC(blf_dec);
    fflush(fdout);

    filesize = ftell(fdout) - 20;
    if(ftruncate(fileno(fdout), filesize) < 0) std_err(0);
    printf("Output filesize: %u\n", filesize);
}



void lin2encode(FILE *fd, FILE *fdout, char *output, char *arg4) {
    int     encbyte,
            linver,
            len;
    u_char  *buff,
            *lin2keyptr = 0;
    blf_ctx bfx;


    buff = malloc(BUFFSZ);
    if(!buff) std_err(0);

    linver = atoi(arg4);
    printf("Format version:  %d\n", linver);

    ENCBYTESWITCH(
        output,
        "choose an existent file format, currently %d is not supported\n");
    printf("Encoding byte:   0x%02x\n", encbyte);
    if(!encbyte) printf("Blowfish key:    %s\n", lin2keyptr);

    len = sizeof(SIGN) - 1;
    char2unicode(SIGN, buff, len);
    if(fwrite(buff, len << 1, 1, fdout) != 1) std_err(2);

    char2unicode(arg4, buff, 3);
    if(fwrite(buff, 6, 1, fdout) != 1) std_err(2);

    LIN2ENCDEC(blf_enc);

    if(fwrite(FINAL, sizeof(FINAL) - 1, 1, fdout) != 1) std_err(2);
    fflush(fdout);
    printf("Output filesize: %u\n", ftell(fdout));
}



void xordata(u_char *data, int size, int encbyte) {
    while(size--) {
        *data++ ^= encbyte;
    }
}



int lin2encbyte(u_char *fname) {
    int     len,
            result = 0;

    len = strlen(fname);
    fname = fname + len - 1;
    while(len--) {
        if((*fname == '\\') || (*fname == '/')) break;
        if((*fname >= 'A') && (*fname <= 'Z')) *fname += 32;
        result += *fname;
        fname--;
    }
    return(result & 0xff);
}



void unicode2char(u_char *data, int size) {
    u_char  *out = data;

    size >>= 1;
    while(size--) {
        *out++ = *data++;
        data++;
    }
    *out = 0;
}



void char2unicode(u_char *in, u_char *out, int size) {
    while(size--) {
        *out++ = *in++;
        *out++ = 0;
    }
}



void std_err(int err) {
    switch(err) {
        case 1: {
            fputs("\nError: file seems corrupted or incomplete\n", stdout);
            } break;
        case 2: {
            fputs("\nError: write problem, probably you have finished disk space or you cannot write on the output file\n", stdout);
            } break;
        default: {
            perror("\nError");
            } break;
    }
    exit(1);
}


