/*
    Copyright 2008,2009,2010 Luigi Auriemma

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    http://www.gnu.org/licenses/gpl-2.0.txt
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <ctype.h>
#include <sys/stat.h>
#include <zlib.h>

#ifdef WIN32
    #include <windows.h>
    #include <direct.h>
    #define PATHSLASH   '\\'
    #define make_dir(x) mkdir(x)
#else
    #define stricmp     strcasecmp
    #define PATHSLASH   '/'
    #define make_dir(x) mkdir(x, 0755)
#endif

typedef uint8_t     u8;
typedef uint16_t    u16;
typedef uint32_t    u32;



#define VER         "0.1"
#define MAX_CHUNK   0x10000



int hex2byte(u8 *hex);
int unhex(u8 *in, int insz, u8 *out, int outsz);
int molebox_dump(FILE *fd, u8 *fname, u32 offset, u32 fsize);
int molebox_extract(FILE *fd);
void molebox_setkey(u8 *data, u8 *key);
void molebox_decrypt(u8 *key, u8 *data, int datasz);
u8 *create_dir(u8 *name);
int check_wildcard(u8 *fname, u8 *wildcard);
void myalloc(u8 **data, u32 wantsize, u32 *currsize);
int check_overwrite(u8 *fname);
u32 getxx(FILE *fd, int num_size);
void myfr(FILE *fd, void *data, u32 size);
void std_err(void);



int     listonly    = 0,
        overwrite   = 0;
u8      *name_filter= NULL;

u8      molebox_key[16],
        molebox_ctx[104];   // the ctx is 104 bytes in this version of Molebox



int main(int argc, char *argv[]) {
    FILE    *fd;
    int     i,
            files;
    u8      *fname,
            *fdir,
            *key;

    setbuf(stdout, NULL);

    fputs("\n"
        "Molebox2 files extractor "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", stdout);

    if(argc < 4) {
        printf("\n"
            "Usage: %s [options] <file> <output_folder> <hex_key>\n"
            "\n"
            "Options:\n"
            "-l      list files without extracting them\n"
            "-f \"W\"  handle only the files which match the wildcard W like \"*.ogg\"\n"
            "-o      overwrite existent files without asking\n"
            "\n"
            "hex_key is the 16 bytes hexadecimal key that is located in the dumped process\n"
            "of the game/software near the \".BOX\" signature, for example in the following:\n"
            "  00 00 00 00 ca 9d 35 c1 5c 7c cc b6 dd 49 90 e4   ......5.\\|...I..\n"
            "  e2 fd 35 cf 61 71 75 61 72 69 61 2e 42 4f 58 00   ..5.aquaria.BOX.\n"
            "the key will be ca9d35c15c7cccb6dd4990e4e2fd35cf for the game Aquaria\n"
            "or 784C46AA6E1A7C677E51DC8BF5A82293 for Kingdom Elemental\n"
            "contact me if you have doubts or proeblems\n"
            "\n", argv[0]);
        exit(1);
    }

    argc -= 3;
    for(i = 1; i < argc; i++) {
        if(((argv[i][0] != '-') && (argv[i][0] != '/')) || (strlen(argv[i]) != 2)) {
            printf("\nError: wrong argument (%s)\n", argv[i]);
            exit(1);
        }
        switch(argv[i][1]) {
            case 'l': listonly    = 1;          break;
            case 'f': name_filter = argv[++i];  break;
            case 'o': overwrite   = 1;          break;
            default: {
                printf("\nError: wrong argument (%s)\n", argv[i]);
                exit(1);
            }
        }
    }
    fname = argv[argc];
    fdir  = argv[argc + 1];
    key   = argv[argc + 2];

    printf("- open %s\n", fname);
    fd = fopen(fname, "rb");
    if(!fd) std_err();

    if(!listonly) {
        printf("- set output folder %s\n", fdir);
        if(chdir(fdir) < 0) std_err();
    }

    printf("- start %s:\n", listonly ? "listing" : "extraction");

    if(unhex(key, strlen(key), molebox_key, sizeof(molebox_key)) != sizeof(molebox_key)) {
        printf("\nError: the key must be 16 bytes long\n");
        exit(1);
    }
    molebox_setkey(molebox_key, molebox_ctx);

    files = molebox_extract(fd);

    fclose(fd);
    printf("\n- %s %d files\n", listonly ? "listed" : "extracted", files);
    return(0);
}



int hex2byte(u8 *hex) {
    static const signed char hextable[256] =
        "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff"
        "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff"
        "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff"
        "\x00\x01\x02\x03\x04\x05\x06\x07\x08\x09\xff\xff\xff\xff\xff\xff"
        "\xff\x0a\x0b\x0c\x0d\x0e\x0f\xff\xff\xff\xff\xff\xff\xff\xff\xff"
        "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff"
        "\xff\x0a\x0b\x0c\x0d\x0e\x0f\xff\xff\xff\xff\xff\xff\xff\xff\xff"
        "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff"
        "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff"
        "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff"
        "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff"
        "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff"
        "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff"
        "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff"
        "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff"
        "\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff\xff";

    if((hextable[hex[0]] < 0) || (hextable[hex[1]] < 0)) return(-1);
    return((hextable[hex[0]] << 4) | hextable[hex[1]]);
}

// automatically does: hexadecimal, quoted printable, percentage encoding
int unhex(u8 *in, int insz, u8 *out, int outsz) {
    unsigned char   *inl    = in + insz,
                    *o      = out,
                    *outl   = out + outsz;
    int     c;

    while(in < inl) {
        c = hex2byte(in);
        if(c < 0) {
            in++;
        } else {
            if(o >= outl) return(-1);
            *o++ = c;
            in += 2;
        }
    }
    return(o - out);
}



int molebox_dump(FILE *fd, u8 *fname, u32 offset, u32 fsize) {
    static int  insz    = 0,
                outsz   = 0,
                chunksz = 0;
    static u8   *in     = NULL,
                *out    = NULL;
    static u32  *chunk  = NULL;
    FILE        *fdo;
    uLongf      destlen;
    int         i,
                dozip,
                chunks;

    if(check_wildcard(fname, name_filter) < 0) return(0);
    printf("  %08x %-10u %s\n", offset, fsize, fname);
    if(listonly) return(1);

    if(!overwrite && (check_overwrite(fname) < 0)) return(0);
    create_dir(fname);
    fdo = fopen(fname, "wb");
    if(!fdo) std_err();

    if(fseek(fd, offset, SEEK_SET)) std_err();

    chunks = fsize / MAX_CHUNK;
    if(fsize % MAX_CHUNK) chunks++;

    myalloc(&out, MAX_CHUNK, &outsz);

    if(fseek(fd, -(chunks * sizeof(u32)), SEEK_CUR)) std_err();
    myalloc((void *)&chunk, chunks * sizeof(u32), &chunksz);
    for(i = 0; i < chunks; i++) {
        chunk[i] = getxx(fd, 4);
    }

    for(i = 0; i < chunks; i++) {
        if(chunk[i] & 0x80000000) {
            chunk[i] &= 0x7fffffff;
            dozip = 0;
        } else {
            dozip = 1;
        }
        myalloc(&in, chunk[i], &insz);
        myfr(fd, in, chunk[i]);
        molebox_decrypt(molebox_ctx, in, chunk[i]);

        if(dozip) {
            destlen = outsz;
            if(uncompress(out, &destlen, in, chunk[i]) != Z_OK) {
                printf("\nError: invalid compressed data at offset %08x\n", (u32)ftell(fd));
                exit(1);
            }
            fwrite(out, 1, destlen, fdo);
        } else {
            fwrite(in, 1, chunk[i], fdo);
        }
    }

    fclose(fdo);
    return(1);
}



int molebox_extract(FILE *fd) {
    typedef struct {
        u32     namefile_zlength;
        u32     namefile_length;
        u32     namefile_offset;
        u32     names_length;
        u32     files;
        u32     total_pak_size;
    } molebox_head_t;
    typedef struct {
        u32     name_offset;
        u32     offset;
        u32     file_size;
        u32     three;      // type??? it's ever '3'
    } molebox_file_t;

    molebox_head_t  molebox_head;
    molebox_file_t  *molebox_file;
    u32     sign;
    int     i,
            tot;
    u8      hash[16],   // it's more something like a signature
            *names,
            *namefile;

    if(fseek(fd, -4, SEEK_END)) std_err();
    sign = getxx(fd, 4);
    if(sign != 0xcafebabe) sign ^= 0xcafebabe ^ (u32)ftell(fd);
    if(sign != 0x584f424d) {    // MBOX
        printf("\nError: invalid signature (%08x), it's different than \"MBOX\"\n", sign);
        exit(1);
    }

    if(fseek(fd, -20, SEEK_END)) std_err();
    myfr(fd, hash, 16);

    if(fseek(fd, -44, SEEK_END)) std_err();
    myfr(fd, &molebox_head, sizeof(molebox_head_t));

    printf("\n"
        "  description length  %08x\n"
        "  description length  %08x\n"
        "  description offset  %08x\n"
        "  files               %u\n"
        "  archive size        %u\n",
        molebox_head.namefile_zlength,
        molebox_head.namefile_length,
        molebox_head.namefile_offset,
        molebox_head.files,
        molebox_head.total_pak_size);

    namefile = malloc(molebox_head.namefile_length);
    if(!namefile) std_err();
    if(fseek(fd, molebox_head.namefile_offset, SEEK_SET)) std_err();
    myfr(fd, namefile, molebox_head.namefile_length);
    molebox_decrypt(molebox_ctx, namefile, molebox_head.namefile_length);

    names        = (u8 *)namefile;
    molebox_file = (molebox_file_t *)(namefile + molebox_head.names_length);

    printf("\n"
        "  offset   size       filename\n"
        "------------------------------\n");

    tot = 0;
    for(i = 0; i < molebox_head.files; i++) {
        if(molebox_file[i].name_offset > molebox_head.namefile_length) {
            printf("\nError: seems that the master key is wrong because the filenames are too big\n");
            exit(1);
        }

        tot += molebox_dump(fd,
            names + molebox_file[i].name_offset,
            molebox_file[i].offset,
            molebox_file[i].file_size);
    }
    return(tot);
}



// anti DEP limitation! if you apply VirtualAlloc on a static char
// it will cover also the rest of the page included other variables!
void *antidep_alloc(u8 *dump, int dumpsz) {
    int     pagesz;
    void    *ret;

    pagesz = (dumpsz + 4095) & (~4095); // useful for pages? mah

#ifdef WIN32
    ret = VirtualAlloc(
        NULL,
        pagesz,
        MEM_COMMIT | MEM_RESERVE,
        PAGE_EXECUTE_READWRITE);    // write for memcpy
#else
    ret = malloc(pagesz);
    mprotect(
        ret,
        pagesz,
        PROT_EXEC | PROT_WRITE);    // write for memcpy
#endif
    memcpy(ret, dump, dumpsz);
    return(ret);
}



void molebox_setkey(u8 *data, u8 *key) {   // I don't know what version of Molebox was used, anyway the code it's all at 16 bit
    static __stdcall void (*molebox_setkey_func)(u8 *, const u8 *) = NULL;
    static unsigned char  dump[] =
    "\x55\x8b\xec\x83\xec\x78\x53\x56\x8b\x75\x08\x57\x6a\x02\x89\x4d\xfc\x66\x8b\x06\x5b\x50\x03\xf3\xe8\x2d\x01\x00\x00\x66\x8b\x3e"
    "\x03\xf3\x89\x45\xf8\x8b\x4d\xfc\x66\x8b\x06\x03\xf3\x66\xf7\xd8\x89\x45\x08\x66\x8b\x06\x50\x03\xf3\x66\xf7\xdf\xe8\x09\x01\x00"
    "\x00\x66\x89\x45\xee\x66\x8b\x45\x08\x66\x89\x45\xec\x66\x8b\x45\xf8\x66\x89\x7d\xea\x8d\x7d\xe8\x66\x89\x45\xe8\xc7\x45\xf0\x07"
    "\x00\x00\x00\x66\x8b\x06\x66\x8b\x0c\x1e\x03\xf3\x2b\xfb\x03\xf3\x66\x89\x0f\x8b\x4d\xfc\x2b\xfb\x66\x89\x07\x66\x8b\x06\x50\x03"
    "\xf3\xe8\xc4\x00\x00\x00\x89\x45\xf8\x66\x8b\x06\x66\xf7\xd8\x03\xf3\x89\x45\xf4\x8b\x4d\xfc\x2b\xfb\x66\x8b\x06\x03\xf3\x66\xf7"
    "\xd8\x89\x45\x08\x66\x8b\x06\x50\x03\xf3\xe8\x9b\x00\x00\x00\x66\x89\x07\x66\x8b\x45\xf4\x2b\xfb\x66\x89\x07\x66\x8b\x45\x08\x2b"
    "\xfb\x66\x89\x07\x66\x8b\x45\xf8\x2b\xfb\xff\x4d\xf0\x66\x89\x07\x75\x91\x66\x8b\x06\x66\x8b\x0c\x1e\x03\xf3\x2b\xfb\x03\xf3\x66"
    "\x89\x0f\x8b\x4d\xfc\x2b\xfb\x66\x89\x07\x66\x8b\x06\x50\x03\xf3\xe8\x55\x00\x00\x00\x89\x45\xf8\x66\x8b\x06\x8b\x4d\xfc\x03\xf3"
    "\x66\xf7\xd8\x89\x45\xf4\x66\x8b\x06\x66\x8b\x76\x02\x2b\xfb\x66\xf7\xd8\x56\x89\x45\x08\xe8\x2f\x00\x00\x00\x66\x89\x07\x66\x8b"
    "\x45\x08\x2b\xfb\x6a\x1a\x59\x8d\x75\x88\x66\x89\x07\x66\x8b\x45\xf4\x2b\xfb\x66\x89\x07\x66\x8b\x45\xf8\x66\x89\x47\xfe\x8b\x7d"
    "\x0c\xf3\xa5\x5f\x5e\x5b\xc9\xc2\x08\x00\x55\x8b\xec\x51\x8b\x45\x08\x53\x56\x57\x6a\x01\x5b\x66\x3b\xc3\x76\x74\xb9\x01\x00\x01"
    "\x00\x0f\xb7\xf8\x8b\xc1\x99\xf7\xff\x89\x45\x08\x8b\xc1\x99\xf7\xff\x66\x3b\xd3\x74\x55\x8b\xc7\x0f\xb7\xca\x99\xf7\xf9\x8b\xf0"
    "\x8b\xc7\x0f\xaf\x75\x08\x99\xf7\xf9\x46\x89\x75\xfc\x66\x3b\xd3\x74\x33\x8b\xc1\x0f\xb7\xf2\x99\xf7\xfe\x8b\xf8\x8b\xc1\x99\x0f"
    "\xaf\x7d\xfc\xf7\xfe\x01\x7d\x08\x66\x3b\xd3\x74\x1e\x8b\xc6\x0f\xb7\xca\x99\xf7\xf9\x8b\xf8\x8b\xc6\x0f\xaf\x7d\x08\x99\xf7\xf9"
    "\x01\x7d\xfc\xeb\xc8\x66\x8b\x45\xfc\xeb\x05\x8b\xc3\x2b\x45\x08\x5f\x5e\x5b\xc9\xc2\x04\x00";
    int     i;
    u16     *in,
            *out;
    u8      tmp[sizeof(molebox_ctx)];

    if(!molebox_setkey_func) {
        molebox_setkey_func = antidep_alloc(dump, sizeof(dump));
    }

    out = (u16 *)tmp;
    in  = (u16 *)data;
    for(i = 0; i < 8; i++) {
        out[i] = (in[i] >> 8) | (in[i] << 8);
    }
    for(; i < (sizeof(molebox_ctx)/2); i++) {
        if(!(i & 7)) in = (out + i) - 8;
        out[i] = (in[(i + 1) & 7] << 9) | (in[(i + 2) & 7] >> 7);
    }
    molebox_setkey_func(tmp, key);
}



void molebox_decrypt(u8 *key, u8 *data, int datasz) {   // I don't know what version of Molebox was used, anyway the code it's all at 16 bit
    static __stdcall void (*molebox_decrypt_func)(u8 *, u8 *, const u8 *) = NULL;
    static unsigned char  dump[] =
    "\x55\x8b\xec\x83\xec\x0c\x8b\x45\x08\x53\x56\x57\x66\x8b\x08\x66\x8b\x50\x02\x83\xc0\x02\xc7\x45\xfc\x08\x00\x00\x00\x40\x40\x66"
    "\x8b\x18\x66\x8b\x40\x02\x66\x89\x45\x08\x33\xc0\x8a\xc5\x8a\xe1\x8b\xc8\x33\xc0\x8a\xc6\x8a\xe2\x8b\xd0\x33\xc0\x8a\xc7\x8a\xe3"
    "\x8b\xf8\x33\xc0\x8a\x45\x09\x8a\x65\x08\x89\x45\x08\x8b\x45\x10\xeb\x03\x8b\x4d\x10\x66\x8b\x30\x40\x40\x66\x85\xf6\x74\x2d\x66"
    "\x85\xc9\x74\x1e\x0f\xb7\xf6\x0f\xb7\xc9\x0f\xaf\xf1\x8b\xce\xc1\xe9\x10\x66\x3b\xf1\x1b\xdb\xf7\xdb\x2b\xd9\x03\xde\x89\x5d\x10"
    "\xeb\x12\x6a\x01\x59\x2b\xce\x89\x4d\x10\xeb\x08\x6a\x01\x5e\x2b\xf1\x89\x75\x10\x66\x8b\x08\x03\xd1\x40\x40\x66\x8b\x08\x03\xf9"
    "\x40\x40\x66\x8b\x08\x40\x40\x66\x85\xc9\x74\x30\x66\x83\x7d\x08\x00\x74\x1f\x0f\xb7\xf1\x0f\xb7\x4d\x08\x0f\xaf\xf1\x8b\xce\xc1"
    "\xe9\x10\x66\x3b\xf1\x1b\xdb\xf7\xdb\x2b\xd9\x03\xde\x89\x5d\x08\xeb\x13\x6a\x01\x5e\x2b\xf1\x89\x75\x08\xeb\x09\x6a\x01\x59\x2b"
    "\x4d\x08\x89\x4d\x08\x66\x8b\x08\x89\x7d\xf8\x33\x7d\x10\x40\x40\x66\x85\xc9\x74\x27\x66\x85\xff\x74\x1b\x0f\xb7\xf1\x0f\xb7\xcf"
    "\x0f\xaf\xf1\x8b\xce\xc1\xe9\x10\x66\x3b\xf1\x1b\xff\xf7\xdf\x2b\xf9\x03\xfe\xeb\x0e\x6a\x01\x5f\x2b\xf9\xeb\x07\x6a\x01\x59\x2b"
    "\xcf\x8b\xf9\x8b\x75\x08\x66\x8b\x08\x33\xf2\x89\x55\xf4\x03\xf7\x40\x40\x66\x85\xc9\x74\x27\x66\x85\xf6\x74\x1b\x0f\xb7\xd9\x0f"
    "\xb7\xce\x0f\xaf\xd9\x8b\xcb\xc1\xe9\x10\x66\x3b\xd9\x1b\xd2\xf7\xda\x2b\xd1\x03\xd3\xeb\x0c\x6a\x01\x5a\x2b\xd1\xeb\x05\x6a\x01"
    "\x5a\x2b\xd6\x31\x55\x10\x03\xfa\x31\x7d\x08\x33\x55\xf8\x33\x7d\xf4\xff\x4d\xfc\x0f\x85\xd8\xfe\xff\xff\x66\x8b\x08\x40\x40\x66"
    "\x85\xc9\x74\x2a\x66\x83\x7d\x10\x00\x74\x1c\x0f\xb7\xf1\x0f\xb7\x4d\x10\x0f\xaf\xf1\x8b\xce\xc1\xe9\x10\x66\x3b\xf1\x1b\xdb\xf7"
    "\xdb\x2b\xd9\x03\xde\xeb\x0d\x6a\x01\x5b\x2b\xd9\xeb\x06\x6a\x01\x5b\x2b\x5d\x10\x66\x8b\x08\x03\xf9\x40\x40\x89\x7d\x10\x66\x8b"
    "\x08\x66\x8b\x40\x02\x03\xd1\x66\x85\xc0\x74\x2a\x66\x83\x7d\x08\x00\x74\x1c\x0f\xb7\x4d\x08\x0f\xb7\xc0\x0f\xaf\xc1\x8b\xf0\xc1"
    "\xee\x10\x66\x3b\xc6\x1b\xc9\xf7\xd9\x2b\xce\x03\xc8\xeb\x0d\x6a\x01\x59\x2b\xc8\xeb\x06\x6a\x01\x59\x2b\x4d\x08\x8b\x75\x0c\x33"
    "\xc0\x8a\xc7\x5f\x8a\xe3\x66\x89\x06\x33\xc0\x8a\x45\x11\x83\xc6\x02\x8a\x65\x10\x66\x89\x06\x33\xc0\x46\x8a\xc6\x46\x8a\xe2\x66"
    "\x89\x06\x33\xc0\x8a\xc5\x8a\xe1\x66\x89\x46\x02\x5e\x5b\xc9\xc2\x0c\x00";
    int     rounds;

    if(!molebox_decrypt_func) {
        molebox_decrypt_func = antidep_alloc(dump, sizeof(dump));
    }

    for(rounds = datasz >> 3; rounds; rounds--) {
        molebox_decrypt_func(data, data, key);
        data += 8;
    }
}



u8 *create_dir(u8 *fname) {
    u8      *p,
            *l;

    p = strchr(fname, ':'); // unused
    if(p) {
        *p = '_';
        fname = p + 1;
    }
    for(p = fname; *p && strchr("\\/. \t:", *p); p++) *p = '_';
    fname = p;

    for(p = fname; ; p = l + 1) {
        for(l = p; *l && (*l != '\\') && (*l != '/'); l++);
        if(!*l) break;
        *l = 0;

        if(!strcmp(p, "..")) {
            p[0] = '_';
            p[1] = '_';
        }

        make_dir(fname);
        *l = PATHSLASH;
    }
    return(fname);
}



int check_wildcard(u8 *fname, u8 *wildcard) {
    u8      *f,
            *w,
            *a;

    if(!wildcard) return(0);
    f = fname;
    w = wildcard;
    a = NULL;
    while(*f || *w) {
        if(!*w && !a) return(-1);
        if(*w == '?') {
            if(!*f) break;
            w++;
            f++;
        } else if(*w == '*') {
            w++;
            a = w;
        } else {
            if(!*f) break;
            if(tolower(*f) != tolower(*w)) {
                if(!a) return(-1);
                f++;
                w = a;
            } else {
                f++;
                w++;
            }
        }
    }
    if(*f || *w) return(-1);
    return(0);
}



void myalloc(u8 **data, u32 wantsize, u32 *currsize) {
    if(wantsize <= *currsize) return;
    *data = realloc(*data, wantsize);
    if(!*data) std_err();
    *currsize = wantsize;
}



int check_overwrite(u8 *fname) {
    FILE    *fd;
    u8      ans[16];

    fd = fopen(fname, "rb");
    if(!fd) return(0);
    fclose(fd);
    printf("  the file already exists, do you want to overwrite it (y/N)? ");
    fgets(ans, sizeof(ans), stdin);
    if(tolower(ans[0]) != 'y') return(-1);
    return(0);
}



u32 getxx(FILE *fd, int num_size) {
    u32     num;
    int     i;
    u8      tmp[8];

    myfr(fd, tmp, num_size);

    num = 0;
    for(i = 0; i < num_size; i++) {
        num |= (tmp[i] << (i << 3));
    }
    return(num);
}



void myfr(FILE *fd, void *data, u32 size) {
    if(fread(data, 1, size, fd) != size) {
        printf("\nError: incomplete input file, can't read %u bytes\n", size);
        exit(1);
    }
}



void std_err(void) {
    perror("\nError");
    exit(1);
}


