/*
    Copyright 2008,2009 Luigi Auriemma

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    http://www.gnu.org/licenses/gpl-2.0.txt
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <ctype.h>
#include <sys/stat.h>

typedef uint8_t     u8;
typedef uint16_t    u16;
typedef uint32_t    u32;



#define VER         "0.1.1"



void motorm4x_decrypt(u8 *fname, u8 *data, int datasz);
void motorm4x_encrypt(u8 *fname, u8 *data, int datasz);
int check_overwrite(u8 *fname);
u8 *create_dir(u8 *name);
u8 *fdload(u8 *fname, int *fsize);
void fddump(u8 *fname, u8 *buff, int buffsz, int overwrite);
void std_err(void);



int main(int argc, char *argv[]) {
    int     i,
            buffsz,
            encode      = 0,
            overwrite   = 0;
    u8      *buff,
            *fin,
            *fout;

    setbuf(stdout, NULL);

    fputs("\n"
        "MotorM4x files decoder "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", stdout);

    if(argc < 3) {
        printf("\n"
            "Usage: %s [options] <unzipped_file> <output_file>\n"
            "\n"
            "Options:\n"
            "-e      re-encode the file\n"
            "-o      automatically overwrites the output file is already existent\n"
            "\n"
            "the input file must be one of the files included in the MDL/DTF archives that\n"
            "are just simple ZIP which can be opened and extracted with any normal program,\n"
            "so use this tool only with the unzipped files extracted from them.\n"
            "important: the name of the input filename MUST be the original one!!!\n"
            "\n", argv[0]);
        exit(1);
    }

    argc -= 2;
    for(i = 1; i < argc; i++) {
        if(((argv[i][0] != '-') && (argv[i][0] != '/')) || (strlen(argv[i]) != 2)) {
            printf("\nError: wrong argument (%s)\n", argv[i]);
            exit(1);
        }
        switch(argv[i][1]) {
            case 'e': encode    = 1;    break;
            case 'o': overwrite = 1;    break;
            default: {
                printf("\nError: wrong argument (%s)\n", argv[i]);
                exit(1);
            }
        }
    }
    fin  = argv[argc];
    fout = argv[argc + 1];

    buff = fdload(fin, &buffsz);

    if(encode) {
        motorm4x_encrypt(fin, buff, buffsz);
    } else {
        motorm4x_decrypt(fin, buff, buffsz);
    }

    fddump(fout, buff, buffsz, overwrite);

    printf("- %d bytes %s\n", buffsz, encode ? "encoded" : "decoded");
    free(buff);
    return(0);
}



void motorm4x_decrypt(u8 *fname, u8 *data, int datasz) {
    int     i,
            j,
            fnamelen;
    u8      *p;

    p = strrchr(fname, '\\');
    if(!p) p = strrchr(fname, '/');
    if(p) fname = p + 1;
    fnamelen = strlen(fname);
    j = 0;
    for(i = 0; i < datasz; i++) {
        if(!j) j = fnamelen;
        data[i] += (0xd6 - fname[--j]);
    }
}



void motorm4x_encrypt(u8 *fname, u8 *data, int datasz) {
    int     i,
            j,
            fnamelen;
    u8      *p;

    p = strrchr(fname, '\\');
    if(!p) p = strrchr(fname, '/');
    if(p) fname = p + 1;
    fnamelen = strlen(fname);
    j = 0;
    for(i = 0; i < datasz; i++) {
        if(!j) j = fnamelen;
        data[i] -= (0xd6 - fname[--j]);
    }
}



int check_overwrite(u8 *fname) {
    FILE    *fd;
    u8      ans[16];

    fd = fopen(fname, "rb");
    if(!fd) return(0);
    fclose(fd);
    printf("  the file already exists, do you want to overwrite it (y/N)? ");
    fgets(ans, sizeof(ans), stdin);
    if(tolower(ans[0]) != 'y') return(-1);
    return(0);
}



u8 *fdload(u8 *fname, int *fsize) {
    struct stat xstat;
    FILE    *fd;
    int     size;
    u8      *buff;

    printf("- open %s\n", fname);
    fd = fopen(fname, "rb");
    if(!fd) std_err();
    fstat(fileno(fd), &xstat);
    size = xstat.st_size;
    buff = malloc(size);
    if(!buff) std_err();
    fread(buff, 1, size, fd);
    fclose(fd);
    *fsize = size;
    return(buff);
}



void fddump(u8 *fname, u8 *buff, int buffsz, int overwrite) {
    FILE    *fd;

    printf("- create file %s\n", fname);
    if(!overwrite) {
        if(check_overwrite(fname) < 0) exit(1);
    }
    fd = fopen(fname, "wb");
    if(!fd) std_err();
    fwrite(buff, 1, buffsz, fd);
    fclose(fd);
}



void std_err(void) {
    perror("\nError");
    exit(1);
}


