/*
    Copyright 2008,2009,2010 Luigi Auriemma

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    http://www.gnu.org/licenses/gpl-2.0.txt
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <ctype.h>
#include <sys/stat.h>
#include <zlib.h>

#ifdef WIN32
    #include <direct.h>
    #define PATHSLASH   '\\'
    #define make_dir(x) mkdir(x)
#else
    #define stricmp     strcasecmp
    #define PATHSLASH   '/'
    #define make_dir(x) mkdir(x, 0755)
#endif

typedef uint8_t     u8;
typedef uint16_t    u16;
typedef uint32_t    u32;



#define VER         "0.1.3"
#define NAMESZ      0x104



#pragma pack(1)
typedef struct {
    u32     zero;
    u32     files;
    u32     offset;
} mix_head_t;
typedef struct {
    // u32 namelen
    // ? name
    u32     offset;
    u32     zfilelen;
    u32     filelen;
    u8      type;
} mix_file_t;
#pragma pack()



uint32_t unlzwx(uint8_t *outbuff, uint32_t maxsize, uint8_t *in, uint32_t insize);
int msmixext(FILE *fd);
void check_mix_version(FILE *fd, int files);
void myalloc(u8 **data, unsigned wantsize, unsigned *currsize);
int check_wildcard(u8 *fname, u8 *wildcard);
void mix_extract(FILE *fd, u8 *name, mix_file_t *mix_file, int external);
void check_overwrite(u8 *fname);
void mix_dumpa(u8 *fname, u8 *data, int datasz);
void myfr(FILE *fd, void *data, unsigned size);
void mix_name_decrypt(u8 *name, int namelen);
void std_err(void);



int     mix_listonly    = 0,
        mix_oldver      = 0,
        mix_newver      = 0;
u8      *mix_wildcard   = NULL;



int main(int argc, char *argv[]) {
    FILE    *fd;
    int     i,
            files;
    u8      *fname,
            *fdir   = NULL;

    setbuf(stdout, NULL);

    fputs("\n"
        "Milestone MIX files extractor "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", stdout);

    if(argc < 2) {
        printf("\n"
            "Usage: %s [options] <file.MIX>\n"
            "\n"
            "Options:\n"
            "-l      list files without extracting them\n"
            "-d DIR  specify the output directory (default is the current one)\n"
            "-f \"W\"  extract only the files which match the wildcard W like \"*.dds\"\n"
            "\n", argv[0]);
        exit(1);
    }

    argc--;
    for(i = 1; i < argc; i++) {
        if(((argv[i][0] != '-') && (argv[i][0] != '/')) || (strlen(argv[i]) != 2)) {
            printf("\nError: wrong argument (%s)\n", argv[i]);
            exit(1);
        }
        switch(argv[i][1]) {
            case 'l': mix_listonly   = 1;            break;
            case 'd': fdir           = argv[++i];    break;
            case 'f': mix_wildcard   = argv[++i];    break;
            default: {
                printf("\nError: wrong argument (%s)\n", argv[i]);
                exit(1);
            }
        }
    }
    fname = argv[argc];

    printf("- open %s\n", fname);
    fd = fopen(fname, "rb");
    if(!fd) std_err();

    if(!mix_listonly && fdir) {
        printf("- set output folder %s\n", fdir);
        if(chdir(fdir) < 0) std_err();
    }

    files = msmixext(fd);

    fclose(fd);
    printf("\n- %u files in the archive\n", files);
    return(0);
}



int msmixext(FILE *fd) {
    mix_head_t   mix_head;
    mix_file_t   mix_file;
    u32     total_size,
            check_old_size,
            check_old_sign;
    int     external    = 0,
            files,
            namelen;
    u8      name[NAMESZ + 1];

    myfr(fd, &mix_head, sizeof(mix_head_t));

    // this part is needed only for verifying if it's an old format or not... not easy to guess
    if(fseek(fd, 0, SEEK_END)) std_err();
    total_size = ftell(fd);
    if(fseek(fd, -8, SEEK_END)) std_err();
    myfr(fd, &check_old_size, 4);
    myfr(fd, &check_old_sign, 4);
    if(
      ((check_old_sign == 0x3f800000) && (check_old_size < total_size)) ||
      (mix_head.offset > total_size)
    ) {
    //if(mix_head.zero) {
        printf("- set old MIX format\n");
        mix_oldver = 1;
        if(fseek(fd, -8, SEEK_END)) std_err();
        myfr(fd, &mix_head.offset, 4);
        mix_head.files = mix_head.offset / 0x90;
        mix_head.offset = (ftell(fd) - 4) - mix_head.offset;
    }
    if(fseek(fd, mix_head.offset, SEEK_SET)) std_err();

    check_mix_version(fd, mix_head.files);

    printf("\n"
        "  offset   zipsize  filesize filename\n"
        "---------------------------------------\n");
    for(files = 0; files < mix_head.files; files++) {
        if(mix_oldver) {
            namelen = 0x80;
        } else {
            myfr(fd, &namelen,  4);
            if((namelen < 0) || (namelen > NAMESZ)) {
                printf("\nError: filename too long (%d)\n", namelen);
                exit(1);
            }
        }
        myfr(fd, name, namelen);
        mix_name_decrypt(name, namelen);
        name[namelen] = 0;
        myfr(fd, &mix_file, sizeof(mix_file_t));
        if(mix_newver) external = fgetc(fd);
        fseek(fd, 8, SEEK_CUR);
        if(mix_oldver) fseek(fd, -5, SEEK_CUR);

        mix_extract(fd, name, &mix_file, external);
    }

    return(files);
}



    // the new version is used in SBK 09 and adds an additional 8bit field which says
    // if a file is stored internally in the MIX file or externally (music.mix)
    // there are no other identifiers in the file to know what version it is
void check_mix_version(FILE *fd, int files) {
    int     i,
            oldoff,
            namelen;

    if(mix_oldver) return;

    oldoff = ftell(fd);

    if(files > 3) files = 3;
    for(i = 0; i < files; i++) {
        myfr(fd, &namelen,  4);
        if((namelen < 0) || (namelen >= NAMESZ)) {
            mix_newver = 1;
            break;
        }
        if(fseek(fd, namelen + sizeof(mix_file_t) + 8, SEEK_CUR)) {
            mix_newver = 1;
            break;
        }
    }

    fseek(fd, oldoff, SEEK_SET);
}



void myalloc(u8 **data, unsigned wantsize, unsigned *currsize) {
    if(wantsize <= *currsize) return;
    *data = realloc(*data, wantsize);
    if(!*data) std_err();
    *currsize = wantsize;
}



u8 *create_dir(u8 *fname) {
    u8      *p,
            *l;

    p = strchr(fname, ':'); // unused
    if(p) {
        *p = '_';
        fname = p + 1;
    }
    for(p = fname; *p && strchr("\\/. \t:", *p); p++) *p = '_';
    fname = p;

    for(p = fname; ; p = l + 1) {
        for(l = p; *l && (*l != '\\') && (*l != '/'); l++);
        if(!*l) break;
        *l = 0;

        if(!strcmp(p, "..")) {
            p[0] = '_';
            p[1] = '_';
        }

        make_dir(fname);
        *l = PATHSLASH;
    }
    return(fname);
}



int check_wildcard(u8 *fname, u8 *wildcard) {
    u8      *f,
            *w,
            *a;

    if(!wildcard) return(0);
    f = fname;
    w = wildcard;
    a = NULL;
    while(*f || *w) {
        if(!*w && !a) return(-1);
        if(*w == '?') {
            if(!*f) break;
            w++;
            f++;
        } else if(*w == '*') {
            w++;
            a = w;
        } else {
            if(!*f) break;
            if(tolower(*f) != tolower(*w)) {
                if(!a) return(-1);
                f++;
                w = a;
            } else {
                f++;
                w++;
            }
        }
    }
    if(*f || *w) return(-1);
    return(0);
}



#define UNZIP_BASE(NAME,WBITS) \
int NAME(u8 *in, int insz, u8 *out, int outsz) { \
    static z_stream *z  = NULL; \
    int     ret; \
    \
    if(!z) { \
        z = malloc(sizeof(z_stream)); \
        if(!z) return(-1); \
        z->zalloc = Z_NULL; \
        z->zfree  = Z_NULL; \
        z->opaque = Z_NULL; \
        if(inflateInit2(z, WBITS)) { \
            printf("\nError: zlib initialization error\n"); \
            return(-1); \
        } \
    } \
    inflateReset(z); \
    \
    z->next_in   = in; \
    z->avail_in  = insz; \
    z->next_out  = out; \
    z->avail_out = outsz; \
    ret = inflate(z, Z_FINISH); \
    if(ret != Z_STREAM_END) return(-1); \
    return(z->total_out); \
}
UNZIP_BASE(unzip_zlib,    15)
UNZIP_BASE(unzip_deflate, -15)



void mix_extract(FILE *fd, u8 *name, mix_file_t *mix_file, int external) {
    static int  comp_type = 0;
    u32     oldoff;
    int     outlen    = -1;
    static  int insz  = 0,
                outsz = 0;
    static  u8  *in   = NULL,
                *out  = NULL;

    if(mix_wildcard && (check_wildcard(name, mix_wildcard) < 0)) return;

    printf("%c %08x %08x %08x %s\n",
        external ? 'x' : ' ',
        mix_file->offset, mix_file->zfilelen, mix_file->filelen, name);

    if(mix_listonly) return;
    if(external) return;

    oldoff = ftell(fd);
    if(fseek(fd, mix_file->offset, SEEK_SET)) std_err();

    // type is 1 with both deflate and lzw files... blah
    if(mix_file->type) {    // compressed
        myalloc(&in, mix_file->zfilelen, &insz);
        myfr(fd, in, mix_file->zfilelen);
        myalloc(&out, mix_file->filelen, &outsz);
        // separated "if"s!
        if(!comp_type) {
            outlen = unlzwx(out, mix_file->filelen, in, mix_file->zfilelen);
            if(outlen != mix_file->filelen) comp_type++;
        }
        if(comp_type == 1) {
            outlen = unzip_deflate(in, mix_file->zfilelen, out, mix_file->filelen);
            if(outlen != mix_file->filelen) comp_type++;
        }
        if(comp_type == 2) {    // doesn't exist
            outlen = unzip_zlib(in, mix_file->zfilelen, out, mix_file->filelen);
            if(outlen != mix_file->filelen) comp_type++;
        }
        if(outlen != mix_file->filelen) {
            printf("\nError: wrong compressed output size (%d instead of %d)\n", outlen, mix_file->filelen);
            exit(1);
        }
    } else {
        myalloc(&out, mix_file->filelen, &outsz);
        myfr(fd, out, mix_file->filelen);
    }
    mix_dumpa(name, out, mix_file->filelen);

    fseek(fd, oldoff, SEEK_SET);
}



void check_overwrite(u8 *fname) {
    static int  skip_ans = 0;
    FILE    *fd;
    u8      ans[16];

    if(skip_ans) return;
    fd = fopen(fname, "rb");
    if(!fd) return;
    fclose(fd);
    printf("  the file already exists, do you want to overwrite it (y/N/a)? ");
    fgets(ans, sizeof(ans), stdin);
    if(tolower(ans[0]) == 'a') {
        skip_ans = 1;
        return;
    }
    if(tolower(ans[0]) != 'y') exit(1);
}



void mix_dumpa(u8 *fname, u8 *data, int datasz) {
    FILE    *fdo;

    check_overwrite(fname);
    create_dir(fname);
    fdo = fopen(fname, "wb");
    if(!fdo) std_err();
    fwrite(data, 1, datasz, fdo);
    fclose(fdo);
}



void myfr(FILE *fd, void *data, unsigned size) {
    int     len;

    len = fread(data, 1, size, fd);
    if(len != size) {
        printf("\nError: incomplete input file, can't read %u bytes\n", size - len);
        exit(1);
    }
}



void mix_name_decrypt(u8 *name, int namelen) {
    int     i;

    if(mix_oldver) {
        for(i = 0; i < namelen; i++) {
            name[i] ^= 58;
        }
    } else {
        for(i = 0; i < namelen; i++) {
            name[i] ^= (i * 58);
        }
    }
}



void std_err(void) {
    perror("\nError");
    exit(1);
}


