/*
    Copyright 2008 Luigi Auriemma

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    http://www->gnu.org/licenses/gpl.txt
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>

#ifdef WIN32
#else
    #define stricmp strcasecmp
    #define stristr strcasestr
#endif

typedef uint8_t     u8;
typedef uint16_t    u16;
typedef uint32_t    u32;



#define VER             "0.1.2"
#define BUFFSZ          0x10000 // do NOT modify!
#define ISETHERNET(x)   ((x & 7) == 0)
#define ISWIFI(x)       ((x & 7) == 1)
#define ISTOKENRING(x)  ((x & 7) == 2)



void putxx(FILE *fd, u32 num, int bits);
void create_acp(FILE *fd, int type);
int ncfccf_dump(FILE *fd, FILE *fdo, int ccf, int num);
int fgetz(FILE *fd, u8 *data, int size);
void std_err(void);



int main(int argc, char *argv[]) {
    FILE    *fd,
            *fdo;
    int     ccf     = -1,
            pcks;
    u8      *in_file,
            *out_file,
            *p;

    setbuf(stdout, NULL);

    fputs("\n"
        "NCF/CCF packet format to tcpdump capture format "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", stdout);

    if(argc < 3) {
        printf("\n"
            "Usage: %s <input.NCF/CCF> <output.CAP>\n"
            "\n", argv[0]);
        exit(1);
    }

    in_file  = argv[1];
    out_file = argv[2];

    p = strrchr(in_file, '.');
    if(p) {
        if(!stricmp(p, ".ncf")) {
            ccf = 0;
        } else if(!stricmp(p, ".ccf")) {
            ccf = 1;
        }
    }
    if(ccf < 0) {
        printf("\nError: the input file doesn't have the NCF or CCF extension\n");
        exit(1);
    }

    printf("- open input file %s\n", in_file);
    fd = fopen(in_file, "rb");
    if(!fd) std_err();

    printf("- create output file %s\n", out_file);
    fdo = fopen(out_file, "wb");
    if(!fdo) std_err();

    for(pcks = 0; !ncfccf_dump(fd, fdo, ccf, pcks); pcks++) {
        fputc('.', stdout);
    }

    printf("\n- %u packets converted\n", pcks);
    fclose(fd);
    fclose(fdo);
    return(0);
}



void putxx(FILE *fd, u32 num, int bits) {
    int         i,
                bytes;

    bytes = bits >> 3;
    for(i = 0; i < bytes; i++) {
        fputc(num >> (i << 3), fd);
    }
}



void create_acp(FILE *fd, int type) {
    putxx(fd, 0xa1b2c3d4, 32);
    putxx(fd, 2,          16);
    putxx(fd, 4,          16);
    putxx(fd, 0,          32);
    putxx(fd, 0,          32);
    putxx(fd, 65535,      32);
    if(ISETHERNET(type)) {
        printf("- set Ethernet type\n");
        putxx(fd, 0x01, 32);
    } else if(ISTOKENRING(type)) {
        printf("- set Tokenring type\n");
        putxx(fd, 0x06, 32);
    } else if(ISWIFI(type)) {
        printf("- set 802.11/wifi type\n");
        putxx(fd, 0x69, 32);
    } else {
        printf("- set Ethernet type\n");
        putxx(fd, 0x01, 32);
    }
}



u8 hex2byte(u8 *hex) {
    static const u8 hextable[256] =
        "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
        "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
        "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
        "\x00\x01\x02\x03\x04\x05\x06\x07\x08\x09\x00\x00\x00\x00\x00\x00"
        "\x00\x0a\x0b\x0c\x0d\x0e\x0f\x00\x00\x00\x00\x00\x00\x00\x00\x00"
        "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
        "\x00\x0a\x0b\x0c\x0d\x0e\x0f\x00\x00\x00\x00\x00\x00\x00\x00\x00"
        "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
        "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
        "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
        "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
        "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
        "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
        "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
        "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"
        "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00";

    return((hextable[hex[0]] << 4) | hextable[hex[1]]);
}



int ncfccf_dump(FILE *fd, FILE *fdo, int ccf, int num) {
    static int  ccfbuffsz   = 0;
    int     i,
            j,
            len;
    u16     n1,
            n2;
    u8      ncf[24];
    static u8   *buff       = NULL,
                *ccfbuff    = NULL;

    if(!buff) {
        buff = malloc(BUFFSZ);
        if(!buff) std_err();
    }

    if(ccf) {
        if(!num) create_acp(fdo, 0);

        len = fgetz(fd, buff, 20 + 1);
        if(len < 0) return(-1);
        if(len < 20) return(0);
        if(buff[0] != '#') return(-1);
        for(j = 0;;) {
            len = fgetz(fd, buff, BUFFSZ);
            if(len < 0) break;
            if(!len) return(0);
            for(i = 0; i < len; i += 2, j++) {
                if(buff[i] == '#') {
                    fwrite("\0\0\0\0\0\0\0\0", 8, 1, fdo);
                    putxx(fdo, j, 32);
                    putxx(fdo, j, 32);
                    fwrite(ccfbuff, j, 1, fdo);
                    return(0);
                }
                if(j >= ccfbuffsz) {
                    ccfbuffsz += BUFFSZ;
                    ccfbuff = realloc(ccfbuff, ccfbuffsz);
                    if(!ccfbuff) std_err();
                }
                ccfbuff[j] = hex2byte(buff + i);
            }
        }
        return(-1);
    }

    if(fread(ncf, sizeof(ncf), 1, fd) != 1) return(-1);
    n1 = (ncf[0]) | (ncf[1] << 8);
    n2 = (ncf[2]) | (ncf[3] << 8);

    if(!num) create_acp(fdo, ncf[16]);

    for(i = 11; i >= 4; i--) fputc(ncf[i], fdo);
    putxx(fdo, n1, 32);
    putxx(fdo, n2, 32);

    if(fread(buff, 1, n1, fd) != n1) return(-1);

    if(ISWIFI(ncf[16])) buff[1] &= 0xbf;    // removes the proceted bit

    fwrite(buff, n1, 1, fdo);
    return(0);
}



int fgetz(FILE *fd, u8 *data, int size) {
    u8     *p;

    if(!fgets(data, size, fd)) return(-1);
    for(p = data; *p && (*p != '\n') && (*p != '\r'); p++);
    *p = 0;
    return(p - data);
}



void std_err(void) {
    perror("\nError");
    exit(1);
}


