/*
    Copyright 2007-2011 Luigi Auriemma

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    http://www.gnu.org/licenses/gpl-2.0.txt
*/

#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <string.h>
#include <ctype.h>
#include <time.h>

#ifdef WIN32
    #include <winsock.h>
    #include "winerr.h"

    #define close       closesocket
    #define sleep       Sleep
    #define in_addr_t   u32
#else
    #include <unistd.h>
    #include <sys/socket.h>
    #include <sys/types.h>
    #include <arpa/inet.h>
    #include <netinet/in.h>
    #include <netdb.h>

    #define stricmp     strcasecmp
#endif

typedef uint8_t     u8;
typedef uint16_t    u16;
typedef uint32_t    u32;



#define VER         "0.1.12"



void getguid(u8 *guid, u8 *src);
u8 *fgetz(u8 *buff, int buffsz, FILE *fd);
int is_a_guid(u8 *data);
int create_rand_string(u8 *data, int len, u32 rnd);
int getpbver(u8 *ver);
void pbxor(u8 *data, int len, u8 *key);
void hex2byte(u8 *in, u8 *out);
u32 get_num(u8 *str);
int send_recv(int sd, u8 *in, int insz, u8 *out, int outsz, int err, struct sockaddr_in *peer, int tries);
int timeout(int sock, int secs);
in_addr_t resolv(char *host);
void std_err(void);



typedef struct {
    u32     id;
    u8      *host;
    u8      *name;
    u8      *pbsv;  // useless
    u8      *pbcl;  // useless
    u8      *pba;   // useless
} gamedb_t;



gamedb_t gamedb[] = {
//    ID          name      long name                           server    client
    { 0x357afe15, "aao",    "America's Army",                   "v1.739", "v2.243", "A1371" },
    { 0x357afe3f, "aav3",   "America's Army 3",                 "v1.741", "v2.184", "A1372" },
    { 0x357afe47, "acb",    "Assassin's Creed Brotherhood",     "v1.802", "v2.261", "A1379" },
    { 0x357afe3d, "apb",    "APB: All Points Bulletin",         "v1.827", "v2.271", "A1373" },
    { 0x357afe17, "bf1942", "Battlefield 1942",                 "v1.457", "v2.110", "A1331" },
    { 0x357afe20, "bf2",    "Battlefield 2",                    "v1.800", "v2.241", "A1392" },
    { 0x357afe29, "bf2142", "Battlefield 2142",                 "v1.822", "v2.262", "A1375" },
    { 0x357afe35, "heroes", "Battlefield Heroes",               "v1.821", "v2.260", "A1376" },
    { 0x357afe48, "bfp4f",  "Battlefield Play4Free",            "v1.794", "v2.244", "A1377" },
    { 0x357afe18, "bfv",    "Battlefield Vietnam",              "v1.458", "v1.757", "A1334" },
    { 0x357afe41, "bc2",    "Battlefield: Bad Company 2",       "v1.826", "v2.272", "A1382" },
    { 0x357afe19, "cod",    "Call of Duty",                     "v1.729", "v2.165", "A1362" },
    { 0x357afe24, "cod2",   "Call of Duty 2",                   "v1.760", "v2.208", "A1383" },
    { 0x357afe32, "cod4",   "Call of Duty 4",                   "v1.828", "v2.258", "A1407" },
    { 0x357afe3a, "waw",    "Call of Duty: World at War",       "v1.814", "v2.259", "A1409" },
    { 0x357afe2d, "crysis", "Crysis",                           "v1.735", "v2.161", "A1391" },
    { 0x357afe3b, "wh",     "Crysis Wars",                      "v1.742", "v2.160", "A1394" },
    { 0x357afe1e, "doom3",  "DOOM 3",                           "v1.744", "v1.201", "A1308" },
    { 0x357afe14, "et",     "Enemy Territory",                  "v1.809", "v2.254", "A1382" },
    { 0x357afe2b, "etqw",   "Enemy Territory: QUAKE Wars",      "v1.685", "v2.114", "A1313" },
    { 0x357afe39, "fcx",    "Far Cry 2",                        "v1.734", "v2.159", "A1392" },
    { 0x357afe31, "fearpm", "F.E.A.R. Perseus Mandate",         "v1.746", "v2.019", "A1391" },
    { 0x357afe34, "ffow",   "Frontlines: Fuel of War",          "v1.699", "v2.154", "A1356" },
    { 0x357afe44, "moh",    "Medal of Honor",                   "v1.810", "v2.262", "A1384" },
    { 0x357afe30, "moha",   "Medal of Honor: Airborne",         "v1.483", "v2.022", "A1396" },
    { 0x357afe2f, "nfs",    "Need for Speed Pro Street",        "v1.272", "v1.297", "A1353" },
    { 0x357afe3c, "unco",   "Need for Speed: Undercover",       "v1.651", "v2.100", "A1390" },
    { 0x357afe25, "prey",   "Prey",                             "v1.745", "v1.269", "A1309" },
    { 0x357afe12, "q3a",    "Quake III Arena",                  "v1.641", "v2.041", "A1353" },
    { 0x357afe21, "q4",     "Quake 4",                          "v1.743", "v1.292", "A1307" },
    { 0x357afe38, "ql",     "QUAKE LIVE",                       "v1.642", "v2.042", "A1354" },
    { 0x357afe16, "rvs",    "Rainbow Six 3: Raven Shield",      "v1.731", "v2.158", "A1348" },
    { 0x357afe2c, "vegas",  "Rainbow Six: Vegas",               "v1.732", "v2.157", "A1351" },
    { 0x357afe33, "vegas2", "Rainbow Six: Vegas 2",             "v1.733", "v2.156", "A1352" },
    { 0x357afe11, "rtcw",   "Return to Castle Wolfenstein",     "v1.730", "v2.164", "A1345" },
    { 0x357afe13, "sof2",   "Soldier of Fortune II",            "v1.728", "v2.152", "A1363" },
    { 0x357afe28, "wr",     "War Rock",                         "v1.694", "v2.123", "A1404" },
    { 0x357afe36, "war",    "Warhammer Online",                 "",       "",       ""      },
    { 0x357afe3e, "wolf",   "Wolfenstein",                      "1.751",  "2.202",  "A1316" },
    // no longer supported
    //{ 0x357afe1b, "farcry", "Far Cry",                          "v1.229", "v1.192", "A1335" },
    //{ 0x357afe22, "fear",   "F.E.A.R.",                         "v1.308", "v1.716", "A1386" },
    //{ 0x357afe1c, "jotr",   "Joint Operations: Typhoon Rising", "v1.229", "v1.243", "A1343" },
    //{ 0x357afe26, "kol",    "Knight Online",                    "v1.229", "v1.243", "A1343" },
    //{ 0x357afe1d, "mohpa",  "Medal of Honor: Pacific Assault",  "v1.150", "v1.066", "A1322" },
    //{ 0x357afe23, "r6v4",   "Rainbow Six 4: Lockdown",          "v1.229", "v1.100", "A1304" },
    //{ 0x357afe1a, "scpt",   "Splinter Cell: Pandora Tomorrow",  "v1.081", "v1.058", "A1336" },
    //{ 0x357afe27, "uo",     "Ultima Online",                    "v1.242", "v1.700", "A1336" },
    { 0x00000000, NULL,     NULL,                               NULL,     NULL,     NULL    },
    { 0x00000000, "",       "",                                 "",       "",       ""      }   // force_id
};



int main(int argc, char *argv[]) {
    struct  sockaddr_in peer,
                        peerc;
    FILE    *fd         = NULL;
    u32     force_id    = 0;
    int     sd,
            i,
            len,
            gamenum,
            maxgames,
            session_id,
            game_id,
            pb_sv_ver,
            pb_cl_ver,
            pb_a_ver;
    u16     pbport      = 24340;
    u8      line[1024],
            buff[512],
            pbhost[64],
            guid[33],
            key[33],
            nick[10],
            *gamename,
            *guid_arg,
            *p;

#ifdef WIN32
    WSADATA    wsadata;
    WSAStartup(MAKEWORD(1,0), &wsadata);
#endif

    setbuf(stdout, NULL);
    setbuf(stderr, NULL);

    fputs("\n"
        "PunkBuster online GUID checker "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", stderr);

    for(maxgames = 0; gamedb[maxgames].id; maxgames++);

    if(argc < 3) {
        fprintf(stderr, "\n"
            "Usage: %s <gamename> <GUID_or_listfile> [force_gameid]\n"
            "\n"
            "the tool can check a single GUID or a text file with a list of GUIDs.\n"
            "the format of the input list can be the classical \"one per line\" or even the\n"
            "PunkBusted Master Ban Lists, the tool will recognize the GUIDs automatically.\n"
            "is also possible to redirect the result of each check to file like with:\n"
            "  %s bf2142 list_file.txt > results.txt\n"
            "\n", argv[0], argv[0]);

        fprintf(stderr,
            "num gamename fullname\n"
            "---------------------\n");
        for(i = 0; i < maxgames; i++) {
            fprintf(stderr, "%2d  %-7s  %s\n", i + 1, gamedb[i].host, gamedb[i].name);
        }
        fprintf(stderr, "\n");
        exit(1);
    }

    gamename = argv[1];
    guid_arg = argv[2];
    if(argc >= 4) force_id = get_num(argv[3]);

    gamenum = atoi(gamename);
    if(!gamenum) {
        for(i = 0; i < maxgames; i++) {
            if(!stricmp(gamename, gamedb[i].host)) break;
            if(!stricmp(gamename, gamedb[i].name)) break;
        }
        gamenum = i;
    } else {
        gamenum--;
    }
    if(force_id) {
        gamenum = maxgames + 1;
        gamedb[gamenum].id   = force_id;
        gamedb[gamenum].host = gamename;
        gamedb[gamenum].name = gamename;
    } else if((gamenum < 0) || (gamenum >= maxgames)) {
        fprintf(stderr, "\nError: invalid gamenum or gamename\n");
        exit(1);
    }

    if(is_a_guid(guid_arg)) {
        getguid(guid, guid_arg);
    } else {
        if(!strcmp(guid_arg, "-")) {
            fprintf(stderr, "- load the list from stdin\n");
            fd = stdin;
        } else {
            fprintf(stderr, "- try to load the file %s\n", guid_arg);
            fd = fopen(guid_arg, "rb");
            if(!fd) std_err();
        }
    }

    fprintf(stderr, "- game: %s\n", gamedb[gamenum].name);
    sprintf(pbhost, "%s-b.%s", gamedb[gamenum].host, "evenbalance.com");

    fprintf(stderr, "- resolv %s: ", pbhost);
    peer.sin_addr.s_addr = resolv(pbhost);
    peer.sin_port        = htons(pbport);
    peer.sin_family      = AF_INET;
    fprintf(stderr, "%s\n", inet_ntoa(peer.sin_addr));

    sd = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
    if(sd < 0) std_err();

    game_id    = gamedb[gamenum].id;
    pb_sv_ver  = getpbver(gamedb[gamenum].pbsv);
    pb_cl_ver  = getpbver(gamedb[gamenum].pbcl);
    pb_a_ver   = getpbver(gamedb[gamenum].pba);

redo_file:
    if(fd) {
        for(;;) {
            if(!fgetz(line, sizeof(line), fd)) goto quit;
            if(line[0] == '[') {
                p = strchr(line, ']');
                if(!p) continue;
                for(p++; *p && ((*p == ' ') || (*p == '\t')); p++);
                if(p[0] == '<') {
                    p = strchr(p, '>');
                    if(!p) continue;
                    for(p++; *p && ((*p == ' ') || (*p == '\t')); p++);
                }
                getguid(guid, p);
                if(!is_a_guid(guid)) continue;
                break;
            }
            if(line[0] == '<') continue;
            if(line[0] == ';') continue;
            for(p = line; *p && ((*p == ' ') || (*p == '\t')); p++);
            getguid(guid, p);
            if(is_a_guid(guid)) break;
        }
    }

    fprintf(stderr, "\n- guid: %s\n", guid);

    len = sprintf(
        buff,
        "%c%u %x %u %c %d %d %d %d \"%s\" \"%s\"",
        'G',
        (unsigned)time(NULL),
        game_id,
        0,
        'w',
        pb_sv_ver,
        pb_cl_ver,
        pb_a_ver,
        1,
        "e",
        "h");

    fprintf(stderr, "- send: %s\n", buff);
    len = send_recv(sd, buff, len, buff, sizeof(buff), 1, &peer, 2);
    fprintf(stderr, "- recv: %.*s\n", len, buff);

    sscanf(
        buff,
        "\xff\xff\xff\xff" "PB_B" "C" "%u %32s",
        &session_id,    // that's wrong but I like compatibility
        key);

    hex2byte(key, key);

    peerc.sin_addr.s_addr = ((~session_id * game_id) * 0x343FD) + 0x269EC3; // something random
    peerc.sin_port        = htons(27960);                                   // Q3 client port
    create_rand_string(nick, sizeof(nick), session_id);

    len = sprintf(
        buff,
        "%cBanServ%u %x %s %s:%hu \"%s\" %s",
        'K',
        session_id,
        game_id,
        guid,
        inet_ntoa(peerc.sin_addr),
        ntohs(peerc.sin_port),
        nick,
        "");

    fprintf(stderr, "- send: %s\n", buff);
    pbxor(buff + 1, len - 1, key);
    len = send_recv(sd, buff, len, buff, sizeof(buff), 0, &peer, 2);
    if(len < 0) {
        fprintf(stdout, "  the GUID %s is NOT banned\n", guid);
    } else {
        fprintf(stderr, "- recv: %.*s\n", len, buff);
        fprintf(stdout, "  the GUID %s is BANNED!!!\n", guid);
    }

    if(fd) goto redo_file;

quit:
    if(fd && (fd != stdin)) fclose(fd);
    close(sd);
    fprintf(stderr, "- done\n");
    return(0);
}



void getguid(u8 *guid, u8 *src) {
    int     i;

    for(i = 0; i < 32; i++) {
        guid[i] = tolower(src[i]);
    }
    guid[i] = 0;
}



u8 *fgetz(u8 *buff, int buffsz, FILE *fd) {
    u8      *p;

    if(!fgets(buff, buffsz, fd)) return(NULL);
    for(p = buff; *p && (*p != '\r') && (*p != '\n'); p++);
    *p = 0;
    return(buff);
}



int is_a_guid(u8 *data) {
    int     i;
    u8      c;

    if(strlen(data) != 32) return(0);
    for(i = 0; i < 32; i++) {
        c = tolower(data[i]);
        if(!(((c >= '0') && (c <= '9')) || ((c >= 'a') && (c <= 'z')))) return(0);
    }
    return(1);
}



int create_rand_string(u8 *data, int len, u32 rnd) {
    u8      *p = data;
    static const u8 table[] =
            // "0123456789"
            "ABCDEFGHIJKLMNOPQRSTUVWXYZ"
            "abcdefghijklmnopqrstuvwxyz";

    len = rnd % len;        // len must be the full length of data, NULL included
    if(len < 3) len = 3;

    while(len--) {
        rnd = (rnd * 0x343FD) + 0x269EC3;
        rnd >>= 3;
        *p++ = table[rnd % (sizeof(table) - 1)];
    }
    *p = 0;

    return(p - data);
}



int getpbver(u8 *ver) {
    int     numver;
    u8      *p;

    if(!ver[0]) return(0);
    while(!isdigit(*ver)) ver++;

    numver = atoi(ver);

    p = strchr(ver, '.');
    if(p) numver = (numver * 1000) + atoi(p + 1);

    return(numver);
}



void pbxor(u8 *data, int len, u8 *key) {
    int     i;

    for(i = 0; i < len; i++) {
        data[i] ^= key[i & 15];
    }
}



void hex2byte(u8 *in, u8 *out) {
    int     tmp;

    while(in[0] && in[1]) {
        sscanf(in, "%02x", &tmp);
        *out = tmp;
        in += 2;
        out++;
    }
}



u32 get_num(u8 *str) {
    u32     num;

    if((str[0] == '0') && (tolower(str[1]) == 'x')) {
        sscanf(str + 2, "%x", &num);
    } else {
        sscanf(str, "%u", &num);
    }
    return(num);
}



int send_recv(int sd, u8 *in, int insz, u8 *out, int outsz, int err, struct sockaddr_in *peer, int tries) {
    int     i,
            len;

    if(in && !out) {
        if(sendto(sd, in, insz, 0, (struct sockaddr *)peer, sizeof(struct sockaddr_in))
          < 0) std_err();
        return(0);
    }
    if(in) {
        for(i = 0; i < tries; i++) {
            if(sendto(sd, in, insz, 0, (struct sockaddr *)peer, sizeof(struct sockaddr_in))
              < 0) std_err();
            if(!timeout(sd, tries - i)) break;  // wait a dynamic amount of time
        }
        if(i >= tries) {
            if(!err) return(-1);
            fprintf(stderr, "\nError: socket timeout, no reply received\n");
            exit(1);
        }
    } else {
        if(timeout(sd, 2) < 0) return(-1);
    }

    len = recvfrom(sd, out, outsz - 1, 0, NULL, NULL);  // -1 for NULL delimiter
    if(len < 0) std_err();
    out[len] = 0;
    return(len);
}



int timeout(int sock, int secs) {
    struct  timeval tout;
    fd_set  fd_read;
    int     err;

    tout.tv_sec  = secs;
    tout.tv_usec = 0;
    FD_ZERO(&fd_read);
    FD_SET(sock, &fd_read);
    err = select(sock + 1, &fd_read, NULL, NULL, &tout);
    if(err < 0) std_err();
    if(!err) return(-1);
    return(0);
}



in_addr_t resolv(char *host) {
    struct      hostent *hp;
    in_addr_t   host_ip;

    host_ip = inet_addr(host);
    if(host_ip == INADDR_NONE) {
        hp = gethostbyname(host);
        if(!hp) {
            fprintf(stderr, "\nError: Unable to resolv hostname (%s)\n", host);
            exit(1);
        } else host_ip = *(in_addr_t *)hp->h_addr;
    }
    return(host_ip);
}



#ifndef WIN32
    void std_err(void) {
        perror("\nError");
        exit(1);
    }
#endif

