/*
    Copyright 2007 Luigi Auriemma

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    http://www.gnu.org/licenses/gpl.txt
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <stdarg.h>
#include <ctype.h>
#include <time.h>
#include <zlib.h>

#ifdef WIN32
    #include <winsock.h>
    #include "winerr.h"

    #define close   closesocket
    #define sleep   Sleep
    #define ONESEC  1000
#else
    #include <unistd.h>
    #include <sys/socket.h>
    #include <sys/types.h>
    #include <arpa/inet.h>
    #include <netinet/in.h>
    #include <netdb.h>

    #define ONESEC  1
    #define stristr strcasestr
#endif



#define VER         "0.1.1"
#define BUFFSZ      0xffff
#define EBFH        0x45424648
#define EBFB        0x45424642
#define dounzip     size != zsize

typedef uint8_t     u8;
typedef uint16_t    u16;
typedef uint32_t    u32;



void myerr(char *fmt, ...);
int unzip(u8 *in, int size, u8 *out, int maxsz);
void xor(u8 *data, int off, int len, u8 *key);
void rnds(u8 *data, int len);
int send_recv(int sd, u8 *in, int insz, u8 *out, int outsz, struct sockaddr_in *peer, int err);
int getmm(u8 *data, u8 *dst, int len);
int getxx(u8 *data, u32 *ret, int bits);
int putxx(u8 *data, u32 num, int bits);
int putmm(u8 *data, u8 *mem, int len);
void create_dir(u8 *fname);
int timeout(int sock, int sec);
u32 resolv(char *host);
void std_err(void);



int main(int argc, char *argv[]) {
    struct  sockaddr_in peer;
    FILE    *fd;
    u32     id,
            myid,
            size,
            zsize,
            ebf,
            num;
    int     sd,
            len,
            zlen,
            need,
            rate    = 4;
    u16     port;
    u8      header[16],
            key[12],
            hash[16],
            *buff,
            *zbuff,
            *fbuff,
            *host,
            *fname,
            *p;

#ifdef WIN32
    WSADATA    wsadata;
    WSAStartup(MAKEWORD(1,0), &wsadata);
#endif

    setbuf(stdout, NULL);

    fputs("\n"
        "Punkbuster master server file downloader "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", stdout);

    if(argc < 4) {
        printf("\n"
            "Usage: %s <filename> <PB_server> <PB_port> [rate]\n"
            "\n"
            "Example for Call of Duty (cod) and Battlefield 2142 (bf2142):\n"
            "  pbdown pbq.5 cod1.evenbalance.com 28960\n"
            "  pbdown pbsec.sv bf21421.evenbalance.com 17567\n"
            "  pbdown htm\\ws001604.htm bf21422.evenbalance.com 17567\n"
            "\n"
            "Some files: pbpat.1, pbsec.cl, pbsec.sv, pbq.4, pbq.5 and possibly others\n"
            "The master server hostname and port are available in the pbsec file,\n"
            "like the following example for CoD and bf2142 (other names on EB website):\n"
            "  http://www.evenbalance.com/downloads/cod/pbsec.htm\n"
            "  http://www.evenbalance.com/downloads/bf2142/pbsec.htm\n"
            "\n"
            "Remember that exist two master servers for each game which have some of the\n"
            "files (for example pbsec.sv on the first and the htm\\* on the second) like:\n"
            " S 2 BF21421.EVENBALANCE.COM 17567 0\n"
            " M BF21422.EVENBALANCE.COM 17567\n"
            "\n", argv[0]);
        exit(1);
    }

    if(argc > 4) rate = atoi(argv[4]);
    fname = argv[1];
    host  = argv[2];
    port  = atoi(argv[3]);

    peer.sin_addr.s_addr = resolv(host);
    peer.sin_port        = htons(port);
    peer.sin_family      = AF_INET;

    printf("- target   %s : %hu\n",
        inet_ntoa(peer.sin_addr), ntohs(peer.sin_port));

    sd = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
    if(sd < 0) std_err();

    buff = malloc(BUFFSZ);
    if(!buff) std_err();

    sprintf(header, "%.*s", sizeof(header), "\xff\xff\xff\xff" "PB_S");
    rnds(key, sizeof(key) - 1);

    len = 1 + sprintf(buff,
        "key \"%s\" file \"%s\" header \"%s\" rate \"%d\"",
        key,
        fname,
        header,
        rate);

    printf("- %s\n", buff);
    len = send_recv(sd, buff, len, buff, BUFFSZ, &peer, 1);

    xor(buff, strlen(header) + 4, len, key);

    p = buff + strlen(header);
    p += getxx(p, &id,      32);
    p += getxx(p, &ebf,     32);
    p += getxx(p, &size,    32);
    p += getxx(p, &zsize,   32);
    p += getmm(p, hash,     16);
    printf("- %08x %10u %10u %s\n", id, zsize, size, p);

    if(ebf != EBFH) myerr("no EBFH header received (%08x)\n", ebf);
    myid = id;

    fbuff = malloc(size);
    if(!fbuff) std_err();

    if(dounzip) {
        zbuff = malloc(zsize);
        if(!zbuff) std_err();
    } else {
        zbuff = fbuff;
    }

    for(need = zlen = 0;;) {
        len = send_recv(sd, NULL, 0, buff, BUFFSZ, &peer, 0);
        if(len <= 0) break;

        xor(buff, strlen(header) + 4, len, key);

        p = buff + strlen(header);
        p += getxx(p, &id,      32);
        p += getxx(p, &ebf,     32);
        p += getxx(p, &num,     32);

        if(id  != myid) continue;
        if(ebf == EBFH) continue;
        if(ebf != EBFB) myerr("no EBFB header received (%08x)\n", ebf);

        printf("  %08x\r", num);
        if(num != need) myerr("the received fragment (%u) wasn't expected (%u)\n", num, need);

        len -= (p - buff);
        if(len <= 0) continue;

        if((zlen + len) > zsize) myerr("input data (at least %u) is major than expected (%u)\n", zlen + len, zsize);
        memcpy(zbuff + zlen, p, len);
        zlen += len;
        need++;
    }

    if(zlen != zsize) myerr("input data (%u) is different than expected (%u)\n", zlen, zsize);

    if(dounzip) {
        len = unzip(zbuff, zsize, fbuff, size);
        if(len != size) myerr("uncompressed data (%u) is different than expected (%u)\n", len, size);
    }

    printf("- create file %s\n", fname);
    create_dir(fname);
    fd = fopen(fname, "wb");
    if(!fd) std_err();
    fwrite(fbuff, size, 1, fd);
    fclose(fd);

    close(sd);
    if(dounzip) free(zbuff);
    free(fbuff);
    free(buff);
    printf("- finished\n");
    return(0);
}



void myerr(char *fmt, ...) {
    va_list ap;

    printf("\nError: ");
    va_start(ap, fmt);
    vprintf(fmt, ap);
    va_end(ap);
    exit(1);
}



int unzip(u8 *in, int size, u8 *out, int maxsz) {
    z_stream    z;
    int         ret;

    z.zalloc = (alloc_func)0;
    z.zfree  = (free_func)0;
    z.opaque = (voidpf)0;

    if(inflateInit2(&z, 15)) {
        printf("\nError: zlib initialization error\n");
        exit(1);
    }

    z.next_in   = in;
    z.avail_in  = size;
    z.next_out  = out;
    z.avail_out = maxsz;
    inflate(&z, Z_NO_FLUSH);

    ret = z.total_out;
    inflateEnd(&z);
    return(ret);
}



void xor(u8 *data, int off, int len, u8 *key) {
    int     i,
            keylen;

    keylen = strlen(key);
    for(i = off; i < len; i++) {
        data[i] ^= key[i % keylen];
    }
}



void rnds(u8 *data, int len) {
    u32     rnd;
    u8      *p = data;
    static const u8 table[] =
                "+:;$<>*|"
                "0123456789"
                "ABCDEFGHIJKLMNOPQRSTUVWXYZ"
                "abcdefghijklmnopqrstuvwxyz";

    rnd = time(NULL);

    while(len--) {
        rnd = (rnd * 0x343FD) + 0x269EC3;
        *p++ = table[rnd % (sizeof(table) - 1)];
    }
    *p = 0;
}



int send_recv(int sd, u8 *in, int insz, u8 *out, int outsz, struct sockaddr_in *peer, int err) {
    int     retry,
            len;

    if(in && !out) {
        if(sendto(sd, in, insz, 0, (struct sockaddr *)peer, sizeof(struct sockaddr_in))
          < 0) std_err();
        return(0);
    }
    if(in) {
        for(retry = 2; retry; retry--) {
            if(sendto(sd, in, insz, 0, (struct sockaddr *)peer, sizeof(struct sockaddr_in))
              < 0) std_err();
            if(!timeout(sd, 1)) break;
        }

        if(!retry) {
            if(!err) return(-1);
            printf("\nError: socket timeout, no reply received\n\n");
            exit(1);
        }
    } else {
        if(timeout(sd, 3) < 0) return(-1);
    }

    len = recvfrom(sd, out, outsz, 0, NULL, NULL);
    if(len < 0) std_err();
    return(len);
}



int getmm(u8 *data, u8 *dst, int len) {
    memcpy(dst, data, len);
    return(len);
}



int getxx(u8 *data, u32 *ret, int bits) {
    u32     num;
    int     i,
            bytes;

    bytes = bits >> 3;

    for(num = i = 0; i < bytes; i++) {
        num |= (data[i] << ((bytes - 1 - i) << 3));
    }

    *ret = num;
    return(bytes);
}



int putxx(u8 *data, u32 num, int bits) {
    int     i,
            bytes;

    bytes = bits >> 3;

    for(i = 0; i < bytes; i++) {
        data[i] = (num >> ((bytes - 1 - i) << 3)) & 0xff;
    }

    return(bytes);
}



int putmm(u8 *data, u8 *mem, int len) {
    memcpy(data, mem, len);
    return(len);
}



void create_dir(u8 *fname) {
    u8      *p,
            *l;

    for(p = fname; ; p = l + 1) {
        for(l = p; *l && (*l != '\\') && (*l != '/'); l++);
        if(!*l) break;
        *l = 0;

#ifdef WIN32
        mkdir(fname);
        *l = '\\';
#else
        mkdir(fname, 0755);
        *l = '/';
#endif
    }
}



int timeout(int sock, int sec) {
    struct  timeval tout;
    fd_set  fd_read;
    int     err;

    tout.tv_sec  = sec;
    tout.tv_usec = 0;
    FD_ZERO(&fd_read);
    FD_SET(sock, &fd_read);
    err = select(sock + 1, &fd_read, NULL, NULL, &tout);
    if(err < 0) std_err();
    if(!err) return(-1);
    return(0);
}



u32 resolv(char *host) {
    struct  hostent *hp;
    u32     host_ip;

    host_ip = inet_addr(host);
    if(host_ip == INADDR_NONE) {
        hp = gethostbyname(host);
        if(!hp) {
            printf("\nError: Unable to resolv hostname (%s)\n", host);
            exit(1);
        } else host_ip = *(u32 *)hp->h_addr;
    }
    return(host_ip);
}



#ifndef WIN32
    void std_err(void) {
        perror("\nError");
        exit(1);
    }
#endif


