/*
    Copyright 2004-2011 Luigi Auriemma

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    http://www.gnu.org/licenses/gpl-2.0.txt

Contributors:
- CHC (aka Shinto)
- xtrapas
- all the others I don't remember 8-)
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <time.h>
#include <ctype.h>
#include "gs_peerchat.h"
#include "md5.h"
#include "gslist_keys.h"

#ifdef WIN32
    #include <winsock.h>
    #include "winerr.h"

    #define close       closesocket
    #define sleep       Sleep
#else
    #include <unistd.h>
    #include <sys/socket.h>
    #include <sys/types.h>
    #include <arpa/inet.h>
    #include <netinet/in.h>
    #include <netdb.h>
    #include <pthread.h>
    #include <sys/times.h>

    #define stristr     strcasestr
    #define stricmp     strcasecmp
    #define strnicmp    strncasecmp
#endif

#ifdef WIN32
    #define quick_thread(NAME, ARG) DWORD WINAPI NAME(ARG)
    #define thread_id   DWORD
#else
    #define quick_thread(NAME, ARG) void *NAME(ARG)
    #define thread_id   pthread_t
#endif

thread_id quick_threadx(void *func, void *data) {
    thread_id       tid;
#ifdef WIN32
    if(!CreateThread(NULL, 0, func, data, 0, &tid)) return(0);
#else
    pthread_attr_t  attr;
    pthread_attr_init(&attr);
    pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_DETACHED);
    if(pthread_create(&tid, &attr, func, data)) return(0);
#endif
    return(tid);
}

typedef uint8_t     u8;
typedef uint16_t    u16;
typedef uint32_t    u32;



#define VER         "0.3.3b"
#define BUFFSZ      4096



#include "peerchat_ip.h"
void gslist_keys_list(void);
quick_thread(conn, int sock);
int check_drops(u8 *data, int datalen, u8 *dropstr);
void remove_ctcp(u8 *buff, int buffsz);
int rnd_gamekey(void);
int rnds(u8 *data, int len);
int delimit(u8 *data);
int mystrnicmp(u8 *buff, u8 *str, int len);
u8 *get_irc_command(u8 *buff, int *cmd);
u8 *do_md5_auto_check(u8 *data);
void do_md5_key_hash(u8 *key, u8 *ret);
int mysend(int sd, u8 *buff, int len, gs_peerchat_ctx *enc);
int gs_keys(u8 *gamekey, u8 *buff, int len, gs_peerchat_ctx *client, gs_peerchat_ctx *server);
int recv1(int sd, u8 *data, gs_peerchat_ctx *enc);
int myrecv(int sd, u8 *buff, int buffsz);
u32 resolv(char *host);
void std_err(void);



struct  in_addr gsip;
FILE    *fdlog       = NULL;
int     verbose      = 0,
        login_type   = 0;
u32     seed;
u16     lport        = 6667,
        gsport       = 6667;
u8      *gshost      = "peerchat.gamespy.com",
        *gsgamename  = NULL,
        *gsgamekey   = NULL,
        *cdkey       = NULL,
        *cdkeyhash   = NULL,
        *gsa_id      = "0",
        *gsa_ip      = NULL,
        *drop_client = NULL,
        *drop_server = NULL,
        *login_arg1  = NULL,
        *login_arg2  = NULL;



static const u8 hex[] = "0123456789abcdef";



int main(int argc, char *argv[]) {
    struct  sockaddr_in peer;
    int     sdl,
            sda,
            i,
            psz,
            on          = 1;
    u8      *lhost      = "127.0.0.1",
            *logfile    = NULL,
            *p;

#ifdef WIN32
    WSADATA    wsadata;
    WSAStartup(MAKEWORD(1,0), &wsadata);
#endif

    setbuf(stdout, NULL);

    fputs("\n"
        "GS peerchat IRC proxy "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", stdout);

    printf("\n"
        "Usage: %s [options]\n"
        "\n"
        "Options:\n"
        "-g N K    specify the gamename and gamekey to use, default %s %s\n"
        "-G        show the list of gamenames/keys in the internal database, note that\n"
        "          it's not updated so I suggest to refer ever to the gslist.cfg file\n"
        "-l IP     bind a specific interface, default is %s, use 0 for any\n"
        "-p PORT   local port to bind (%hu)\n"
        "-v        show all the IRC messages sent and received\n"
        "-k KEY    in case you want to use a specific cdkey, default is %s\n"
        "-h hash   as above but use directly a specific MD5 hash\n"
        "-f FILE   log all the messages in FILE\n"
        "-s IP:P   custom peerchat server address, default is %s:%hu\n"
        "-i ID     specify a Gamespy ID, default %s\n"
        "-a IP     specify a custom IP address or any other type of string you desire\n"
        "-L N X Y  LOGIN command usually used by Gamespy Arcade (N 1) and some games\n"
        "          (N 4). For LOGIN 1 X is your account password and Y is\n"
        "          uniquenick@email (for example: myuniquenick@aluigi@autistici.org)\n"
        "          while in LOGIN 4 they are your unique nickname and your password\n"
        "          note that if the password is 32 chars long will be considered as a\n"
        "          MD5 hash and will be passed directly to the server as is\n"
        "-D 1,2,N  drop the messages sent by the server, separate them by comma\n"
        "          for example -D 702,332,333,703 will not send them to your client\n"
        "-d 1,2,N  as above but for clients, for example -d whois,quit\n"
        "\n",
        argv[0],
        gsgamename ? gsgamename : (u8 *)"random", gsgamekey ? gsgamekey : (u8 *)"random",
        lhost, lport,
        cdkey ? cdkey : (u8 *)"random",
        gshost, gsport,
        gsa_id);

    for(i = 1; i < argc; i++) {
        if(((argv[i][0] != '-') && (argv[i][0] != '/')) || (strlen(argv[i]) != 2)) {
            printf("\nError: wrong argument (%s)\n", argv[i]);
            exit(1);
        }
        switch(argv[i][1]) {
            case 'g': {
                if(!argv[++i]) exit(1);
                gsgamename  = argv[i];
                if(!argv[++i]) exit(1);
                gsgamekey   = argv[i];
                break;
            }
            case 'G': {
                gslist_keys_list();
                exit(0);
                break;
            }
            case 'l': {
                if(!argv[++i]) exit(1);
                lhost       = argv[i];
                break;
            }
            case 'p': {
                if(!argv[++i]) exit(1);
                lport       = atoi(argv[i]);
                break;
            }
            case 'v': {
                verbose     = 1;
                break;
            }
            case 'k': {
                if(!argv[++i]) exit(1);
                cdkey       = argv[i];
                break;
            }
            case 'h': {
                if(!argv[++i]) exit(1);
                cdkeyhash   = argv[i];
                break;
            }
            case 'f': {
                if(!argv[++i]) exit(1);
                logfile     = argv[i];
                break;
            }
            case 's': {
                if(!argv[++i]) exit(1);
                gshost      = argv[i];
                break;
            }
            case 'i': {
                if(!argv[++i]) exit(1);
                gsa_id      = argv[i];
                break;
            }
            case 'a': {
                if(!argv[++i]) exit(1);
                gsa_ip      = argv[i];
                break;
            }
            case 'L': {
                if(!argv[++i]) exit(1);
                login_type  = atoi(argv[i]);
                if(!argv[++i]) exit(1);
                login_arg1  = argv[i];
                if(!argv[++i]) exit(1);
                login_arg2  = argv[i];
                break;
            }
            case 'D': {
                if(!argv[++i]) exit(1);
                drop_server = argv[i];
                break;
            }
            case 'd': {
                if(!argv[++i]) exit(1);
                drop_client = argv[i];
                break;
            }
            default: {
                printf("\nError: wrong argument (%s)\n", argv[i]);
                exit(1);
            }
        }
    }

    if(cdkeyhash && (strlen(cdkeyhash) != 32)) {
        printf("\nError: your cdkey hash is not 32 bytes long\n");
        exit(1);
    }
    if(logfile) {
        printf("- open file %s in append mode\n", logfile);
        fdlog = fopen(logfile, "ab");
        if(!fdlog) std_err();
    }
    seed = time(NULL);

    peer.sin_addr.s_addr = resolv(lhost);
    peer.sin_port        = htons(lport);
    peer.sin_family      = AF_INET;

    p = strchr(gshost, ':');
    if(p) {
        *p = 0;
        gsport = atoi(p + 1);
    }

    printf("- resolv hostname %s... ", gshost);
    gsip.s_addr = resolv(gshost);
    printf("ok\n");

    if(gsip.s_addr == inet_addr("127.0.0.1")) {
        if(gsport == lport) {
            printf("\n"
                "Error: you have peerchat.gamespy.com resolved as 127.0.0.1 and you use this\n"
                "       server running on the same destination port, so restore your hosts file\n");
            exit(1);
        } else {
            printf("\n"
                "Alert: you have peerchat.gamespy.com resolved as 127.0.0.1\n");
        }
    }

    printf("\n"
        "  gamename    %s\n"
        "  gamekey     %s\n"
        "  server      %s:%hu\n",
        gsgamename ? gsgamename : (u8 *)"valid random for each connection",
        gsgamekey  ? gsgamekey  : (u8 *)"valid random for each connection",
        inet_ntoa(gsip), gsport);

    printf(
        "  local       %s:%hu\n"
        "  cdkey       %s\n"
        "\n",
        inet_ntoa(peer.sin_addr), ntohs(peer.sin_port),
        cdkey ? cdkey : (u8 *)"random for each connection");

    sdl = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
    if(sdl < 0) std_err();
    if(setsockopt(sdl, SOL_SOCKET, SO_REUSEADDR, (char *)&on, sizeof(on))
      < 0) std_err();
    if(bind(sdl, (struct sockaddr *)&peer, sizeof(struct sockaddr_in))
      < 0) std_err();
    if(listen(sdl, SOMAXCONN)
      < 0) std_err();

    printf(
        "HOW TO USE\n"
        "==========\n"
        "1) launch this tool (done)\n"
        "2) open your preferred IRC client (or multiple clients)\n"
        "3) connect to %s port %hu\n"
        "\n"
        "Some tricks and notes:\n"
        "- when you do the /whois of an user you will see his encoded IP address which\n"
        "  you can decode using my peerchat_ip tool and the MD5 hash of his cdkey:\n"
        "  [nickname] (encoded_IP|Gamespy_ID@*): cdkey_hash\n"
        "\n"
        "- if you get banned for the usage of an invalid client, change your IP address\n"
        "  and choose another gamename and gamekey. If the problem continues contact me\n"
        "\n", lhost, lport);

    printf("- wait connections:\n");

    for(;;) {
        psz = sizeof(struct sockaddr_in);
        sda = accept(sdl, (struct sockaddr *)&peer, &psz);
        if(sda < 0) std_err();

        printf("  %s:%hu\n",
            inet_ntoa(peer.sin_addr), ntohs(peer.sin_port));

        if(!quick_threadx(conn, (void *)sda)) close(sda);
    }

    close(sdl);
    if(fdlog) fclose(fdlog);
    return(0);
}



void gslist_keys_list(void) {
    int     i;

    for(i = 0; gslist_keys[i][0]; i++) {
        printf("%-53s %-18s %s\n", gslist_keys[i][0], gslist_keys[i][1], gslist_keys[i][2]);
    }
    printf("\n");
}



quick_thread(conn, int sock) {
    static struct   linger  ling = {1,1};
    gs_peerchat_ctx client,
                    server;
    struct  sockaddr_in peer;
    fd_set  rset;
    int     sd,
            t,
            selsock,
            cmd,
            getckey,
            csize   = BUFFSZ,
            ssize   = BUFFSZ,
            clen,       // client length
            slen,       // server length
            xlen;       // tempoary length
    u8      tmp[16],
            hash[33],
            *gamename,
            *gamekey,
            *cbuff,     // client buffer
            *sbuff,     // server buffer
            *xbuff,     // only for temporary stuff
            *data,
            *p;

    #define SNPRINTF_LEN(X,Y,Z) \
    if((Z < 0) || (Z >= Y)) { \
        X[Y] = 0; \
        Z = Y; \
    }

    cbuff = malloc(csize + 3);
    sbuff = malloc(ssize + 3);
    xbuff = malloc(BUFFSZ + 3);
    if(!cbuff || !sbuff || !xbuff) std_err();

    gamename = gsgamename;
    gamekey  = gsgamekey;
    if(!gamename || !gamekey) {
        t = rnd_gamekey();
        gamename = gslist_keys[t][1];
        gamekey  = gslist_keys[t][2];
        printf("- use the gamekey of %s\n", gslist_keys[t][0]);
    }

    peer.sin_addr.s_addr = gsip.s_addr;
    peer.sin_port        = htons(gsport);
    peer.sin_family      = AF_INET;

    sd = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
    if(sd < 0) std_err();
    setsockopt(sd, SOL_SOCKET, SO_LINGER, (char *)&ling, sizeof(ling));
    if(connect(sd, (struct sockaddr *)&peer, sizeof(struct sockaddr_in)) < 0) {
        printf("\nError: connection refused\n");
        goto quit;
    }

    if(cdkeyhash) {
        strcpy(hash, cdkeyhash);
    } else {
        if(cdkey) {
            do_md5_key_hash(cdkey, hash);
        } else {
            rnds(xbuff, 16);
            do_md5_key_hash(xbuff, hash);
        }
    }

        /* CRYPT */

    clen = snprintf(cbuff, csize, "CRYPT des 1 %s", gamename);
    SNPRINTF_LEN(cbuff, csize, clen)
    if(mysend(sd, cbuff, clen, NULL) <= 0) goto quit;

    slen = myrecv(sd, sbuff, ssize);
    if(slen < 0) goto quit;
    if(gs_keys(gamekey, sbuff, slen, &client, &server) < 0) goto quit;

        /* LOGIN */

    if(login_type) {
        if(login_type == 1) {
            clen = snprintf(cbuff, csize, "LOGIN 1 * %s :%s", do_md5_auto_check(login_arg1), login_arg2);
        } else if(login_type == 4) {
            clen = snprintf(cbuff, csize, "LOGIN 4 %s %s", login_arg1, do_md5_auto_check(login_arg2));
        } else {
            printf("\n"
                "Error: only LOGIN 1 and 4 are supported at the moment\n"
                "       if you know other version please contact me\n");
            exit(1);
        }
        SNPRINTF_LEN(cbuff, csize, clen)
        if(mysend(sd, cbuff, clen, &client) <= 0) goto quit;
    }

        /* USRIP */

    clen = sprintf(cbuff, "USRIP");
    if(mysend(sd, cbuff, clen, &client) <= 0) goto quit;

    selsock = 1 + ((sock > sd) ? sock : sd);
    clen    = 0;
    slen    = 0;
    getckey = 0;

    for(;;) {
        FD_ZERO(&rset);
        FD_SET(sock, &rset);
        FD_SET(sd,   &rset);
        if(select(selsock, &rset, NULL, NULL, NULL)
          < 0) goto quit;

        if(FD_ISSET(sd, &rset)) {                   // SERVER
            t = recv1(sd, sbuff + slen, &server);
            if(t < 0) goto quit;
            slen++;
            if(t) {
                sbuff[slen] = 0;
                slen = delimit(sbuff);

                data = get_irc_command(sbuff, &cmd);
                switch(cmd) {
                    case 705: {
                        if(gs_keys(gamekey, sbuff, slen, &client, &server) < 0) goto quit;
                        break;
                    }
                    case 302: {
                        p = strchr(sbuff, '@');
                        p = p ? (p + 1) : data;
                        xlen = snprintf(xbuff, BUFFSZ,
                            "USER %s%s%s|%s %s %s :%s",
                            gsa_ip ? "" : "X",
                            gsa_ip ? gsa_ip : (u8 *)peerchat_ip_encoder(inet_addr(p), 0),
                            gsa_ip ? "" : "X",
                            gsa_id,
                            "127.0.0.1",
                            gshost,
                            hash);
                        SNPRINTF_LEN(xbuff, BUFFSZ, xlen)
                        if(mysend(sd, xbuff, xlen, &client) <= 0) goto quit;
                        break;
                    }
                    default: {
                        sprintf(tmp, "%u", cmd);
                        if(check_drops(tmp, strlen(tmp), drop_server) < 0) slen = 0;
                        break;
                    }
                }

                remove_ctcp(sbuff, slen);

                if(slen) {  // in case of easy dropping
                    if(mysend(sock, sbuff, slen, NULL) <= 0) goto quit;
                    slen = 0;
                }
            }

        } else if(FD_ISSET(sock, &rset)) {          // CLIENT
            t = recv1(sock, cbuff + clen, NULL);
            if(t < 0) goto quit;
            clen++;
            if(t) {
                cbuff[clen] = 0;
                clen = delimit(cbuff);

                data = cbuff + strcspn(cbuff, " \r\n");
                if(data) {
                    if(
                      !mystrnicmp(cbuff, "CAP",   data - cbuff) ||
                      !mystrnicmp(cbuff, "USER",  data - cbuff) ||
                      !mystrnicmp(cbuff, "ERROR", data - cbuff)) {
                        printf("- %.*s command dropped\n", data - cbuff, cbuff);
                        clen = 0;
                    } else if(!mystrnicmp(cbuff, "WHO", data - cbuff)) {
                        printf("- WHO command converted in GETCKEY\n");
                        data++;
                        xlen = snprintf(xbuff, BUFFSZ,
                            "GETCKEY %.*s * %03d 0 :\\username\\b_flags",
                            strcspn(data, " \r\n"), data,
                            getckey % 1000);
                        SNPRINTF_LEN(xbuff, BUFFSZ, xlen)
                        clen = xlen;
                        strcpy(cbuff, xbuff);
                        getckey++;
                    }
                    if(check_drops(cbuff, data - cbuff, drop_client) < 0) clen = 0;
                }

                if(clen) {  // in case of easy dropping
                    if(mysend(sd, cbuff, clen, &client) <= 0) goto quit;
                    clen = 0;
                }
            }
        }

        if(clen >= csize) {
            csize += BUFFSZ;
            cbuff = realloc(cbuff, csize + 3);
            if(!cbuff) goto quit;
        }
        if(slen >= ssize) {
            ssize += BUFFSZ;
            sbuff = realloc(sbuff, ssize + 3);
            if(!sbuff) goto quit;
        }
    }

quit:
    printf("- disconnected\n");
    close(sock);
    close(sd);
    free(cbuff);
    free(sbuff);
    free(xbuff);
    return(0);
}



int check_drops(u8 *data, int datalen, u8 *dropstr) {
    int     len;
    u8      *p,
            *l;

    if(!dropstr) return(0);

    for(p = l = dropstr; l; p = l + 1) {
        l = strchr(p, ',');
        len = l ? (l - p) : strlen(p);
        if(datalen != len) continue;
        if(!mystrnicmp(data, p, datalen)) return(-1);
    }
    return(0);
}



int rnd_gamekey(void) {
    int     i;

    for(i = 0; gslist_keys[i][0]; i++);

    seed = (seed * 0x343FD) + 0x269EC3;
    return(seed % i);
}



int rnds(u8 *data, int len) {
    int     i;
    static const u8 table[] =
            "0123456789"
            "ABCDEFGHIJKLMNOPQRSTUVWXYZ"
            "abcdefghijklmnopqrstuvwxyz";

    for(i = 0; i < len; i++) {
        seed = ((seed * 0x343FD) + 0x269EC3) >> 3;
        data[i] = table[seed % (sizeof(table) - 1)];
    }
    data[i] = 0;
    return(len);
}



void remove_ctcp(u8 *buff, int buffsz) {
    int     i,
            len;
    u8      *ctcp;

    len  = 0;
    ctcp = NULL;

    for(i = 0; i < buffsz; i++) {
        if(buff[i] == 0x01) {
            if(ctcp) {
                len = (buff + i) - ctcp;
                break;
            }
            ctcp = buff + i + 1;
        }
    }
    if(!ctcp || !len) return;

    if(
      !mystrnicmp(ctcp, "FINGER",     len) ||
      !mystrnicmp(ctcp, "VERSION",    len) ||
      !mystrnicmp(ctcp, "SOURCE",     len) ||
      !mystrnicmp(ctcp, "USERINFO",   len) ||
      !mystrnicmp(ctcp, "CLIENTINFO", len) ||
      !mystrnicmp(ctcp, "ERRMSG",     len) ||
      !mystrnicmp(ctcp, "PING",       len) ||
      !mystrnicmp(ctcp, "TIME",       len)) {
        ctcp[-1]  = '\"';
        ctcp[len] = '\"';
    }
}



int delimit(u8 *data) {
    u8      *p;

    for(p = data; *p && (*p != '\r') && (*p != '\n'); p++);
    *p = 0;
    return(p - data);
}



int mystrnicmp(u8 *buff, u8 *str, int len) {
    if(strlen(str) != len) return(-1);
    if(strnicmp(buff, str, len)) return(-1);
    return(0);
}



u8 *get_irc_command(u8 *buff, int *cmd) {
    u8      *p,
            *c;

    *cmd = 0;
    p = strchr(buff, ' ');
    if(!p) return(buff);
    p++;

    *cmd = atoi(p);
    if(!*cmd) return(buff);

    c = strchr(p, ' ');
    if(!c) return(p);
    return(c + 1);
}



u8 *do_md5_auto_check(u8 *data) {
    md5_context ctx;
    int     i;
    static  u8  ret[33];
    u8      digest[16],
            *p;

    if(strlen(data) == 32) {    // already the hash
        for(i = 0; i < 32; i++) {
            ret[i] = tolower(data[i]);
        }
        ret[i] = 0;
        return(ret);
    }

    md5_starts(&ctx);
    md5_update(&ctx, data, strlen(data));
    md5_finish(&ctx, digest);

    p = ret;
    for(i = 0; i < 16; i++) {
        *p++ = hex[digest[i] >> 4];
        *p++ = hex[digest[i] & 15];
    }
    *p = 0;
    return(ret);
}



void do_md5_key_hash(u8 *key, u8 *ret) {
    md5_context ctx;
    int     i;
    u8      tmpkey[33],
            digest[16],
            *p;

    p = tmpkey;
    for(i = 0; key[i]; i++) {
        if(key[i] == '-') continue;
        *p++ = key[i];
        if((p - tmpkey) >= sizeof(tmpkey)) break;
    }
    *p = 0;

    md5_starts(&ctx);
    md5_update(&ctx, tmpkey, strlen(tmpkey));
    md5_finish(&ctx, digest);

    p = ret;
    for(i = 0; i < 16; i++) {
        *p++ = hex[digest[i] >> 4];
        *p++ = hex[digest[i] & 15];
    }
    *p = 0;
}



int mysend(int sd, u8 *buff, int len, gs_peerchat_ctx *enc) {   // sends one command
    int     i;

    // max GS compatibility!
    for(i = 0; i < len; i++) {
        if((buff[i] == '\r') || (buff[i] == '\n')) break;
    }
    len = i;
    buff[len++] = '\r';
    buff[len++] = '\n';

    if(verbose) printf("%.*s", len, buff);
    if(fdlog) {
        fprintf(fdlog, "%.*s", len, buff);
        fflush(fdlog);
    }

    if(enc) gs_peerchat(enc, buff, len);
    return(send(sd, buff, len, 0));
}



int gs_keys(u8 *gamekey, u8 *buff, int len, gs_peerchat_ctx *client, gs_peerchat_ctx *server) {
    int     i;
    u8      *p;

    buff[len] = 0;
    for(p = buff, i = 0; i < 3; i++, p++) {
        p = strchr(p, ' ');
        if(!p) {
            printf("\nError: invalid server IRC response, no challenges\n");
            return(-1);
        }
    }

    printf(
        "- client encryption key   %.16s\n"
        "- server encryption key   %.16s\n",
        p, p + 17);

    gs_peerchat_init(client, p,      gamekey);
    gs_peerchat_init(server, p + 17, gamekey);
    return(0);
}



int recv1(int sd, u8 *data, gs_peerchat_ctx *enc) {
    if(recv(sd, data, 1, 0) <= 0) return(-1);
    if(enc) gs_peerchat(enc, data, 1);
    if(*data == '\n') return(1);
    return(0);
}



int myrecv(int sd, u8 *buff, int buffsz) {
    int     t,
            len;

    len = 0;
    while(len < buffsz) {
        t = recv1(sd, buff + len, NULL);
        if(t < 0) return(-1);
        len++;
        if(t) break;
    }

    if(verbose) printf("%.*s", len, buff);
    if(fdlog) {
        fprintf(fdlog, "%.*s", len, buff);
        fflush(fdlog);
    }
    return(len);
}



u32 resolv(char *host) {
    struct  hostent *hp;
    u32     host_ip;

    host_ip = inet_addr(host);
    if(host_ip == INADDR_NONE) {
        hp = gethostbyname(host);
        if(!hp) {
            printf("\nError: Unable to resolve hostname (%s)\n", host);
            exit(1);
        }
        host_ip = *(u32 *)hp->h_addr;
    }
    return(host_ip);
}



#ifndef WIN32
    void std_err(void) {
        perror("\nError");
        exit(1);
    }
#endif



