/*

    Copyright 2004,2005,2006 Luigi Auriemma

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    http://www.gnu.org/licenses/gpl.txt

*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include "gs_peerchat.h"

#ifdef WIN32
    #include <winsock.h>
    #include <malloc.h>
    #include <direct.h>
    #include "winerr.h"

    #define close   closesocket
    #define sleep   Sleep
    #define ONESEC  1000
    #define APPDATA "Software\\Microsoft\\Windows\\CurrentVersion\\Explorer\\Shell Folders"
    DWORD   tid;
    HANDLE  th;
#else
    #include <unistd.h>
    #include <sys/socket.h>
    #include <sys/types.h>
    #include <arpa/inet.h>
    #include <netinet/in.h>
    #include <netdb.h>
    #include <sys/stat.h>
    #include <pthread.h>

    #define stristr strcasestr
    #define ONESEC  1
    pthread_t   tid;
    int         th;
#endif

#ifdef WIN32
    #define quick_thread(NAME, ARG) DWORD WINAPI NAME(ARG)
    #define thread_id   DWORD
#else
    #define quick_thread(NAME, ARG) void *NAME(ARG)
    #define thread_id   pthread_t
#endif

thread_id quick_threadx(void *func, void *data) {
    thread_id       tid;
#ifdef WIN32
    if(!CreateThread(NULL, 0, func, data, 0, &tid)) return(0);
#else
    pthread_attr_t  attr;
    pthread_attr_init(&attr);
    pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_DETACHED);
    if(pthread_create(&tid, &attr, func, data)) return(0);
#endif
    return(tid);
}



#define VER         "0.3a"
#define BUFFSZ      2048
#define PEERCHAT    "peerchat.gamespy.com"
#define PEERCHATFD  "peerchat_proxy.ip"
#define PEERCHATLOG "peerchat_proxy.log"
#define PORT        6667

#define GSMAXPATH   256
#define GSLIST      "gslist.cfg"
#define GSLISTSZ    80
#define CNAMEOFF    54
#define CKEYOFF     73



quick_thread(client, int sda);
int recv_tcp(int sock, u_char *data, int len);
void delimit(u_char *data);
void verify_gslist(void);
int get_key(u_char *gamename, u_char *gamekey);
FILE *gslfopen(const char *path, const char *mode);
u_int check_ip(void);
u_int resolv(char *host);
void std_err(void);



struct  sockaddr_in peer;
FILE    *fdlog;
char    gslist_path[GSMAXPATH + 1];



int main(int argc, char *argv[]) {
    struct  sockaddr_in peerl;
    time_t  datex;
    struct  tm  *tmx;
    int     sdl,
            sda,
            on = 1,
            psz;
    u_short port  = PORT,
            lport = PORT;

#ifdef WIN32
    WSADATA    wsadata;
    WSAStartup(MAKEWORD(1,0), &wsadata);
#endif


    setbuf(stdout, NULL);

    fputs("\n"
        "GS peerchat proxy decrypter "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", stdout);

    *gslist_path = 0;
    verify_gslist();

    peer.sin_addr.s_addr  = check_ip();
    peer.sin_port         = htons(port);
    peer.sin_family       = AF_INET;

    peerl.sin_addr.s_addr = INADDR_ANY;
    peerl.sin_port        = htons(lport);
    peerl.sin_family      = AF_INET;

    psz                   = sizeof(peerl);

    printf(
        "\n"
        "- Server: %s:%hu\n"
        "- Listen on TCP port %hu\n",
        inet_ntoa(peer.sin_addr), port,
        lport);

    sdl = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
    if(sdl < 0) std_err();
    if(setsockopt(sdl, SOL_SOCKET, SO_REUSEADDR, (char *)&on, sizeof(on))
      < 0) std_err();
    if(bind(sdl, (struct sockaddr *)&peerl, sizeof(peerl))
      < 0) std_err();
    if(listen(sdl, SOMAXCONN)
      < 0) std_err();

    fputs("- open log file "PEERCHATLOG"\n", stdout);
    fdlog = fopen(PEERCHATLOG, "ab");
    if(!fdlog) std_err();

    time(&datex);
    tmx = localtime(&datex);
    fprintf(fdlog,
        "----------------------------------\n"
        "Log file appended at:\n"
        "  %02hhu/%02hhu/%02hhu   (dd/mm/yy)\n"
        "  %02hhu:%02hhu:%02hhu\n"
        "----------------------------------\n"
        "\n",
        tmx->tm_mday, tmx->tm_mon + 1, tmx->tm_year % 100,
        tmx->tm_hour, tmx->tm_min, tmx->tm_sec);
    fflush(fdlog);
    fputs(
        "  you can open this file in any moment with a text editor since it's updated\n"
        "  in real-time with the decrypted data\n", stdout);

    fputs("- wait connection from your game client:\n", stdout);

    for(;;) {
        sda = accept(sdl, (struct sockaddr *)&peerl, &psz);
        if(sda < 0) std_err();

        printf("- connection from %s:%hu\n",
            inet_ntoa(peerl.sin_addr), ntohs(peerl.sin_port));

        if(!quick_threadx(client, (void *)sda)) close(sda);
    }

    fclose(fdlog);
    close(sdl);
    return(0);
}



quick_thread(client, int sda) {
    gs_peerchat_ctx client,
                    server;
    fd_set  rs;
    int     sd,
            selsock,
            i,
            len;
    u_char  buff[BUFFSZ + 1],
            gamekey[7],
            *p,
            *l;

    sd = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
    if(sd < 0) std_err();
    if(connect(sd, (struct sockaddr *)&peer, sizeof(peer))
      < 0) std_err();

    len = recv_tcp(sda, buff, sizeof(buff));        // get client's data
    if(len <= 0) goto quit;
    if(send(sd, buff, len, 0)
      < 0) goto quit;
    fwrite(buff, len, 1, fdlog);
    fflush(fdlog);

    l = strchr(buff, ' ');
    if(l && !memcmp(buff, "CRYPT", l - buff)) {     // manage the client's data
        p = l + 1;
        l = strchr(p, ' ');
        if(!l || memcmp(p, "des", l - p)) {
            fputs("  Error: no des field received\n", stdout);
            goto quit;
        }

        p = l + 1;
        l = strchr(p, ' ');
        if(!l) {
            fputs("  Error: no type received\n", stdout);
            goto quit;
        }

        for(p = ++l; *l > '\r'; l++);
        *l = 0;
        printf("  gamename: %s   ", p);

        if(!get_key(p, gamekey)) {
            fputs("  Error: no gamekey found for this game\n", stdout);
            goto quit;
        }
        printf("gamekey: %s\n", gamekey);
        fprintf(fdlog, "\n### GAMEKEY: %s\n", gamekey);
        fflush(fdlog);

        len = recv_tcp(sd, buff, sizeof(buff));      // get server's data
        if(len <= 0) goto quit;
        if(send(sda, buff, len, 0)
          < 0) goto quit;
        fwrite(buff, len, 1, fdlog);
        fflush(fdlog);

        if(*buff != ':') {
            printf("\n"
            "Error: received a bad reply from the peerchat server, check it:\n"
            "\n"
            "%s\n"
            "\n", buff);
            goto quit;
        }
            /* :s 705 * chall1 chall2 */
        delimit(buff);
        for(p = buff, i = 0; i < 4; i++, p++) {
            if(i == 3) l = p;
            p = strchr(p, ' ');
            if(!p) {
                printf("\n"
                    "Error: invalid server IRC response:\n"
                    "\n"
                    "%s\n"
                    "\n", buff);
                goto quit;
            }
            *p = 0;
        }

        fprintf(fdlog,
            "### CLIENT CHALLENGE: %s\n"
            "### SERVER CHALLENGE: %s\n",
            l, p);
        fflush(fdlog);
        gs_peerchat_init(&client, p, gamekey);
        gs_peerchat_init(&server, l, gamekey);

    } else {
        fputs("- this client doesn't use encryption\n", stdout);
        memset(server.gs_peerchat_crypt, 0, sizeof(server.gs_peerchat_crypt));
        server.gs_peerchat_1 = 0;
        server.gs_peerchat_2 = 0;
        memset(client.gs_peerchat_crypt, 0, sizeof(client.gs_peerchat_crypt));
        client.gs_peerchat_1 = 0;
        client.gs_peerchat_2 = 0;
    }

    selsock = ((sda > sd) ? sda : sd) + 1;

    for(;;) {
        FD_ZERO(&rs);
        FD_SET(sda, &rs);
        FD_SET(sd,  &rs);
        if(select(selsock, &rs, NULL, NULL, NULL)
          < 0) break;

        if(FD_ISSET(sd, &rs)) {
            len = recv(sd, buff, sizeof(buff), 0);
            if(len <= 0) goto quit;
            if(send(sda, buff, len, 0)
              < 0) goto quit;
            gs_peerchat(&client, buff, len);

        } else if(FD_ISSET(sda, &rs)) {
            len = recv(sda, buff, sizeof(buff), 0);
            if(len <= 0) goto quit;
            if(send(sd, buff, len, 0)
              < 0) goto quit;
            gs_peerchat(&server, buff, len);
        }

        fwrite(buff, len, 1, fdlog);
        fflush(fdlog);
    }

quit:
    shutdown(sda, 2);
    shutdown(sd, 2);
    close(sda);
    close(sd);
    fputs("- disconnected\n", stdout);
    fputs("\n### DISCONNECTED\n", fdlog);
    fflush(fdlog);
    return(0);
}



void delimit(u_char *data) {
    while(*data && (*data != '\r') && (*data != '\n')) data++;
    *data = 0;
}



int recv_tcp(int sock, u_char *data, int len) {
    u_char  *p,
            *limit;

    p = data;
    limit = data + len - 1;
    while(p < limit) {
        if(recv(sock, p, 1, 0) <= 0) return(-1);
        if(*p++ == '\n') break;
    }
    *p = 0;

    return(p - data);
}



void verify_gslist(void) {
    FILE    *fd;

    fd = gslfopen(GSLIST, "rb");
    if(!fd) fd = fopen(GSLIST, "rb");
    if(!fd) {
        fputs("\n"
            "--------------------------------\n"
            "File "GSLIST" not found.\n"
            "You can download it here:\n"
            "\n"
            "  http://aluigi.org/papers/gslist.cfg\n"
            "\n", stdout);
        fgetc(stdin);
        exit(1);
    }
    fclose(fd);
}



int get_key(u_char *gamename, u_char *gamekey) {
    FILE    *fd;
    u_char  buff[GSLISTSZ + 1],
            *p,
            *l;

    fd = gslfopen(GSLIST, "rb");
    if(!fd) fd = fopen(GSLIST, "rb");
    if(!fd) {
        fputs("\n"
            "Error: "GSLIST" not found in the current directory\n"
            "       Download it from here:\n"
            "\n"
            "         http://aluigi.org/papers/gslist.cfg\n"
            "\n", stderr);
        fgetc(stdin);
        exit(1);
    }
    p = buff + CNAMEOFF;
    while(fgets(buff, sizeof(buff), fd)) {
        l = strchr(p, ' ');
        if(l) *l = 0;
        if(!strcmp(gamename, p) && (buff[CKEYOFF] > ' ')) {
            fclose(fd);
            memcpy(gamekey, buff + CKEYOFF, 6);
            gamekey[6] = 0;
            return(1);
        }
    }

    fclose(fd);
    return(0);
}



FILE *gslfopen(const char *path, const char *mode) {
#ifdef WIN32
    HKEY        key;
    char        home[GSMAXPATH + 1];
    int         len;
#else
    const char  *home;
#endif
    char        retpath[GSMAXPATH + 64 + 1];

    if(!*gslist_path) {
#ifdef WIN32
        len = GSMAXPATH;
        if(!RegOpenKeyEx(HKEY_CURRENT_USER, APPDATA, 0, KEY_READ, &key)) {
            if(!RegQueryValueEx(key, "AppData", NULL, NULL, home, (void *)&len)) {
                sprintf(gslist_path, "%s\\gslist\\", home);
                mkdir(gslist_path);
            }
            RegCloseKey(key);
        }
        if(!*gslist_path) strcpy(gslist_path, ".\\");
#else
        home = getenv("HOME");
        if(!home) {
            fputs("\n"
                "Error: impossible to know your $HOME or Application Data directory where\n"
                "       reading/writing the Gslist configuration files.\n"
                "       Modify the source code of the program or report the problem to me.\n"
                "\n", stderr);
            exit(1);
        }
        sprintf(gslist_path, "%s/.gslist/", home);
        mkdir(gslist_path, 0755);
#endif
    }

    sprintf(retpath, "%s%s", gslist_path, path);

    return(fopen(retpath, mode));
}



u_int check_ip(void) {
    FILE    *fd;
    char    directory[100],
            *progress = "/-\\|";
    int     ipresult = 1;   // 0 = 127.0.0.1, 1 = REAL
    u_int  localhost,
            the_ip;

    localhost = inet_addr("127.0.0.1");
    if(resolv(PEERCHAT) == localhost) ipresult = 0;

    fd = fopen(PEERCHATFD, "rb");
    if(!fd) {
        if(!ipresult) {
            printf("\n"
                "------------------------------------------------------------------------------\n"
                "You have set %s to IP 127.0.0.1 and the file %s doesn't exist in the current directory (%s).\n"
                "The file %s is used to keep the real IP of the server in memory so you must execute the next operations only this time and no more.\n"
                "\n"
                "Now EDIT or CREATE your hosts file removing (or adding a # before) the line\n"
                "  \"127.0.0.1   %s\"\n"
                "The hosts file has usually one of the following paths:\n"
                "- c:\\windows\\hosts\n"
                "- c:\\winnt\\system32\\drivers\\etc\\hosts\n"
                "- c:\\windows\\system32\\drivers\\etc\\hosts\n"
                "- /etc/hosts\n"
                "Do it, I will check it each second until it will be ready:\n",
                PEERCHAT,
                PEERCHATFD,
                getcwd(directory, sizeof(directory) - 1),
                PEERCHATFD,
                PEERCHAT);
        }
        do {
            sleep(ONESEC);
            fputc(*progress++, stdout);
            fputc('\b', stdout);
            if(!*progress) progress -= 4;
            the_ip = resolv(PEERCHAT);
        } while(the_ip == localhost);

        printf("\n"
            "------------------------------------------------------------------------------\n"
            "The real IP of %s will now be saved in the file %s\n",
            PEERCHAT,
            PEERCHATFD);
        fd = fopen(PEERCHATFD, "wb");
        if(!fd) std_err();
        fwrite((void *)&the_ip, 4, 1, fd);
        fclose(fd);

        printf("\n"
            "------------------------------------------------------------------------------\n"
            "Well done, the IP of %s is %s\n"
            "Now you must EDIT or CREATE your hosts file adding:\n"
            "\n"
            "   127.0.0.1   %s\n"
            "\n"
            "The hosts file has usually one of the following paths:\n"
            "- c:\\windows\\hosts\n"
            "- c:\\winnt\\system32\\drivers\\etc\\hosts\n"
            "- c:\\windows\\system32\\drivers\\etc\\hosts\n"
            "- /etc/hosts\n"
            "I will check it each second as before:\n",
            PEERCHAT,
            inet_ntoa(*(struct in_addr *)&the_ip),
            PEERCHAT);
        do {
            sleep(ONESEC);
            fputc(*progress++, stdout);
            fputc('\b', stdout);
            if(!*progress) progress -= 4;
        } while(resolv(PEERCHAT) != localhost);
    } else {
        if(ipresult) {
            the_ip = resolv(PEERCHAT);
            printf("\n"
                "------------------------------------------------------------------------------\n"
                "The IP of %s is %s\n"
                "Now you must EDIT or CREATE the hosts file adding:\n"
                "\n"
                "   127.0.0.1   %s\n"
                "\n"
                "The hosts file has usually one of the following paths:\n"
                "- c:\\windows\\hosts\n"
                "- c:\\winnt\\system32\\drivers\\etc\\hosts\n"
                "- c:\\windows\\system32\\drivers\\etc\\hosts\n"
                "- /etc/hosts\n"
                "I will check the operation automatically each second until the file will be ready:\n",
                PEERCHAT,
                inet_ntoa(*(struct in_addr *)&the_ip),
                PEERCHAT);
            do {
                sleep(ONESEC);
                fputc(*progress++, stdout);
                fputc('\b', stdout);
                if(!*progress) progress -= 4;
            } while(resolv(PEERCHAT) != localhost);
        } else fread((void *)&the_ip, 4, 1, fd);
        fclose(fd);
    }
    return(the_ip);
}



u_int resolv(char *host) {
    struct hostent *hp;
    u_int host_ip;

    host_ip = inet_addr(host);
    if(host_ip == INADDR_NONE) {
        hp = gethostbyname(host);
        if(!hp) {
            printf("\nError: Unable to resolve hostname (%s)\n", host);
            exit(1);
        } else host_ip = *(u_int *)hp->h_addr;
    }
    return(host_ip);
}



#ifndef WIN32
    void std_err(void) {
        perror("\nError");
        exit(1);
    }
#endif



