/*
    Copyright 2005,2006,2007 Luigi Auriemma

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    http://www.gnu.org/licenses/gpl.txt
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <pcap.h>
#include "ip.h"
#include "ether_hdrlen.h"
#include "gs_peerchat.h"

#ifdef WIN32
    #include <winsock.h>

    #define APPDATA "Software\\Microsoft\\Windows\\CurrentVersion\\Explorer\\Shell Folders"
#else
    #include <unistd.h>
    #include <sys/socket.h>
    #include <sys/types.h>
    #include <sys/param.h>
    #include <arpa/inet.h>
    #include <netinet/in.h>
    #include <netdb.h>
    #include <sys/stat.h>
#endif



#define VER         "0.2"
#define HOST        "peerchat.gamespy.com"
#define PORT        6667

#define GSMAXPATH   256
#define GSLIST      "gslist.cfg"
#define GSLISTSZ    80
#define CNAMEOFF    54
#define CKEYOFF     73



void pcap_error(pcap_if_t *alldevs, char *err);
pcap_t *pcap_get_device(void);
void pcaploop(uint8_t *ignore, const struct pcap_pkthdr *hdr, const uint8_t *pkt);
void peerchat_do_it(uint8_t flags, uint32_t saddr, uint16_t sport, uint32_t daddr, uint16_t dport, uint8_t *data, int len);
int get_key(uint8_t *gamename, uint8_t *gamekey);
FILE *gslfopen(const char *path, const char *mode);
u_int resolv(char *host);



gs_peerchat_ctx client,
                server;
FILE        *gslist_fd;
uint32_t    sniffhost,
            sniffclient;
static int  if_offset;
uint16_t    sniffport;
uint8_t     gamekey[7],
            gslist_path[GSMAXPATH + 1];



int main(void) {
    pcap_t  *fp;

#ifdef WIN32
    WSADATA wsadata;
    WSAStartup(MAKEWORD(1,0), &wsadata);
#endif

    setbuf(stdout, NULL);
    setbuf(stderr, NULL);

    fputs("\n"
        "GS Peerchat sniffer and decrypter "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n"
        "Notes:\n"
        "- one client at time supported\n"
        "- You need to have gslist.cfg in the Gslist or current folder\n"
        "\n", stderr);

    gslist_fd = gslfopen(GSLIST, "rb");
    if(!gslist_fd) gslist_fd = fopen(GSLIST, "rb");
    if(!gslist_fd) {
        fprintf(stderr, "\n"
            "Error: "GSLIST" not found in the current directory\n"
            "       Download it from here:\n"
            "\n"
            "         http://aluigi.org/papers/gslist.cfg\n"
            "\n");
        fgetc(stdin);
        exit(1);
    }

    fprintf(stderr, "- resolv hostname %s --> ", HOST);
    sniffhost = resolv(HOST);
    sniffport = htons(PORT);
    fprintf(stderr, "%s : %hu\n",
        inet_ntoa(*(struct in_addr *)&sniffhost), PORT);

    sniffclient  = 0;
    *gslist_path = 0;

    fp = pcap_get_device();

    fprintf(stderr, "- ready:\n\n");
    pcap_loop(fp, 0, pcaploop, NULL);

    return(0);
}



void pcap_error(pcap_if_t *alldevs, char *err) {
    fprintf(stderr, "\nError: %s\n", err);
    if(alldevs) pcap_freealldevs(alldevs);
    exit(1);
}



pcap_t *pcap_get_device(void) {
    pcap_t      *fp;
    pcap_if_t   *alldevs,
                *d;
    int         i,
                inum;
    char        errbuf[PCAP_ERRBUF_SIZE];

    if(pcap_findalldevs(&alldevs, errbuf) < 0) pcap_error(alldevs, errbuf);

    fprintf(stderr, "- interfaces:\n\n");
    for(i = 0, d = alldevs; d; d = d->next) {
        fprintf(stderr, "%-2d %s\n", ++i, d->name);
        if(d->description) fprintf(stderr, "   %s\n", d->description);
    }

    if(i > 1) {
        fprintf(stderr, "\n- enter the interface number (1 - %d): ", i);
        scanf("%d", &inum);
        if(inum < 1 || inum > i) pcap_error(alldevs, "interface number out of range");
    } else if(!i) {
        pcap_error(alldevs,
            "No interfaces found or you don't have the needed permissions.\n"
            "       Make sure Pcap/WinPcap is installed and you are root or admin");
    } else {
        inum = 1;
    }

    for(inum--, d = alldevs, i = 0; i < inum; i++, d = d->next);
    fprintf(stderr, "- adapter to use: %s\n", d->name);

    fp = pcap_open_live(d->name, 0xffff, 1, 100, errbuf);
    if(!fp) pcap_error(alldevs, errbuf);

    pcap_freealldevs(alldevs);

    if_offset = pcap_hdrlen(pcap_datalink(fp));
    if(if_offset < 0) pcap_error(alldevs, "pcap interface not supported by this tool");

    return(fp);
}



void pcaploop(uint8_t *ignore, const struct pcap_pkthdr *hdr, const uint8_t *pkt) {
    struct  iphdr   *ip;
    struct  tcphdr  *tcp;
    int         iflen,
                iplen,
                tcplen,
                len;
    uint8_t     *data;

    iflen  = ntohs(*(uint16_t *)(pkt + if_offset - 2));
    ip     = (struct iphdr *)(pkt + if_offset + ether_hdrlen(iflen));
    if((ip->ihl_ver >> 4) != 4) return;
    iplen  = (ip->ihl_ver & 15) << 2;

    if(ip->protocol != IPPROTO_TCP) return;

    tcp    = (struct tcphdr *)((uint8_t *)ip + iplen);
    tcplen = tcp->doff >> 2;

    // if(!(tcp->flags & TH_PSH)) return;      /* needs to be handled later */

    data   = (uint8_t *)((uint8_t *)tcp + tcplen);
    len    = ntohs(ip->tot_len) - (data - (uint8_t *)ip);

    // if(len <= 0) return;                    /* needs to be handled later */

    peerchat_do_it(tcp->flags, ip->saddr, tcp->source, ip->daddr, tcp->dest, data, len);
}



void peerchat_do_it(uint8_t flags, uint32_t saddr, uint16_t sport, uint32_t daddr, uint16_t dport, uint8_t *data, int len) {
    uint8_t     *p,
                *l;

    if((saddr != sniffhost) && (daddr != sniffhost)) return;
    if((sport != sniffport) && (dport != sniffport)) return;

    if(flags & TH_FIN) {
        if(sniffclient && (saddr != sniffclient)) {
            printf("- disconnected\n");
            sniffclient = 0;
        }
        return;
    }
    if(!(flags & TH_PSH)) return;
    if(len <= 0) return;

    if(!sniffclient) {
        fwrite(data, len, 1, stdout);
        data[len] = 0;

        if(sport == sniffport) {      // SERVER
            l = strchr(data, ' ');
            if(!l) return;
            p = l + 1;

            l = strchr(p, ' ');
            if(!l) return;
            if(!strncmp(p, "705", l - p)) {
                l = strchr(l + 1, ' ');     // *
                if(!l) return;

                p = l + 1;
                l = strchr(p, ' ');
                if(!l) return;

                gs_peerchat_init(&client, p, gamekey);
                gs_peerchat_init(&server, l + 1, gamekey);

            } else {
                memset(server.gs_peerchat_crypt, 0, sizeof(server.gs_peerchat_crypt));
                server.gs_peerchat_1 = 0;
                server.gs_peerchat_2 = 0;
                memset(client.gs_peerchat_crypt, 0, sizeof(client.gs_peerchat_crypt));
                client.gs_peerchat_1 = 0;
                client.gs_peerchat_2 = 0;
            }

            sniffclient = daddr;

        } else {                            // CLIENT
            l = strchr(data, ' ');
            if(l && !memcmp(data, "CRYPT", l - data)) {   // CRYPT
                p = strrchr(data, ' ');
                if(!p) return;
                p++;

                for(l = p; *l > '\r'; l++);
                *l = 0;
                printf("- gamename: %s   ", p);

                if(get_key(p, gamekey) < 0) {
                    fprintf(stderr, "- Error: no gamekey found for this game\n");
                    return;
                }
                printf("gamekey: %s\n", gamekey);
            }
        }

        return;
    }

    if((saddr != sniffclient) && (daddr != sniffclient)) return;

    gs_peerchat(
        (sport == sniffport) ? &server : &client,
        data,
        len);

    fwrite(data, len, 1, stdout);
}



int get_key(uint8_t *gamename, uint8_t *gamekey) {
    uint8_t     buff[GSLISTSZ + 1],
                *p;

    rewind(gslist_fd);
    *gamekey = 0;

    while(fgets(buff, sizeof(buff), gslist_fd)) {
        p = strchr(buff + CNAMEOFF, ' ');
        if(p) *p = 0;
        if(!strcmp(gamename, buff + CNAMEOFF)) {
            memcpy(gamekey, buff + CKEYOFF, 6);
            gamekey[6] = 0;
            if(*gamekey > ' ') return(0);
            *gamekey = 0;
        }
    }

    return(-1);
}



FILE *gslfopen(const char *path, const char *mode) {
#ifdef WIN32
    HKEY        key;
    char        home[GSMAXPATH + 1];
    int         len;
#else
    const char  *home;
#endif
    char        retpath[GSMAXPATH + 64 + 1];

    if(!*gslist_path) {
#ifdef WIN32
        len = GSMAXPATH;
        if(!RegOpenKeyEx(HKEY_CURRENT_USER, APPDATA, 0, KEY_READ, &key)) {
            if(!RegQueryValueEx(key, "AppData", NULL, NULL, home, (void *)&len)) {
                snprintf(gslist_path, sizeof(gslist_path), "%s\\gslist\\", home);
                mkdir(gslist_path);
            }
            RegCloseKey(key);
        }
        if(!*gslist_path) strcpy(gslist_path, ".\\");
#else
        home = getenv("HOME");
        if(!home) {
            fputs("\n"
                "Error: impossible to know your $HOME or Application Data directory where\n"
                "       reading/writing the Gslist configuration files.\n"
                "       Modify the source code of the program or report the problem to me.\n"
                "\n", stderr);
            exit(1);
        }
        snprintf(gslist_path, sizeof(gslist_path), "%s/.gslist/", home);
        mkdir(gslist_path, 0755);
#endif
    }

//    snprintf(retpath, sizeof(retpath), "%s%s", gslist_path, path);
    sprintf(retpath, "%s%s", gslist_path, path);

    fprintf(stderr, "- %s\n", retpath);
    return(fopen(retpath, mode));
}



u_int resolv(char *host) {
    struct  hostent *hp;
    u_int   host_ip;

    host_ip = inet_addr(host);
    if(host_ip == INADDR_NONE) {
        hp = gethostbyname(host);
        if(!hp) {
            fprintf(stderr, "\nError: Unable to resolv hostname (%s)\n\n", host);
            exit(1);
        } else host_ip = *(u_int *)(hp->h_addr);
    }
    return(host_ip);
}


