/*
    Copyright 2007 Luigi Auriemma

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    http://www.gnu.org/licenses/gpl.txt
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <sys/stat.h>

#ifdef WIN32
    #include <direct.h>
    #define PATHSLASH   '\\'
#else
    #include <unistd.h>
    #include <dirent.h>
    #include <sys/stat.h>

    #define strnicmp    strncasecmp
    #define stristr     strcasestr
    #define PATHSLASH   '/'
#endif

typedef uint8_t     u8;
typedef uint16_t    u16;
typedef uint32_t    u32;



#define VER     "0.1"
#define BUFFSZ  8192
#define SIGN    0x47425856



int fgetz(FILE *fd, u8 *data, int len);
int frss(FILE *fd, u8 *str, int size);
int fwss(FILE *fd, u8 *str);
int frxx(FILE *fd, u32 *ret, int bits);
int fwxx(FILE *fd, u32 num, int bits);
int frff(FILE *fdi, u8 *fname, u32 off, u32 size);
int fwff(FILE *fdo, u8 *fname, u32 *size);
u8 *create_dir(u8 *name);
void std_err(void);



int main(int argc, char *argv[]) {
    struct stat xstat;
    FILE    *fd,
            *fdlist   = NULL;
    u32     data_offset,
            nameoff,
            dataoff,
            len,
            i,
            sign,
            extracted = 0;
    int     listonly  = 0,
            build     = 0;
    u8      fname[256],
            *folder   = NULL,
            *pattern  = NULL,
            *listfile = NULL;

    setbuf(stdout, NULL);

    fputs("\n"
        "PS2/VXBG files extractor/rebuilder "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", stdout);

    if(argc < 2) {
        printf("\n"
            "Usage: %s [options] <file.PS2>\n"
            "\n"
            "Options:\n"
            "-l       list files without extracting them\n"
            "-d DIR   the folder where the files will be extracted\n"
            "-p PAT   extracts only the files containing PAT, like -p .vag\n"
            "-f FILE  output file which will contain the name of the files extracted or the\n"
            "         files to place in the output PS2 file if -b is specified\n"
            "-b       build the PS2 file, -f is REQUIRED. Examples:\n"
            "           extract:   ps2ext -f list.txt -d myfolder    input.ps2\n"
            "           rebuild:   ps2ext -f list.txt -d myfolder -b output.ps2\n"
            "\n",
            argv[0]);
        exit(1);
    }

    argc--;
    for(i = 1; i < argc; i++) {
        if(((argv[i][0] != '-') && (argv[i][0] != '/')) || (strlen(argv[i]) != 2)) {
            printf("\nError: recheck your options (%s is not valid)\n", argv[i]);
            exit(1);
        }
        switch(argv[i][1]) {
            case 'l': listonly = 1;         break;
            case 'd': folder   = argv[++i]; break;
            case 'p': pattern  = argv[++i]; break;
            case 'f': listfile = argv[++i]; break;
            case 'b': build    = 1;         break;
            default: {
                printf("\nError: wrong command-line argument (%s)\n\n", argv[i]);
                exit(1);
                } break;
        }
    }

    printf("- open file %s\n", argv[argc]);
    fd = fopen(argv[argc], "rb");
    if(build) {
        if(fd) {
            fclose(fd);
            printf("\nError: the output file already exist, please rename or delete it\n");
            exit(1);
        }
        fd = fopen(argv[argc], "wb");
    }
    if(!fd) std_err();

    if(listfile) {
        printf("- open file list %s\n", listfile);
        fdlist = fopen(listfile, build ? "rb" : "wb");
        if(!fdlist) std_err();
    }
    if(folder) {
        printf("- enter in folder %s\n", folder);
        if(chdir(folder) < 0) std_err();
    }

    printf("\n"
        "  offset   size       filename\n"
        "  ----------------------------\n");

    if(build) {
        if(!fdlist) {
            printf("\nError: you must use -f for rebuilding the file\n");
            exit(1);
        }
        if(fwxx(fd, SIGN,          32) < 0) goto quit;
        if(fwxx(fd, 0,             32) < 0) goto quit;

        for(i = 0; !fgetz(fdlist, fname, sizeof(fname)); i++) {
            if(stat(fname, &xstat) < 0) std_err();
            if(fwss(fd, fname) < 0) goto quit;
            if(fwxx(fd, xstat.st_size, 32) < 0) goto quit;
        }
        extracted = i;

        data_offset = ftell(fd);
        fseek(fd, 4,            SEEK_SET);
        if(fwxx(fd, data_offset - 8,   32) < 0) goto quit;
        fseek(fd, data_offset,  SEEK_SET);

        rewind(fdlist);
        for(dataoff = data_offset; !fgetz(fdlist, fname, sizeof(fname)); dataoff += len) {
            if(fwff(fd, fname, &len) < 0) goto quit;
            printf("  %08x %10u %s\n", dataoff, len, fname);
        }

    } else {
        if(frxx(fd, &sign,         32) < 0) goto quit;
        if(sign != SIGN) printf("- sign is not %08x, the file seems NOT valid!\n", SIGN);
        if(frxx(fd, &data_offset,  32) < 0) goto quit;

        nameoff      = ftell(fd);
        data_offset += nameoff;
        for(dataoff = data_offset; nameoff < data_offset; dataoff += len) {
            if(frss(fd, fname,     sizeof(fname)) < 0) goto quit;
            if(frxx(fd, &len,      32) < 0) goto quit;
            nameoff = ftell(fd);

            if(pattern && !stristr(fname, pattern)) continue;
            printf("  %08x %10u %s\n", dataoff, len, fname);
            if(fdlist) fprintf(fdlist, "%s\n", fname);
            extracted++;
            if(listonly) continue;

            if(frff(fd, fname, dataoff, len) < 0) goto quit;
            fseek(fd, nameoff, SEEK_SET);
        }
    }

    printf("\n"
        "- %u files %s\n\n",
        extracted,
        listonly ? "listed" : "extracted");
    fclose(fd);
    if(fdlist) fclose(fdlist);
    return(0);
quit:
    printf("\nError: a read or write error occurred\n");
    fclose(fd);
    if(fdlist) fclose(fdlist);
    return(0);
}



int fgetz(FILE *fd, u8 *data, int len) {
    u8      *p;

    if(!fgets(data, len, fd)) return(-1);
    for(p = data; *p && (*p != '\n') && (*p != '\r'); p++);
    *p = 0;
    return(0);
}



int frss(FILE *fd, u8 *str, int size) {
    int     i,
            n;

    size--;
    for(i = 0; i < size; i++) {
        n = fgetc(fd);
        if(n < 0) return(-1);
        if(!n) break;
        str[i] = n;
    }
    str[i] = 0;
    return(0);
}



int fwss(FILE *fd, u8 *str) {
    int     len;

    len = strlen(str) + 1;
    if(fwrite(str, 1, len, fd) != len) return(-1);
    return(0);
}



int frxx(FILE *fd, u32 *ret, int bits) {
    u32     num;
    int     i,
            bytes;
    u8      data[4];

    bytes = bits >> 3;
    if(fread(data, 1, bytes, fd) != bytes) return(-1);

    for(num = i = 0; i < bytes; i++) {
        num |= (data[i] << (i << 3));
    }

    *ret = num;
    return(0);
}



int fwxx(FILE *fd, u32 num, int bits) {
    int     i,
            bytes;
    u8      data[4];

    bytes = bits >> 3;

    for(i = 0; i < bytes; i++) {
        data[i] = (num >> (i << 3)) & 0xff;
    }

    if(fwrite(data, 1, bytes, fd) != bytes) return(-1);
    return(0);
}



int frff(FILE *fdi, u8 *fname, u32 off, u32 size) {
    FILE    *fdo;
    int     len;
    u8      buff[BUFFSZ];

    fdo = fopen(fname, "wb");
    if(!fdo) std_err();

    if(fseek(fdi, off, SEEK_SET) < 0) return(-1);

    for(len = sizeof(buff); size; size -= len) {
        if(len > size) len = size;
        if(fread(buff, 1, len, fdi) != len) return(-1);
        if(fwrite(buff, 1, len, fdo) != len) return(-1);
    }

    fclose(fdo);
    return(0);
}



int fwff(FILE *fdo, u8 *fname, u32 *size) {
    FILE    *fdi;
    int     len;
    u8      buff[BUFFSZ];

    fdi = fopen(fname, "rb");
    if(!fdi) std_err();

    *size = 0;
    while((len = fread(buff, 1, sizeof(buff), fdi))) {
        if(fwrite(buff, 1, len, fdo) != len) return(-1);
        *size += len;
    }

    fclose(fdi);
    return(0);
}



u8 *create_dir(u8 *name) {
    u8      *p,
            *l;

    for(p = name; (*p == '\\') || (*p == '/') || (*p == '.'); p++);
    name = p;

    for(;;) {
        if((p[0] && (p[1] == ':')) || ((p[0] == '.') && (p[1] == '.'))) p[0] = '_';

        l = strchr(p, '\\');
        if(!l) {
            l = strchr(p, '/');
            if(!l) break;
        }
        *l = 0;
        p = l + 1;

#ifdef WIN32
        mkdir(name);
#else
        mkdir(name, 0755);
#endif
        *l = PATHSLASH;
    }
    return(name);
}



void std_err(void) {
    perror("\nError");
    exit(1);
}


