/*
    Copyright 2008 Luigi Auriemma

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    http://www.gnu.org/licenses/gpl-2.0.txt
*/

#define _LARGE_FILES        // if it's not supported the tool will work
#define __USE_LARGEFILE64   // without support for large files
#define __USE_FILE_OFFSET64
#define _LARGEFILE_SOURCE
#define _LARGEFILE64_SOURCE
#define _FILE_OFFSET_BITS   64

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <ctype.h>
#include <sys/stat.h>

#ifdef WIN32
    #include <direct.h>
    #define PATHSLASH   '\\'
    #define make_dir(x) mkdir(x)
#else
    #define stricmp     strcasecmp
    #define PATHSLASH   '/'
    #define make_dir(x) mkdir(x, 0755)
#endif

#if defined(_LARGE_FILES)
    #if defined(__APPLE__)
        #define fseek   fseeko
        #define ftell   ftello
    #elif defined(__FreeBSD__)
    #elif !defined(NOLFS)   // use -DNOLFS if this tool can't be compiled on your OS!
        #define off_t   off64_t
        #define fopen   fopen64
        #define fseek   fseeko64
        #define ftell   ftello64
    #endif
#endif

typedef uint8_t     u8;
typedef uint16_t    u16;
typedef uint32_t    u32;
typedef uint64_t    u64;



#define VER         "0.1"
#define BUFFSZ      0xffff
#define PRINTF64(x) (u32)(((x) >> 32) & 0xffffffff), (u32)((x) & 0xffffffff)    // I64x, llx? blah



int eipix_getfiles(FILE *fd, int tab);
int eipix_extract(FILE *fd);
void eipix_decrypt(u8 *data, int datasz, int offset);
void create_dir(u8 *name);
int check_wildcard(u8 *fname, u8 *wildcard);
int check_overwrite(u8 *fname);
u64 getxx(FILE *fd, int num_size);
void myfr(FILE *fd, void *data, unsigned size);
void std_err(void);



int     listonly    = 0,
        overwrite   = 0;
u8      *wildcard   = NULL;



int main(int argc, char *argv[]) {
    FILE    *fd;
    int     i,
            files;
    u8      *fname,
            *fdir;

    setbuf(stdout, NULL);

    fputs("\n"
        "EipiX Pyroblazer packages/files extractor "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", stdout);

    if(argc < 3) {
        printf("\n"
            "Usage: %s [options] <packages.dat> <output_folder>\n"
            "\n"
            "Options:\n"
            "-l      list files without extracting them\n"
            "-f \"W\"  extract only the files which match the wildcard W like \"ig_*\"\n"
            "-o      overwrites existent files without asking, usually some of the files of\n"
            "        the archive have the same name and folder so this option is useful\n"
            "\n", argv[0]);
        exit(1);
    }

    argc -= 2;
    for(i = 1; i < argc; i++) {
        if(((argv[i][0] != '-') && (argv[i][0] != '/')) || (strlen(argv[i]) != 2)) {
            printf("\nError: wrong argument (%s)\n", argv[i]);
            exit(1);
        }
        switch(argv[i][1]) {
            case 'l': listonly  = 1;            break;
            case 'f': wildcard  = argv[++i];    break;
            case 'o': overwrite = 1;            break;
            default: {
                printf("\nError: wrong argument (%s)\n", argv[i]);
                exit(1);
            }
        }
    }
    fname = argv[argc];
    fdir  = argv[argc + 1];

    printf("- open %s\n", fname);
    fd = fopen(fname, "rb");
    if(!fd) std_err();

    if(!listonly) {
        printf("- set output folder %s\n", fdir);
        if(chdir(fdir) < 0) std_err();
    }

    printf("- start %s:\n\n", listonly ? "listing" : "extraction");

    printf("      type offset           size       filename\n"
           "-----------------------------------------------\n");
    files = eipix_extract(fd);

    fclose(fd);
    printf("\n- %s %d files\n", listonly ? "listed" : "extracted", files);
    return(0);
}



void dumpa(FILE *fd, u8 *fname, u64 offset, u32 fsize) {
    FILE    *fdo;
    u64     oldoff;
    int     len;
    static u8   *buff = NULL;

    if(check_wildcard(fname, wildcard) < 0) return;
    if(listonly) return;

    if(!overwrite) {
        if(check_overwrite(fname) < 0) return;
    }
    // create_dir(fname); not needed for pyroblazer
    fdo = fopen(fname, "wb");
    if(!fdo) std_err();

    if(!buff) {
        buff = malloc(BUFFSZ);
        if(!buff) std_err();
    }

    oldoff = ftell(fd);
    if(fseek(fd, offset, SEEK_SET)) std_err();

    for(len = BUFFSZ; fsize; fsize -= len) {
        if(len > fsize) len = fsize;
        myfr(fd, buff, len);
        fwrite(buff, 1, len, fdo);
    }
    fclose(fdo);

    fseek(fd, oldoff, SEEK_SET);
}



int eipix_getfiles(FILE *fd, int tab) {
    u64     foffset;
    u32     ftype,
            files,
            fsize;
    int     i,
            len,
            tot     = 0;
    u8      *fname  = NULL;

    files = getxx(fd, 4);
    for(i = 0; i < files; i++) {
        len = getxx(fd, 2);
        fname = realloc(fname, len + 1);
        myfr(fd, fname, len);
        fname[len] = 0;
        getxx(fd, 1);

        if(tab < 3) {
            if(!check_wildcard(fname, wildcard)) {
                printf("%*s%s\\\n", tab * 2, "", fname);
            }
            if(!listonly) {
                make_dir(fname);
                chdir(fname);
            }
            tot += eipix_getfiles(fd, tab + 1);
            if(!listonly) {
                chdir("..");
            }
        } else {
            ftype   = getxx(fd, 4);
            foffset = getxx(fd, 8);
            fsize   = getxx(fd, 4);
            if(!check_wildcard(fname, wildcard)) {
                printf("%*s%4u %08x%08x %-10u %s\n", tab * 2, "", ftype, PRINTF64(foffset), fsize, fname);
                tot++;
            }
            if(!listonly) {
                dumpa(fd, fname, foffset, fsize);
            }
        }
    }
    if(fname) free(fname);
    return(tot);
}



int eipix_extract(FILE *fd) {
    u64     offset;
    u32     version;

    version = getxx(fd, 4);
    if(version != 1) {
        printf("- alert: this file is version %u and could be not supported by this tool\n", version);
    }
    offset  = getxx(fd, 8); // in reality it's read a 32 bit number but there is place (and logic) for a 64bit one
    if(fseek(fd, offset, SEEK_SET)) {
        offset &= 0xffffffff;
        if(fseek(fd, offset, SEEK_SET)) std_err();
    }
    return(eipix_getfiles(fd, 1));
}



void eipix_decrypt(u8 *data, int datasz, int offset) {
    int     a,
            b,
            x,
            y;
    int     i;
    u8      c;

    for(i = 0; i < datasz; i++) {
        a = offset ^ 0xd1a85b27;
        b = offset ^ 0x4f63c9e6;
        x = (-offset) & 0x1f;
        y = offset & 0x1f;
        c = 0x20 - y;
        c = (((b >> (0x20 - x)) & ~(0xff << x)) | (b << x)) ^
            ((((a >> c) & (~((c | 0xff) << y))) & 0xff) | (a << y));
        data[i] = (c ^ data[i]) - offset;
        offset++;
    }
}



void create_dir(u8 *name) {
    struct  stat    xstat;
    u8      *p,
            *l;

        // removed any security check since the files are ever officials
    if(!stat(name, &xstat)) return;

    for(p = name;; p = l + 1) {
        l = strchr(p, '\\');
        if(!l) l = strchr(p, '/');
        if(!l) break;
        *l = 0;
        make_dir(name);
        *l = PATHSLASH;
    }
}



int check_wildcard(u8 *fname, u8 *wildcard) {
    u8      *f,
            *w,
            *a;

    if(!fname || !wildcard) return(0);
    f = fname;
    w = wildcard;
    a = NULL;
    while(*f || *w) {
        if(!*w && !a) return(-1);
        if(*w == '?') {
            if(!*f) break;
            w++;
            f++;
        } else if(*w == '*') {
            w++;
            a = w;
        } else {
            if(!*f) break;
            if(tolower(*f) != tolower(*w)) {
                if(!a) return(-1);
                f++;
                w = a;
            } else {
                f++;
                w++;
            }
        }
    }
    if(*f || *w) return(-1);
    return(0);
}



int check_overwrite(u8 *fname) {
    FILE    *fd;
    u8      ans[16];

    fd = fopen(fname, "rb");
    if(!fd) return(0);
    fclose(fd);
    printf("  the file already exists, do you want to overwrite it (y/N)? ");
    fgets(ans, sizeof(ans), stdin);
    if(tolower(ans[0]) != 'y') return(-1);
    return(0);
}



u64 getxx(FILE *fd, int num_size) {
    u64     num;
    int     i;
    u8      tmp[8];

    myfr(fd, tmp, num_size);

    num = 0;
    for(i = 0; i < num_size; i++) {
        num |= (tmp[i] << (i << 3));
    }
    return(num);
}



void myfr(FILE *fd, void *data, unsigned size) {
    u64     offset;

    offset = ftell(fd);
    if(fread(data, 1, size, fd) != size) {
        printf("\nError: incomplete input file, can't read %u bytes\n", size);
        exit(1);
    }
    eipix_decrypt(data, size, offset);
}



void std_err(void) {
    perror("\nError");
    exit(1);
}


