/*

by Luigi Auriemma

*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifdef WIN32
    #include <direct.h>

    typedef unsigned char   u_char;
    typedef unsigned int   u_int;
#else
    #include <unistd.h>
    #include <sys/stat.h>

    #define stricmp strcasecmp
#endif



#define VER     "0.1.1"
#define NAMESZ  1024
#define BUFFSZ  32768
#define PACK    "SND.PACK"



void extract_file(FILE *fd, u_char *name, u_int off, u_int filelen);
void create_dir(u_char *name);
void std_err(int err);



int main(int argc, char *argv[]) {
    FILE    *fd_idx,
            *fd_dat;
    u_int  off,
            len,
            num;
    u_char  *name,
            *p;


    setbuf(stdout, NULL);

    fputs("\n"
        "Rome Total War sounds extractor "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", stdout);

    if(argc < 3) {
        printf("\n"
            "Usage: %s <file.IDX> <destination_directory>\n"
            "\n"
            " Supported only the "PACK" files:\n"
            " voice1.idx, voice2.idx, voice3.idx, music.idx and SFX.idx\n"
            "\n", argv[0]);
        exit(1);
    }

    p = strrchr(argv[1], '.');
    if(!p) std_err(2);
    p++;
    if(stricmp(p, "idx")) std_err(2);

    printf("- open index file   %s\n", argv[1]);
    fd_idx = fopen(argv[1], "rb");
    if(!fd_idx) std_err(0);

    memcpy(p, "dat", 3);
    printf("- open data file    %s\n", argv[1]);
    fd_dat = fopen(argv[1], "rb");
    if(!fd_dat) std_err(0);

    printf("- enter in folder   %s\n", argv[2]);
    if(chdir(argv[2]) < 0) std_err(0);

    name = malloc(NAMESZ);
    if(!name) std_err(0);

    if(fread(name, 8, 1, fd_idx) != 1) std_err(1);
    if(memcmp(name, PACK, 8)) std_err(4);
    if(fseek(fd_idx, 4, SEEK_CUR) < 0) std_err(1);
    if(fread((void *)&num, 4, 1, fd_idx) != 1) std_err(1);
    if(fseek(fd_idx, 8, SEEK_CUR) < 0) std_err(1);

    printf("- files to extract  %u\n", num);

    fputs("- start extraction:\n", stdout);
    while(num--) {
        if(fread((void *)&off, 4, 1, fd_idx) != 1) std_err(1);
        if(fread((void *)&len, 4, 1, fd_idx) != 1) std_err(1);
        if(fseek(fd_idx, 16, SEEK_CUR) < 0) std_err(1);
        p = name;
        do { *p = fgetc(fd_idx); } while(*p++);
        printf("  %s\n", name);
        if(fseek(fd_idx, 3, SEEK_CUR) < 0) std_err(1);

        extract_file(fd_dat, name, off, len);
    }

    fclose(fd_dat);
    fclose(fd_idx);
    fputs("\n- finished\n", stdout);
    return(0);
}



void extract_file(FILE *fd, u_char *name, u_int off, u_int filelen) {
    FILE    *fd_out;
    u_char  *buff;
    u_int  rlen;

    if(fseek(fd, off, SEEK_SET) < 0) std_err(1);

    create_dir(name);
    fd_out = fopen(name, "wb");
    if(!fd_out) std_err(3);

    buff = malloc(BUFFSZ);
    if(!buff) std_err(0);

    for(rlen = BUFFSZ; filelen; filelen -= rlen) {
        if(filelen < BUFFSZ) rlen = filelen;

        if(fread(buff, rlen, 1, fd) != 1) std_err(1);
        if(fwrite(buff, rlen, 1, fd_out) != 1) std_err(3);
    }

    fclose(fd_out);
    free(buff);
}



void create_dir(u_char *name) {
    u_char  *stri,
            *strf;

    stri = name;
    for(;;) {
        strf = strchr(stri, '/');
        if(!strf) {
            strf = strchr(stri, '\\');
            if(!strf) break;
        }
        *strf = 0;
        stri = strf + 1;
#ifdef WIN32
        mkdir(name);
        *strf = '\\';
#else
        mkdir(name, 0);
        *strf = '/';
#endif
    }
}



void std_err(int err) {
    switch(err) {
        case 0: perror("\nError"); break;
        case 1: fputs("\nError: the music file is corrupted\n", stdout); break;
        case 2: fputs("\nError: the input file has not the IDX extension\n", stdout); break;
        case 3: fputs("\nError: impossible to write the output file, probably the disk space is finished\n", stdout); break;
        case 4: fputs("\nError: this tool supports only the sound and music files ("PACK")\n", stdout); break;
    }
    exit(1);
}


