/*
    Copyright 2009 Luigi Auriemma

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    http://www.gnu.org/licenses/gpl-2.0.txt
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <ctype.h>
#include <sys/stat.h>
#include <zlib.h>

#ifdef WIN32
    #include <direct.h>
    #include <windows.h>
    #define PATHSLASH   '\\'
    #define make_dir(x) mkdir(x)
#else
    #define stricmp     strcasecmp
    #define PATHSLASH   '/'
    #define make_dir(x) mkdir(x, 0755)
    // this tools doesn't work on non-Windows systems
#endif

typedef uint8_t     u8;
typedef uint16_t    u16;
typedef uint32_t    u32;



#define VER         "0.4.1a"
#define FNAMESZ     256
#define CHUNKSZ     0x200   // default chunk size
#define ZOAGZIPSZ   (8 + 4 + 1 + 1 + 1 + 1 + 4 + 1 + 1)
#define MAXZIPLEN(n) ((n)+(((n)/1000)+1)+12)



typedef struct {
    u8      *name;
    int     offset;
    int     size;
} files_t;

typedef struct {
    u32     version;
    u32     chunk_size;
    u32     chunks;
    u32     chunks_crc;
    u32     files;
    u32     files_crc;
} zdx1_t;
typedef struct {
    u32     zsize;
    u32     start_chunk;
    u32     chunks;
    FILETIME    created;
    FILETIME    modified;
    FILETIME    accessed;
} zdx2_t;

int     listonly        = 0,
        rebuild         = 0,
        force_overwrite = 0;
u8      *filter_files   = NULL;



FILE *fd_fname(u8 *fname);
int unzip(int wbits, u8 *in, int insz, u8 *out, int outsz);
int unzoagzip(u8 *in, int insz, u8 **ret_out, int *full_outsz);
FILE *create_file(u8 *fname, int offset, int fsize);
int sdgundam_rebuild(FILE *fd, FILE *fdx, u8 *datname);
files_t *add_files(u8 *fname, int fsize, int *ret_files);
int recursive_dir(u8 *filedir);
int sdgundam_extract(FILE *fd, FILE *fdx);
int check_wildcard(u8 *fname, u8 *wildcard);
void create_dir(u8 *name);
int check_overwrite(u8 *fname);
u32 getxx(u8 *tmp, int bytes);
u32 fgetxx(FILE *fd, int bytes);
int myfr(FILE *fd, void *data, int size);
int putxx(u8 *data, u32 num, int bytes);
int fputxx(FILE *fd, u32 num, int bytes);
int myfw(FILE *fd, void *data, int size);
void myalloc(u8 **data, int wantsize, int *currsize);
void std_err(void);



int main(int argc, char *argv[]) {
    FILE    *fd         = NULL,
            *fdx        = NULL;
    int     i,
            files;
    u8      *fname,
            *fdir,
            *ext,
            *datname    = NULL;

    setbuf(stdout, NULL);

    fputs("\n"
        "SD Gundam Capsule Fighter Online ZPK/ZDX/DAT files extractor/rebuilder "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", stdout);

    if(argc < 3) {
        printf("\n"
            "Usage: %s [options] <file.ZPK/ZDX/DAT> <folder>\n"
            "\n"
            "Options\n"
            "-l     list the files in the archive without extracting them\n"
            "-f W   filter the files to extract using the W wildcard, example -f \"*.fx\"\n"
            "-b     rebuild option, instead of extracting it performs the building of the\n"
            "       zpk/zdx/dat files, examples:\n"
            "         sdgundamext -b c:\\output.zpk c:\\input_folder\n"
            "         sdgundamext -b c:\\output.dat c:\\unpacked.dat\n"
            "       note that the rebuilding is needed only in some recent versions of\n"
            "       SDGCFO, for the others it's enough to leave the uncompressed files in\n"
            "       the main folder of the game like \"9you\\SD??Online\\anrs\\10359.ani\"\n"
            "\n", argv[0]);
        exit(1);
    }

    argc -= 2;
    for(i = 1; i < argc; i++) {
        if(((argv[i][0] != '-') && (argv[i][0] != '/')) || (strlen(argv[i]) != 2)) {
            printf("\nError: wrong argument (%s)\n", argv[i]);
            exit(1);
        }
        switch(argv[i][1]) {
            case 'l': listonly      = 1;            break;
            case 'f': filter_files  = argv[++i];    break;
            case 'b': rebuild       = 1;            break;
            default: {
                printf("\nError: wrong argument (%s)\n", argv[i]);
                exit(1);
            }
        }
    }
    fname = malloc(strlen(argv[argc]) + 5);
    strcpy(fname, argv[argc]);
    fdir  = argv[argc + 1];

    ext = strrchr(fname, '.');
    if(ext) {
        ext++;
    } else {
        ext = fname + strlen(fname);
        *ext++ = '.';
        *ext   = 0;
    }

    if(rebuild) {
        if(!stricmp(ext, "dat")) {
            printf("- start building of %s\n", fname);
            if(check_overwrite(fname) < 0) exit(1);
            fd = fopen(fname, "wb");
            fdx = NULL;
            datname = fdir;
            printf("- one file only (DAT) %s\n", datname);
        } else {
            strcpy(ext, "zpk");
            printf("- start building of %s\n", fname);
            if(check_overwrite(fname) < 0) exit(1);
            fd = fopen(fname, "wb");

            strcpy(ext, "zdx");
            printf("- start building of %s\n", fname);
            if(check_overwrite(fname) < 0) exit(1);
            fdx = fopen(fname, "wb");
        }
    } else {
        fd = fd_fname(fname);
        if(!stricmp(ext, "zpk")) {
            strcpy(ext, "zdx");
            fdx = fd_fname(fname);
        } else if(!stricmp(ext, "zdx")) {
            fdx = fd;
            strcpy(ext, "zpk");
            fd = fd_fname(fname);
        }
    }

    if(!listonly && !datname) {
        printf("- set folder %s\n", fdir);
        if(chdir(fdir) < 0) std_err();
    }

    if(rebuild) {
        files = sdgundam_rebuild(fd, fdx, datname);
    } else {
        files = sdgundam_extract(fd, fdx);
    }

    if(fdx) fclose(fdx);
    fclose(fd);
    printf("\n- %d files found\n", files);
    return(0);
}



FILE *fd_fname(u8 *fname) {
    FILE    *fd;

    printf("- open %s\n", fname);
    fd = fopen(fname, "rb");
    if(!fd) std_err();
    return(fd);
}



FILE *create_file(u8 *fname, int offset, int fsize) {
    FILE    *fdo;

    if(filter_files && (check_wildcard(fname, filter_files) < 0)) return(NULL);
    printf("  %08x %-10u %s\n", offset, fsize, fname);
    if(listonly) return(NULL);
    if(check_overwrite(fname) < 0) return(NULL);
    create_dir(fname);
    fdo = fopen(fname, "wb");
    if(!fdo) std_err();
    return(fdo);
}



files_t *add_files(u8 *fname, int fsize, int *ret_files) {
    static int      filesi  = 0,
                    filesn  = 0;
    static files_t  *files  = NULL;

    if(ret_files) {
        *ret_files = filesi;
        return(files);
    }

    if(filesi >= filesn) {
        filesn += 1024;
        files = realloc(files, sizeof(files_t) * filesn);
        if(!files) std_err();
    }
    files[filesi].name   = strdup(fname);
    files[filesi].offset = 0;
    files[filesi].size   = fsize;
    filesi++;
    return(NULL);
}



int recursive_dir(u8 *filedir) {
    int     plen,
            ret     = -1;

#ifdef WIN32
    static int      winnt = -1;
    OSVERSIONINFO   osver;
    WIN32_FIND_DATA wfd;
    HANDLE  hFind;

    if(winnt < 0) {
        osver.dwOSVersionInfoSize = sizeof(OSVERSIONINFO);
        GetVersionEx(&osver);
        if(osver.dwPlatformId >= VER_PLATFORM_WIN32_NT) {
            winnt = 1;
        } else {
            winnt = 0;
        }
    }

    plen = strlen(filedir);
    strcpy(filedir + plen, "\\*.*");
    plen++;

    if(winnt) { // required to avoid problems with Vista and Windows7!
        hFind = FindFirstFileEx(filedir, FindExInfoStandard, &wfd, FindExSearchNameMatch, NULL, 0);
    } else {
        hFind = FindFirstFile(filedir, &wfd);
    }
    if(hFind == INVALID_HANDLE_VALUE) return(0);
    do {
        if(!strcmp(wfd.cFileName, ".") || !strcmp(wfd.cFileName, "..")) continue;

        strcpy(filedir + plen, wfd.cFileName);

        if(wfd.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) {
            if(recursive_dir(filedir) < 0) goto quit;
        } else {
            add_files(filedir + 2, wfd.nFileSizeLow, NULL);
        }
    } while(FindNextFile(hFind, &wfd));
    ret = 0;

quit:
    FindClose(hFind);
#else
    struct  stat    xstat;
    struct  dirent  **namelist;
    int     n,
            i;

    n = scandir(filedir, &namelist, NULL, NULL);
    if(n < 0) {
        if(stat(filedir, &xstat) < 0) {
            printf("**** %s", filedir);
            std_err();
        }
        add_files(filedir + 2, xstat.st_size, NULL);
        return(0);
    }

    plen = strlen(filedir);
    strcpy(filedir + plen, "/");
    plen++;

    for(i = 0; i < n; i++) {
        if(!strcmp(namelist[i]->d_name, ".") || !strcmp(namelist[i]->d_name, "..")) continue;
        strcpy(filedir + plen, namelist[i]->d_name);

        if(stat(filedir, &xstat) < 0) {
            printf("**** %s", filedir);
            std_err();
        }
        if(S_ISDIR(xstat.st_mode)) {
            if(recursive_dir(filedir) < 0) goto quit;
        } else {
            add_files(filedir + 2, xstat.st_size, NULL);
        }
        free(namelist[i]);
    }
    ret = 0;

quit:
    for(; i < n; i++) free(namelist[i]);
    free(namelist);
#endif
    filedir[plen - 1] = 0;
    return(ret);
}



int unzip(int wbits, u8 *in, int insz, u8 *out, int outsz) {
    static z_stream *z  = NULL;
    int     zerr;

    if(!z) {
        z = malloc(sizeof(z_stream));
        if(!z) std_err();
        z->zalloc = (alloc_func)0;
        z->zfree  = (free_func)0;
        z->opaque = (voidpf)0;
        if(inflateInit2(z, wbits)) {
            printf("\nError: zlib initialization error\n");
            exit(1);
        }
    }
    inflateReset(z);

    z->next_in   = in;
    z->avail_in  = insz;
    z->next_out  = out;
    z->avail_out = outsz;
    zerr = inflate(z, Z_FINISH);
    if(zerr != Z_STREAM_END) {
        printf("\nError: the compressed zlib/deflate input is wrong or incomplete (%d)\n", zerr);
        exit(1);
    }
    return(z->total_out);
}



int unzoagzip(u8 *in, int insz, u8 **ret_out, int *full_outsz) {
    int     fsize;
    u8      flags,
            *limit,
            *out;

    if(insz < 14) return(-1);
    limit = in + insz;
    //fsize = getxx(limit - 4, 4);  // skip the gzip stuff
    if(memcmp(in, "ZOAGZIP1", 8)) {
        fsize = insz;
        out = *ret_out;
        myalloc(&out, fsize, full_outsz);
        *ret_out = out;
        insz = limit - in;
        memcpy(out, in, fsize);
        return(fsize);
    }

    in += 8;
    fsize = getxx(in, 4);
    in += 4;
    if((in[0] != 0x1f) && (in[1] != 0x8b)) return(-1);
    in += 2;        // id1/id2
    in++;           // cm
    flags = *in++;  // flg
    in += 4;        // mtime
    in++;           // xfl
    in++;           // os
    if(flags & 4) {
        in += 2 + (in[0] | (in[1] << 8));
        if(in > limit) return(-1);
    }
    if(flags & 8)  in += strlen(in) + 1;
    if(flags & 16) in += strlen(in) + 1;
    if(flags & 2)  in += 2;
    if(in > limit) return(-1);

    out = *ret_out;
    myalloc(&out, fsize, full_outsz);
    *ret_out = out;
    insz = limit - in;
    return(unzip(-15, in, insz, out, fsize));
}



int zip(u_char *in, int size, u_char *out, int maxsz) {
    static z_stream    *z = NULL;
    int         ret;

    if(!z) {
        z = malloc(sizeof(z_stream));
        if(!z) std_err();
        z->zalloc  = Z_NULL;
        z->zfree   = Z_NULL;
        z->opaque  = Z_NULL;
        if(deflateInit2(z, Z_BEST_SPEED, Z_DEFLATED, -15, 9, Z_DEFAULT_STRATEGY)) {   // Z_BEST_COMPRESSION
            printf("\nError: zlib initialization error\n");
            exit(1);
        }
    } else {
        deflateReset(z);
    }

    z->next_in   = in;
    z->avail_in  = size;
    z->next_out  = out;
    z->avail_out = maxsz;
    deflate(z, Z_FINISH);

    ret = z->total_out;
    //deflateEnd(z);   // this is a static buffer
    return(ret);
}



void fill_it(FILE *fd, int chr, int len) {
    int     t;
    u8      buff[4096];

    memset(buff, chr, sizeof(buff));
    for(t = sizeof(buff); len; len -= t) {
        if(t > len) t = len;
        myfw(fd, buff, t);
    }
}



int sdgundam_import(FILE *fdo, u8 *fname, int chunk, int *chunks) {
    static int  buffsz  = 0,
                zbuffsz = 0;
    static u8   *buff   = NULL,
                *zbuff  = NULL;
    struct stat xstat;
    FILE    *fd;
    int     off,
            t,
            size,
            zsize,
            next_chunk,
            prev_chunk;
    u8      *p,
            *l;

    printf("- import %s\n", fname);
    fd = fopen(fname, "rb");
    if(!fd) std_err();
    fstat(fileno(fd), &xstat);
    size = xstat.st_size;
    myalloc(&buff, size, &buffsz);
    myfr(fd, buff, size);
    fclose(fd);

    zsize = MAXZIPLEN(size);
    zsize += 8; // crc + isize
    myalloc(&zbuff, zsize, &zbuffsz);
    zsize = zip(buff, size, zbuff, zsize);
    *(u32 *)(zbuff + zsize) = crc32(0, buff, size);
    zsize += 4;
    *(u32 *)(zbuff + zsize) = size;
    zsize += 4;

    if(!chunks) {   // one file mode
        myfw(fdo, "ZOAGZIP1", 8);
        fputxx(fdo, size, 4);    // size
        fputxx(fdo, 0x1f, 1);    // id1
        fputxx(fdo, 0x8b, 1);    // id2
        fputxx(fdo, 0x08, 1);    // cm
        fputxx(fdo, 0x00, 1);    // flg
        fputxx(fdo, 0x00, 4);    // mtime
        fputxx(fdo, 0x00, 1);    // xfl
        fputxx(fdo, 0x0b, 1);    // os
        myfw(fdo, zbuff, zsize);
    } else {
        off = 0;
        p = zbuff;
        l = zbuff + zsize;
        *chunks = 0;
        while(p < l) {
            next_chunk = chunk + 1;
            prev_chunk = chunk - 1;
            if(!chunk) prev_chunk = 0;

            fputxx(fdo, next_chunk, 4);
            fputxx(fdo, prev_chunk, 4);

            if(!*chunks) {
                myfw(fdo, "ZOAGZIP1", 8);
                fputxx(fdo, size, 4);    // size
                fputxx(fdo, 0x1f, 1);    // id1
                fputxx(fdo, 0x8b, 1);    // id2
                fputxx(fdo, 0x08, 1);    // cm
                fputxx(fdo, 0x00, 1);    // flg
                fputxx(fdo, 0x00, 4);    // mtime
                fputxx(fdo, 0x00, 1);    // xfl
                fputxx(fdo, 0x0b, 1);    // os
                t = ZOAGZIPSZ;
            } else {
                t = 0;
            }
            chunk++;
            (*chunks)++;

            off += t;
            t = CHUNKSZ - t;            // available space
            if((p + t) > l) t = l - p;  // required space

            myfw(fdo, p, t);
            p += t;
            off += t;
        }
        t = off % CHUNKSZ;
        if(t) fill_it(fdo, 0, CHUNKSZ - t);
    }
    return(ZOAGZIPSZ + zsize);
}



int sdgundam_rebuild(FILE *fd, FILE *fdx, u8 *datname) {
    FILETIME    quick_time;
    files_t *files  = NULL;
    zdx1_t  zdx1;
    zdx2_t  zdx2;
    int     i,
            j,
            len,
            chunk,
            tot     = 0;
    wchar_t wfname[FNAMESZ + 1];
    u8      filedir[4096];

    if(datname) {
        sdgundam_import(fd, datname, -1, NULL);
        return(1);
    }
    printf("- start recursive scanning\n");
    strcpy(filedir, ".");
    recursive_dir(filedir);
    files = add_files(NULL, 0, &tot);
    printf("- collected %d files\n", tot);

    GetSystemTimeAsFileTime(&quick_time);

    zdx1.version    = 1;
    zdx1.chunk_size = CHUNKSZ;
    zdx1.chunks     = 0;            // will be filled later
    zdx1.chunks_crc = quick_time.dwLowDateTime * 123;   // no idea of where is calculated this crc, use something rand
    zdx1.files      = tot; 
    zdx1.files_crc  = quick_time.dwHighDateTime * 456;  // no idea of where is calculated this crc, use something rand
    myfw(fdx, &zdx1, sizeof(zdx1_t));

    chunk = 0;
    for(i = 0; i < tot; i++) {
        zdx2.zsize       = sdgundam_import(fd, files[i].name, chunk, &zdx2.chunks);
        zdx2.start_chunk = chunk;
        //zdx2.chunks
        zdx2.created     = quick_time;
        zdx2.modified    = quick_time;
        zdx2.accessed    = quick_time;

        len = MultiByteToWideChar(CP_UTF8, 0, files[i].name, -1, wfname, FNAMESZ);
        for(j = 0; j < len; j++) wfname[j] ^= 0xaa;
        myfw(fdx, wfname, len * sizeof(wchar_t));
        myfw(fdx, &zdx2, sizeof(zdx2_t));

        chunk += zdx2.chunks;
    }

    fill_it(fdx, 0xff, 210000); // some useless 0xff bytes (not needed)

    zdx1.chunks     = chunk;
    fflush(fdx);
    fseek(fdx, 0, SEEK_SET);
    myfw(fdx, &zdx1, sizeof(zdx1_t));
    return(tot);
}



int sdgundam_extract(FILE *fd, FILE *fdx) {
    static int  buffsz  = 0,
                zbuffsz = 0;
    static u8   *buff   = NULL,
                *zbuff  = NULL;
    struct stat xstat;
    zdx1_t  zdx1;
    zdx2_t  zdx2;
    FILE    *fdo    = NULL;
    u32     next_chunk,
            prev_chunk,
            chunk_size  = CHUNKSZ;
    int     i,
            len,
            zlen,
            files;
    u16     c;
    wchar_t wfname[FNAMESZ + 1];
    u8      fname[FNAMESZ + 1],
            sign[8];

    myfr(fd, sign, 8);  // .dat
    fseek(fd, -8, SEEK_CUR);

    if(!fdx && !memcmp(sign, "ZOAGZIP1", 8)) {
        fstat(fileno(fd), &xstat);
        zlen = xstat.st_size;
        fdo = create_file("unpacked.dat", 0, zlen);
        if(fdo) {
            myalloc(&zbuff, zlen, &zbuffsz);
            myfr(fd, zbuff, zlen);
            len = unzoagzip(zbuff, zlen, &buff, &buffsz);
            if(len < 0) {
                printf("\nError: invalid compression\n");
                exit(1);
            }
            myfw(fdo, buff, len);
            fclose(fdo);
        }
        return(1);
    }

    myfr(fdx, &zdx1, sizeof(zdx1_t));
    if(zdx1.version != 1) {
        printf("\nAlert: the ZDX archive has an unknown version (%d)\n", zdx1.version);
        //exit(1);
    }
    chunk_size = zdx1.chunk_size;
    printf("- chunk size: 0x%08x\n", chunk_size);

    printf("\n"
        "  offset   filesize   filename\n"
        "==============================\n");

    for(files = 0; files < zdx1.files; files++) {
        for(i = 0; i < FNAMESZ; i++) {  // filename
            c = fgetxx(fdx, 2);
            //if(c == 0xffff) goto quit;  // the file is padded with various 0xff (not needed)
            wfname[i] = c ^ 0xaa;
            if(!wfname[i]) break;
        }
        if(i >= FNAMESZ) {
            printf("\nError: the input filename is longer than how much supported\n");
            exit(1);
        }
        WideCharToMultiByte(CP_UTF8, 0, wfname, -1, fname, FNAMESZ, NULL, NULL);
        myfr(fdx, &zdx2, sizeof(zdx2_t));

        fdo = create_file(fname, zdx2.start_chunk * (4 + 4 + chunk_size), zdx2.zsize);
        if(fdo) {
            myalloc(&zbuff, zdx2.zsize + chunk_size, &zbuffsz); // "+ chunk_size" avoids the boring padding

            zlen = 0;
            next_chunk = zdx2.start_chunk;
            for(i = 0; i < zdx2.chunks; i++) {
                if(fseek(fd, next_chunk * (4 + 4 + chunk_size), SEEK_SET)) std_err();
                next_chunk = fgetxx(fd, 4); 
                prev_chunk = fgetxx(fd, 4);
                if(zlen > zdx2.zsize) { // remember the previous "+ chunk_size": if((zlen + chunk_size) > (zdx2.zsize + chunk_size)) {
                    printf("\nError: the data to read (%d) is bigger than how much expected (%d)\n", zlen + chunk_size, zdx2.zsize);
                    exit(1);
                }
                myfr(fd, zbuff + zlen, chunk_size);
                zlen += chunk_size;
            }
            zlen = zdx2.zsize;  // not needed
            len = unzoagzip(zbuff, zlen, &buff, &buffsz);
            if(len < 0) {
                printf("\nError: invalid compression\n");
                exit(1);
            }
            myfw(fdo, buff, len);
            fclose(fdo);
        }
    }
    return(files);
}



int check_wildcard(u8 *fname, u8 *wildcard) {
    u8      *f,
            *w,
            *a;

    if(!wildcard) return(0);
    f = fname;
    w = wildcard;
    a = NULL;
    while(*f || *w) {
        if(!*w && !a) return(-1);
        if(*w == '?') {
            if(!*f) break;
            w++;
            f++;
        } else if(*w == '*') {
            w++;
            a = w;
        } else {
            if(!*f) break;
            if(tolower(*f) != tolower(*w)) {
                if(!a) return(-1);
                f++;
                w = a;
            } else {
                f++;
                w++;
            }
        }
    }
    if(*f || *w) return(-1);
    return(0);
}



void create_dir(u8 *name) {
    struct  stat    xstat;
    u8      *p,
            *l;

        // removed any security check since the files are ever officials
    if(!stat(name, &xstat)) return;

    for(p = name;; p = l + 1) {
        l = strchr(p, '\\');
        if(!l) l = strchr(p, '/');
        if(!l) break;
        *l = 0;
        make_dir(name);
        *l = PATHSLASH;
    }
}



int check_overwrite(u8 *fname) {
    FILE    *fd;
    u8      ans[16];

    if(force_overwrite) return(0);
    if(!fname) return(0);
    fd = fopen(fname, "rb");
    if(!fd) return(0);
    fclose(fd);
    printf("- the file \"%s\" already exists\n  do you want to overwrite it (y/N/all)? ", fname);
    fgets(ans, sizeof(ans), stdin);
    if(tolower(ans[0]) == 'y') return(0);
    if(tolower(ans[0]) == 'a') {
        force_overwrite = 1;
        return(0);
    }
    return(-1);
}



u32 getxx(u8 *tmp, int bytes) {
    u32     num;
    int     i;

    num = 0;
    for(i = 0; i < bytes; i++) {
        num |= (tmp[i] << (i << 3));
    }
    return(num);
}



u32 fgetxx(FILE *fd, int bytes) {
    u8      tmp[bytes];

    myfr(fd, tmp, bytes);
    return(getxx(tmp, bytes));
}



int myfr(FILE *fd, void *data, int size) {
    int     len;

    len = fread(data, 1, size, fd);
    if(len != size) {
        printf("\nError: incomplete input file, can't read %u bytes\n", size - len);
        exit(1);
    }
    return(len);
}



int putxx(u8 *data, u32 num, int bytes) {
    int     i;

    for(i = 0; i < bytes; i++) {
        data[i] = num >> (i << 3);
    }
    return(bytes);
}



int fputxx(FILE *fd, u32 num, int bytes) {
    u8      tmp[bytes];

    putxx(tmp, num, bytes);
    myfw(fd, tmp, bytes);
    return(bytes);
}



int myfw(FILE *fd, void *data, int size) {
    int     len;

    len = fwrite(data, 1, size, fd);
    if(len != size) {
        printf("\nError: impossible to write %u bytes\n", size - len);
        exit(1);
    }
    return(len);
}



void myalloc(u8 **data, int wantsize, int *currsize) {
    if(!wantsize) return;
    if(wantsize <= *currsize) {
        if(*currsize > 0) return;
    }
    *data = realloc(*data, wantsize);
    if(!*data) std_err();
    *currsize = wantsize;
}



void std_err(void) {
    perror("\nError");
    exit(1);
}


