/*
    Copyright 2009 Luigi Auriemma

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    http://www.gnu.org/licenses/gpl-2.0.txt
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <ctype.h>
#include <sys/stat.h>
#include <zlib.h>
#include "blast.h"

#ifdef WIN32
    #include <direct.h>
    #define PATHSLASH   '\\'
    #define make_dir(x) mkdir(x)
#else
    #define stricmp     strcasecmp
    #define PATHSLASH   '/'
    #define make_dir(x) mkdir(x, 0755)
#endif

typedef uint8_t     u8;
typedef uint16_t    u16;
typedef uint32_t    u32;



#define VER         "0.1"



int ssa_extract(FILE *fd);
void dumpa(u8 *fname, u8 *data, int size, u32 offset);
void ssa_xor(u8 *xor_key, u8 *data, int len);
void create_dir(u8 *name);
int check_overwrite(u8 *fname);
int unzip(u8 *in, int insz, u8 *out, int outsz);
void myalloc(u8 **data, unsigned wantsize, unsigned *currsize);
void myfr(FILE *fd, void *data, unsigned size);
void std_err(void);



int main(int argc, char *argv[]) {
    FILE    *fd;
    int     files;
    u8      *fname,
            *fdir;

    setbuf(stdout, NULL);

    fputs("\n"
        "Stainless Steel Studios SSA files extractor "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", stdout);

    if(argc < 3) {
        printf("\n"
            "Usage: %s <file.SSA> <output_folder>\n"
            "\n", argv[0]);
        exit(1);
    }

    fname = argv[1];
    fdir  = argv[2];

    printf("- open %s\n", fname);
    fd = fopen(fname, "rb");
    if(!fd) std_err();

    printf("- set output folder %s\n", fdir);
    if(chdir(fdir) < 0) std_err();

    files = ssa_extract(fd);

    fclose(fd);
    printf("\n- %d files found\n", files);
    return(0);
}



// arghhhhhh!!! blast() is horrible to use!!!
int     blastardo_data_insz     = 0,
        blastardo_data_outsz    = 0;
u8      *blastardo_data_in      = NULL,
        *blastardo_data_out     = NULL;
unsigned blastardo_inf(void *how, u8 **buf) {
    *buf = blastardo_data_in;       // all the buffer in one time
    return(blastardo_data_insz);
}
int blastardo_outf(void *how, u8 *buf, unsigned len) {
    blastardo_data_out   = buf;
    blastardo_data_outsz = len;
    return(0);
}



int ssa_extract(FILE *fd) {
typedef struct {
    u8      sign[4];
    u32     version;
    u32     boh;        // not verified
    u32     size;
} headinfo_t;
typedef struct {
    u32     offset;
    u32     end_offset;
    u32     size;
} fileinfo_t;
typedef struct {
    u8      type[4];    // this is read also separately
    u32     fsize;
    u32     zero;       // ignored by the game
} zinfo_t;

    static const u8 *ssa_xor_keys[32] = {
                "\x34\x29\x12\x54\x39\x22\x69\x38\x22\x45\x99\x12\x83\x53\x92\xFA\x43\x48\x72\x69\xAC\x3B\x55\xDD\x28\x79\xF5\x3A\x65\x93\x27\x84", // Rise and Fall demo
                "\x64\x34\x68\x82\x55\x87\x49\x32\xd3\x02\x01\xb2\x12\x73\x13\xff\x59\x21\x39\x57\x54\x30\xef\xa3\x56\x23\x99\x48\x98\xb1\xa9\x48", // Rise and Fall full game
                NULL };
    static int  insz    = 0,
                outsz   = 0;
    static u8   *in     = NULL,
                *out    = NULL;

    headinfo_t  headinfo;
    fileinfo_t  *fileinfo;
    zinfo_t   zinfo;
    int         n,
                files;
    u8          type[4],
                *xor_key = NULL,
                *header = NULL,
                *fname,
                *ph,
                *phl;

    myfr(fd, &headinfo, sizeof(headinfo_t));
    if(memcmp(headinfo.sign, "rass", 4)) {
        printf("\nError: wrong signature (%4.4s)\n", headinfo.sign);
        exit(1);
    }
    if(!headinfo.version || (headinfo.version > 3)) {
        printf("\nError: this version (%d) has not been tested yet, contact me\n", headinfo.version);
        exit(1);
    }
    if(headinfo.boh) {
        printf("\nError: the headinfo.boh parameter (%08x) is not supported yet, contact me\n", headinfo.boh);
        exit(1);
    }

    if((headinfo.version >= 2) && (headinfo.size > 0x100)) {
        xor_key = (u8 *)ssa_xor_keys[headinfo.version - 2];   // seems that version 3 is used only in retail, needs to recheck in future when there will be other SSA files

        myfr(fd, &n, 4);
        ssa_xor(xor_key, (u8 *)&n, 4);

        myalloc(&in, n, &insz);
        myfr(fd, in, n);
        ssa_xor(xor_key + 4, (u8 *)in, 32 - 4);

        myalloc(&header, headinfo.size, NULL);
        unzip(in, n, header, headinfo.size);
    } else {
        myalloc(&header, headinfo.size, NULL);
        myfr(fd, header, headinfo.size);
    }

    printf("\n"
        "  offset   filesize   filename\n"
        "==============================\n");

    ph  = header;
    phl = header + headinfo.size;
    for(files = 0; ph < phl; files++) {
        n = *(u32 *)ph;                 ph += 4;
        fname = ph;                     ph += n;
        fileinfo = (fileinfo_t *)ph;    ph += sizeof(fileinfo_t);
        if(headinfo.version >= 2) {
            xor_key = ph;               ph += 32;
        }
        if(fseek(fd, fileinfo->offset, SEEK_SET)) std_err();

        // practically PK01 and ZL01 are just like some types of files but they are "handled" internally

        if(fileinfo->size >= 4) {   // verify the first 4 bytes of the file
            myfr(fd, &type, 4);
            if(headinfo.version >= 2) ssa_xor(xor_key, type, 4);
            fseek(fd, -4, SEEK_CUR);
        }
        if((fileinfo->size >= sizeof(zinfo_t)) && (!memcmp(type, "PK01", 4) || !memcmp(type, "ZL01", 4))) {
            myfr(fd, &zinfo, sizeof(zinfo_t));
            if(headinfo.version >= 2) ssa_xor(xor_key, (u8 *)&zinfo, sizeof(zinfo_t));

            fileinfo->size -= sizeof(zinfo_t);
            myalloc(&in, fileinfo->size, &insz);
            myfr(fd, in, fileinfo->size);
            if(headinfo.version >= 2) ssa_xor(xor_key + sizeof(zinfo_t), (u8 *)in, 32 - sizeof(zinfo_t));

            myalloc(&out, zinfo.fsize, &outsz);
            if(!memcmp(zinfo.type, "PK01", 4)) {        // pkware explode
                blastardo_data_in   = in;
                blastardo_data_insz = fileinfo->size;
                if(blast(blastardo_inf, NULL, blastardo_outf, NULL, out, zinfo.fsize)) {
                    printf("\nError: pk01 blast() decompression error\n");
                    exit(1);
                }
                if(zinfo.fsize != blastardo_data_outsz) {
                    printf("\nError: pk01 blast() error, output size differs\n");
                    exit(1);
                }

            } else if(!memcmp(zinfo.type, "ZL01", 4)) { // zlib deflate
                unzip(in, fileinfo->size, out, zinfo.fsize);

            } else {
                printf("\nError: unknown type of compression (%4.4s) at offset %08x\n", zinfo.type, fileinfo->offset);
                exit(1);
            }
        } else {    // no compression
            zinfo.fsize = fileinfo->size;
            myalloc(&out, zinfo.fsize, &outsz);
            myfr(fd, out, zinfo.fsize);
            if(headinfo.version >= 2) ssa_xor(xor_key, (u8 *)out, (zinfo.fsize < 32) ? zinfo.fsize : 32); // simply 32 was good too
        }

        dumpa(fname, out, zinfo.fsize, fileinfo->offset);
    }

    if(header) free(header);
    if(in) free(in);
    if(out) free(out);
    return(files);
}



void dumpa(u8 *fname, u8 *data, int size, u32 offset) {
    FILE    *fdo;

    printf("  %08x %-10u %s\n", offset, size, fname);
    if(check_overwrite(fname) < 0) return;
    create_dir(fname);

    fdo = fopen(fname, "wb");
    if(!fdo) std_err();
    if(fwrite(data, 1, size, fdo) != size) {
        printf("\nError: impossible to write the file on the disk, check your space\n");
        exit(1);
    }
    fclose(fdo);
}



void ssa_xor(u8 *xor_key, u8 *data, int len) {
    int     i;

    for(i = 0; i < len; i++) {
        data[i] ^= xor_key[i];
    }
}



void create_dir(u8 *name) {
    struct  stat    xstat;
    u8      *p,
            *l;

        // removed any security check since the files are ever officials
    if(!stat(name, &xstat)) return;

    for(p = name;; p = l + 1) {
        l = strchr(p, '\\');
        if(!l) l = strchr(p, '/');
        if(!l) break;
        *l = 0;
        make_dir(name);
        *l = PATHSLASH;
    }
}



int check_overwrite(u8 *fname) {
    FILE    *fd;
    u8      ans[16];

    fd = fopen(fname, "rb");
    if(!fd) return(0);
    fclose(fd);
    printf("  the file already exists, do you want to overwrite it (y/N)? ");
    fgets(ans, sizeof(ans), stdin);
    if(tolower(ans[0]) != 'y') return(-1);
    return(0);
}



int unzip(u8 *in, int insz, u8 *out, int outsz) {
    uLongf      outlen;

    outlen = outsz;
    if(uncompress(out, &outlen, in, insz) != Z_OK) {
        printf("\nError: zlib uncompress error\n");
        exit(1);
    }
    return(outlen);
}



void myalloc(u8 **data, unsigned wantsize, unsigned *currsize) {
    if(currsize) {
        if(wantsize <= *currsize) return;
    }
    *data = realloc(*data, wantsize);
    if(!*data) std_err();
    if(currsize) *currsize = wantsize;
}



void myfr(FILE *fd, void *data, unsigned size) {
    int     len;

    len = fread(data, 1, size, fd);
    if(len != size) {
        printf("\nError: incomplete input file, can't read %u bytes (%u remaining)\n", size, size - len);
        exit(1);
    }
}



void std_err(void) {
    perror("\nError");
    exit(1);
}


