/*
    Copyright 2008 Luigi Auriemma

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    http://www.gnu.org/licenses/gpl.txt
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <ctype.h>
#include <sys/stat.h>

#ifdef WIN32
    #include <direct.h>
    #define PATHSLASH   '\\'
    #define make_dir(x) mkdir(x)
#else
    #define stricmp     strcasecmp
    #define PATHSLASH   '/'
    #define make_dir(x) mkdir(x, 0755)
#endif

typedef uint8_t     u8;
typedef uint16_t    u16;
typedef uint32_t    u32;



#define VER         "0.1"
#define BUFFSZ      8192
#define NAMESZ      2048



#pragma pack(4)
typedef struct {
    u32     total_size;
    u32     files;
    u32     assets;
    u32     dunno1;
    u32     dunno2;
    u32     dunno3;
} ssnam67_head_t;
typedef struct {
    u32     size;
    u32     offset;
    u32     name_offset;
    u32     asset;
} ssnam67_file_t;
#pragma pack()



int check_wildcard(u8 *fname, u8 *wildcard);
void ssnam67_extract(FILE *fd, u8 *index, ssnam67_file_t *ssnam67_file);
void check_overwrite(u8 *fname);
void ssnam67_dumpa(FILE *fd, u8 *fname, int offset, int size);
void myfr(FILE *fd, void *data, unsigned size);
void std_err(void);



int     ssnam67_listonly    = 0;
u8      *ssnam67_wildcard   = NULL;



int main(int argc, char *argv[]) {
    ssnam67_head_t  ssnam67_head;
    ssnam67_file_t  *ssnam67_file;
    FILE    *fdi,
            **fd    = NULL;
    int     i,
            files;
    u8      *index,
            *fname,
            *fdir   = NULL,
            *fnamex;

    setbuf(stdout, NULL);

    fputs("\n"
        "ShellShock Nam67 files extractor "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", stdout);

    if(argc < 2) {
        printf("\n"
            "Usage: %s [options] <ASSETS.DAT>\n"
            "\n"
            "Options:\n"
            "-l      list files without extracting them\n"
            "-d DIR  specify the output directory (default is the current one)\n"
            "-f \"W\"  extract only the files which match the wildcard W like \"*music*\"\n"
            "\n", argv[0]);
        exit(1);
    }

    argc--;
    for(i = 1; i < argc; i++) {
        if(((argv[i][0] != '-') && (argv[i][0] != '/')) || (strlen(argv[i]) != 2)) {
            printf("\nError: wrong argument (%s)\n", argv[i]);
            exit(1);
        }
        switch(argv[i][1]) {
            case 'l': ssnam67_listonly  = 1;            break;
            case 'd': fdir              = argv[++i];    break;
            case 'f': ssnam67_wildcard  = argv[++i];    break;
            default: {
                printf("\nError: wrong argument (%s)\n", argv[i]);
                exit(1);
            }
        }
    }
    fname = malloc(strlen(argv[argc]) + 10);
    if(!fname) std_err();
    strcpy(fname, argv[argc]);

    fnamex = strrchr(fname, '\\');
    if(!fnamex) fnamex = strrchr(fname, '/');
    if(fnamex) {
        fnamex++;
    } else {
        fnamex = fname;
    }
    if(stricmp(fnamex, "ASSETS.DAT")) {
        printf("\n"
            "Error: you must specify only the ASSETS.DAT file since it contains all the\n"
            "       info for extracting the data from the other files\n");
        exit(1);
    }

    printf("- open %s\n", fname);
    fdi = fopen(fname, "rb");
    if(!fdi) std_err();

    myfr(fdi, &ssnam67_head, sizeof(ssnam67_head_t));

    index = malloc(ssnam67_head.total_size);
    if(!index) std_err();
    myfr(fdi, index, ssnam67_head.total_size);
    fclose(fdi);

    fd = calloc(sizeof(FILE *), ssnam67_head.assets);
    if(!fd) std_err();
    if(!ssnam67_listonly) {
        for(i = 1; i < ssnam67_head.assets; i++) {
            sprintf(fnamex, "ASSETS%02d.DAT", i);
            printf("- open %s\n", fname);
            fd[i] = fopen(fname, "rb");
            if(!fd[i]) std_err();
        }
        if(fdir) {
            printf("- set output folder %s\n", fdir);
            if(chdir(fdir) < 0) std_err();
        }
    }

    printf("\n"
        "  fd filesize   filename\n"
        "------------------------\n");
    for(ssnam67_file = (ssnam67_file_t *)index, files = 0; files < ssnam67_head.files; files++, ssnam67_file++) {
        ssnam67_extract(fd[ssnam67_file->asset], index, ssnam67_file);
    }

    if(!ssnam67_listonly) {
        for(i = 1; i < ssnam67_head.assets; i++) {
            fclose(fd[i]);
        }
    }
    free(fd);
    printf("\n- %u files in the archive\n", files);
    return(0);
}



void create_dir(u8 *name) {
    struct  stat    xstat;
    u8      *p,
            *l;

        // removed any security check since the files are ever officials
    for(p = name; *p; p++) {
        if((*p == '\\') || (*p == '/')) *p = PATHSLASH;
    }
    if(!stat(name, &xstat)) return;

    for(p = name;; p = l + 1) {
        l = strchr(p, PATHSLASH);
        if(!l) break;
        *l = 0;
        make_dir(name);
        *l = PATHSLASH;
    }
}



int check_wildcard(u8 *fname, u8 *wildcard) {
    u8      *f,
            *w,
            *a;

    f = fname;
    w = wildcard;
    a = NULL;
    while(*f || *w) {
        if(!*w && !a) return(-1);
        if(*w == '?') {
            if(!*f) break;
            w++;
            f++;
        } else if(*w == '*') {
            w++;
            a = w;
        } else {
            if(!*f) break;
            if(tolower(*f) != tolower(*w)) {
                if(!a) return(-1);
                f++;
                w = a;
            } else {
                f++;
                w++;
            }
        }
    }
    if(*f || *w) return(-1);
    return(0);
}



void ssnam67_extract(FILE *fd, u8 *index, ssnam67_file_t *ssnam67_file) {
    u8      *name;

    if(!ssnam67_file->offset && !ssnam67_file->size) return;    // skip the "break" files

    name = index + ssnam67_file->name_offset;
    if(ssnam67_wildcard && (check_wildcard(name, ssnam67_wildcard) < 0)) return;

    printf("  %02d %-10u %s\n",
        ssnam67_file->asset, ssnam67_file->size, name);
    if(ssnam67_listonly) return;

    ssnam67_dumpa(fd, name, ssnam67_file->offset, ssnam67_file->size);
}



void check_overwrite(u8 *fname) {
    FILE    *fd;
    u8      ans[16];

    fd = fopen(fname, "rb");
    if(!fd) return;
    fclose(fd);
    printf("  the file already exists, do you want to overwrite it (y/N)? ");
    fgets(ans, sizeof(ans), stdin);
    if(tolower(ans[0]) != 'y') exit(1);
}



void ssnam67_dumpa(FILE *fd, u8 *fname, int offset, int size) {
    FILE    *fdo;
    int     len;
    u8      buff[BUFFSZ];

    if(fseek(fd, offset, SEEK_SET)) std_err();

    check_overwrite(fname);
    create_dir(fname);
    fdo = fopen(fname, "wb");
    if(!fdo) std_err();
    for(len = BUFFSZ; size; size -= len) {
        if(size < len) len = size;
        myfr(fd, buff, len);
        fwrite(buff, 1, len, fdo);
    }
    fclose(fdo);
}



void myfr(FILE *fd, void *data, unsigned size) {
    if(fread(data, 1, size, fd) != size) {
        printf("\nError: incomplete input file, can't read %u bytes\n", size);
        exit(1);
    }
}



void std_err(void) {
    perror("\nError");
    exit(1);
}


