/*
    Copyright 2005,2006 Luigi Auriemma

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    http://www.gnu.org/licenses/gpl.txt
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifdef WIN32
    #include <winsock.h>
    #include <malloc.h>
    #include "winerr.h"

    #define close   closesocket
    #define sleep   Sleep
#else
    #include <unistd.h>
    #include <sys/socket.h>
    #include <sys/types.h>
    #include <arpa/inet.h>
    #include <netinet/in.h>
    #include <netdb.h>
#endif



#define VER         "0.1a"
#define MS          "hlmaster1.hlauth.net"
#define MSPORT      27010
#define BUFFSZ      65536   // max UDP packets from Steam are about 1400 bytes
                            // I use this size to avoid a realloc() abuse
#define TIMEOUT     3
#define STEAMPCK    2000    // its max 1400 but is better to exagerate
#define FORMAT      "%16s : %hu\n"
#define NORMFORMAT  "%s:%hu\n"



int timeout(int sock);
uint32_t resolv(char *host);
void std_err(void);



int main(int argc, char *argv[]) {
    struct  sockaddr_in peer;
    uint32_t    len,
                servers,
                ip;
    int         sd,
                slen,
                tmp,
                i,
                dynsz,
                execlen          = 0,
                pcklimit         = 0;
    uint16_t    msport           = MSPORT,
                port;
    uint8_t     buffsend[STEAMPCK],
                *buff            = NULL,
                *ms              = MS,
                *tmpexec         = NULL,
                *execstring      = NULL,
                *execstring_ip   = NULL,
                *execstring_port = NULL,
                *execptr         = NULL,
                *filter          = NULL,
                *format          = FORMAT,
                *p;

#ifdef WIN32
    WSADATA wsadata;
    WSAStartup(MAKEWORD(1,0), &wsadata);
#endif

    setbuf(stdout, NULL);
    setbuf(stderr, NULL);

    fputs("\n" \
        "Steamlist "VER"\n" \
        "by Luigi Auriemma\n" \
        "e-mail: aluigi@autistici.org\n" \
        "web:    aluigi.org\n" \
        "\n", stdout);

    if(argc < 2) {
        printf("\n"
            "Usage: %s [options] <filter>\n"
            "\n"
            "Options:\n"
            "-r \"prog...\"  lets you to execute a specific program for each IP found (as the\n"
            "              ping command for example).\n"
            "              There are 2 available parameters and are #IP and #PORT that will\n"
            "              be substituited with the IP and port of the current online game\n"
            "              server found in the list. Example: -r \"echo #IP and #PORT yeah\"\n"
            "-l PCKS       limit the amount of data to receive to max PCKS packets\n"
            "-c            shows IP and port in the normal IP:PORT format\n"
            "-x S[:P]      lets you to specify a different master server (S) and port (P,\n"
            "              optional) instead of "MS":%d\n"
            "\n"
            " Note: leave filter empty (\"\") if you don't need it, while an example of valid\n"
            "       filter is \\gamedir\\cstrike or \\gamedir\\valve\n"
            "\n", argv[0], msport);
        exit(1);
    }

    argc--;
    for(i = 1; i < argc; i++) {
        switch(argv[i][1]) {
            case 'r': {
                i++;
                execstring = argv[i];
                execlen = strlen(execstring) + 23;

                tmpexec = malloc(execlen);
                if(!tmpexec) std_err();

                execstring_ip = strstr(execstring, "#IP");
                execstring_port = strstr(execstring, "#PORT");
                if(execstring_ip) *execstring_ip = 0;
                if(execstring_port) *execstring_port = 0;

                execlen = strlen(execstring);
                memcpy(tmpexec, execstring, execlen);
                } break;
            case 'l': {
                i++;
                pcklimit = atoi(argv[i]);
                } break;
            case 'c': {
                format = NORMFORMAT;
                } break;
            case 'x': {
                i++;
                if(!argv[i]) {
                    fputs("\n"
                        "Error: you have forgotten some parameters\n"
                        "\n", stderr);
                    exit(1);
                }
                ms = strchr(argv[i], ':');
                if(ms) {
                    msport = atoi(ms + 1);
                    *ms = 0;
                }
                ms = argv[i];
                } break;
            default: {
                fprintf(stderr, "\n"
                    "Error: wrong argument (%s)\n"
                    "\n", argv[i]);
                exit(1);
            } break;
        }
    }

    filter = argv[argc];

    if(strlen(filter) > (BUFFSZ - 25)) {
        fputs("\nError: your filter is too long\n\n", stderr);
        exit(1);
    }

    printf("- resolv     %s\n", ms);
    peer.sin_addr.s_addr = resolv(ms);
    peer.sin_port        = htons(msport);
    peer.sin_family      = AF_INET;

    buff = malloc(BUFFSZ);
    if(!buff) std_err();
    dynsz = BUFFSZ;

    printf(
        "  Server:    %s:%hu\n",
        inet_ntoa(peer.sin_addr), msport);

    sd = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
    if(sd < 0) std_err();

    fputs("- receive:   ", stdout);

    buffsend[0] = '1';
    buffsend[1] = 0xff;

    len = ip = port = i = 0;
    do {
        slen = 2 + sprintf(
            buffsend + 2,
            "%s:%d",
            inet_ntoa(*(struct in_addr *)&ip), port) + 1;
        slen += sprintf(
            buffsend + slen,
            "%s",
            filter) + 1;

        sendto(sd, buffsend, slen, 0, (struct sockaddr *)&peer, sizeof(peer));

        if(timeout(sd) < 0) {   // if one timeout, we resend
            sendto(sd, buffsend, slen, 0, (struct sockaddr *)&peer, sizeof(peer));
            if(timeout(sd) < 0) break;
        }

        if((dynsz - len) < STEAMPCK) {
            dynsz += BUFFSZ;
            buff = realloc(buff, dynsz);
            if(!buff) std_err();
        }

        tmp = recvfrom(sd, buff + len, dynsz - len, 0, NULL, NULL);
        if(tmp < 0) std_err();
        len += tmp;
        fputc('.', stdout);

            /* I need to use a fixed buffer for obvious reasons (-r option) */
        ip   = *(uint32_t *)(buff + len - 6);
        port = ntohs(*(uint16_t *)(buff + len - 2));
    } while(ip && (++i != pcklimit));

    printf("\n"
        "- %u bytes received\n"
        "\n"
        "              IP : PORT\n"
        "------------------------\n", len);

    for(servers = 0, p = buff; len >= 6; len -= 6, p += 6) {
        ip   = *(uint32_t *)p;
        if(ip == 0xffffffff) continue;
        if(!ip) break;
        port = ntohs(*(uint16_t *)(p + 4));

        printf(format, inet_ntoa(*(struct in_addr *)&ip), port);

        if(execstring) {
            execptr = tmpexec + execlen;
            if(execstring_ip && !execstring_port) {
                execptr += sprintf(execptr, "%s", inet_ntoa(*(struct in_addr *)&ip));
                strcpy(execptr, execstring_ip + 3);

            } else if(execstring_port && !execstring_ip) {
                execptr += sprintf(execptr, "%d", port);
                strcpy(execptr, execstring_port + 5);

            } else if(execstring_ip < execstring_port) {
                execptr += sprintf(execptr, "%s", inet_ntoa(*(struct in_addr *)&ip));
                execptr += sprintf(execptr, "%s", execstring_ip + 3);
                execptr += sprintf(execptr, "%d", port);
                strcpy(execptr, execstring_port + 5);

            } else if(execstring_port < execstring_ip) {
                execptr += sprintf(execptr, "%d", port);
                execptr += sprintf(execptr, "%s", execstring_port + 5);
                execptr += sprintf(execptr, "%s", inet_ntoa(*(struct in_addr *)&ip));
                strcpy(execptr, execstring_ip + 3);
            }

            printf("   Execute program: \"%s\"\n", tmpexec);
            system(tmpexec);
        }

        servers++;
    }

    close(sd);

    printf("\n\nOnline there are %u servers\n\n", servers);

    free(buff);
    return(0);
}



int timeout(int sock) {
    struct  timeval tout;
    fd_set  fd_read;
    int     err;

    tout.tv_sec = TIMEOUT;
    tout.tv_usec = 0;
    FD_ZERO(&fd_read);
    FD_SET(sock, &fd_read);
    err = select(sock + 1, &fd_read, NULL, NULL, &tout);
    if(err < 0) std_err();
    if(!err) return(-1);
    return(0);
}



uint32_t resolv(char *host) {
    struct  hostent *hp;
    uint32_t  host_ip;

    host_ip = inet_addr(host);
    if(host_ip == INADDR_NONE) {
        hp = gethostbyname(host);
        if(!hp) {
            fprintf(stderr, "\nError: Unable to resolv hostname (%s)\n", host);
            exit(1);
        } else host_ip = *(uint32_t *)(hp->h_addr);
    }
    return(host_ip);
}



#ifndef WIN32
    void std_err(void) {
        perror("\nError");
        exit(1);
    }
#endif


