/*
    Copyright 2005,2006,2007,2008 Luigi Auriemma

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    http://www.gnu.org/licenses/gpl.txt
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>

#ifdef WIN32
    #include <winsock.h>
    #include "winerr.h"

    #define close   closesocket
    #define sleep   Sleep
#else
    #include <unistd.h>
    #include <sys/socket.h>
    #include <sys/types.h>
    #include <arpa/inet.h>
    #include <netinet/in.h>
    #include <netdb.h>

    #define stristr strcasestr
#endif

typedef uint8_t     u8;
typedef uint16_t    u16;
typedef uint32_t    u32;



#define VER     "0.1.1"
#define TLHOST  "weblist.teamspeak.org"
#define TLPORT  45648



void exec_arg(char *data);
void exec_run(u32 ip, u16 port);
void send_tl(int sock, u8 query, u8 countryt, u16 country, u8 type, u8 pass, u8 usert, u16 user, u8 namet, char *name);
int recv_tl(int sock, u8 *data);
char *get_country(u16 *country_num, char *country);
u32 resolv(char *host);
void std_err(void);



#pragma pack(2)
struct tspeak_weblist {
    u16     info;
    u16     users;
    u16     maxusers;
    u16     country;
    u32     ip;
    u16     port;
} *tl;
#pragma pack()



int     execlen          = 0;
u8      *tmpexec         = NULL,
        *execstring      = NULL,
        *execstring_ip   = NULL,
        *execstring_port = NULL,
        *execptr         = NULL;



int main(int argc, char *argv[]) {
    struct  sockaddr_in peer;
    int     sd,
            i,
            show_type   = 0,
            servers;
    u16     tlport      = TLPORT,
            country     = 0,
            user        = 0;
    u8      buff[256],
            countryt    = 0,
            type        = 0,
            pass        = 0,
            usert       = 0,
            namet       = 0,
            *name       = "",
            *tlhost     = TLHOST;

#ifdef WIN32
    WSADATA wsadata;
    WSAStartup(MAKEWORD(1,0), &wsadata);
#endif

    setbuf(stdout, NULL);

    fputs("\n"
        "Teamspeaklist "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", stderr);

    if(argc < 2) {
        fprintf(stderr,
            "\n"
            "Usage: %s [options] <filter>\n"
            "\n"
            "Filter:\n"
            "-a            Show All\n"
            "-t TYPE       Server Type Filter:\n"
            "                1 = Show clan servers only\n"
            "                2 = Show public servers only\n"
            "-p TYPE       Password Filter:\n"
            "                1 = Show servers with passwords only\n"
            "                2 = Show servers without passwords only\n"
            "-n TYPE TEXT  Server Name Filter:\n"
            "                1 = Show servers containing TEXT\n"
            "                2 = Show servers starting with TEXT\n"
            "-u TYPE NUM   User Filter:\n"
            "                1 = Only empty servers\n"
            "                2 = Only servers with atleast NUM people\n"
            "                3 = Only servers with NUM people max\n"
            "                4 = Only full servers\n"
            "-c COUNTRY    Country Filter, case insensitive and partial (erman = GeRManY)\n"
            "\n"
            "Options:\n"
            "-r \"prog...\"  lets you to execute a specific program for each IP found (like\n"
            "              the ping command for example).\n"
            "              There are 2 available parameters and are #IP and #PORT that will\n"
            "              be substituited with the IP and port of the current online game\n"
            "              server found in the list. Example: -r \"echo #IP and #PORT yeah\"\n"
            "              Note that for performance reasons the program is executed during\n"
            "              the receiving of the data from the server!\n"
            "-s            IP:PORT visualization\n"
            "-x S[:P]      lets you to specify a different weblist server (S) and port (P,\n"
            "              optional) instead of %s:%hu\n"
            "\n", argv[0], tlhost, tlport);
        exit(1);
    }

    for(i = 1; i < argc; i++) {
        switch(argv[i][1]) {
            case 'a': {
                } break;
            case 't': {
                type = atoi(argv[++i]);
                } break;
            case 'p': {
                pass = atoi(argv[++i]);
                } break;
            case 'n': {
                namet = atoi(argv[++i]);
                i++;
                if(argv[i][0] == '-') {
                    i--;
                    continue;
                }
                if(i < argc) name = argv[i];
                } break;
            case 'u': {
                usert = atoi(argv[++i]);
                i++;
                if(argv[i][0] == '-') {
                    i--;
                    continue;
                }
                if(i < argc) user = atoi(argv[i]);
                } break;
            case 'c': {
                countryt = 1;
                get_country(&country, argv[++i]);
                fprintf(stderr, "- country: %s (%hu)\n",
                    get_country(&country, NULL), country);
                } break;
            case 'r': {
                exec_arg(argv[++i]);
                } break;
            case 's': {
                show_type = 1;
                } break;
            case 'x': {
                i++;
                tlhost = strchr(argv[i], ':');
                if(tlhost) {
                    tlport = atoi(tlhost + 1);
                    *tlhost = 0;
                }
                tlhost = argv[i];
                } break;
            default: {
                printf("\nError: wrong command-line argument (%s)\n\n", argv[i]);
                exit(1);
                } break;
        }
    }

    peer.sin_addr.s_addr = resolv(tlhost);
    peer.sin_port        = htons(tlport);
    peer.sin_family      = AF_INET;

    fprintf(stderr, "- connect to   %s : %hu\n",
        inet_ntoa(peer.sin_addr), tlport);

    sd = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
    if(sd < 0) std_err();
    if(connect(sd, (struct sockaddr *)&peer, sizeof(peer))
      < 0) std_err();

    fputs(
        "T P Users       Country         Server Address        Server Name\n"
        "-----------------------------------------------------------------------------\n",
        stderr);

    send_tl(sd,
        1,
        countryt,
        country,
        type,
        pass,
        usert,
        user,
        namet,
        name);

    memset(buff, 0, sizeof(buff));  // useless
    tl = (struct tspeak_weblist *)buff;

    for(servers = 0; recv_tl(sd, buff); servers++) {
        tl->ip = htonl(tl->ip);

        if(!show_type) {
            printf("%c %c %5hu/%-5hu %-15s %15s:%-5hu %s\n",
                (tl->info & 2) ? 'C' : 'P',     // "Clan" : "Public",
                (tl->info & 1) ? 'Y' : 'N',     // "Yes"  : "No",
                tl->users, tl->maxusers,
                get_country(&tl->country, NULL),
                inet_ntoa(*(struct in_addr *)&tl->ip), tl->port,
                buff + sizeof(struct tspeak_weblist));
        } else {
            printf("%s:%hu\n", inet_ntoa(*(struct in_addr *)&tl->ip), tl->port);
        }

        if(execstring) exec_run(tl->ip, tl->port);
    }

    close(sd);

    fprintf(stderr, "- %d servers online\n\n", servers);
    return(0);
}



void exec_arg(char *data) {
    execstring = data;
    execlen = strlen(execstring) + 23;

    tmpexec = malloc(execlen);
    if(!tmpexec) std_err();

    execstring_ip = strstr(execstring, "#IP");
    execstring_port = strstr(execstring, "#PORT");
    if(execstring_ip) *execstring_ip = 0;
    if(execstring_port) *execstring_port = 0;

    execlen = strlen(execstring);
    memcpy(tmpexec, execstring, execlen);
}



void exec_run(u32 ip, u16 port) {
    char    *ipc = inet_ntoa(*(struct in_addr *)&ip);

    execptr = tmpexec + execlen;
    if(execstring_ip && !execstring_port) {
        execptr += sprintf(execptr, "%s", ipc);
        strcpy(execptr, execstring_ip + 3);

    } else if(execstring_port && !execstring_ip) {
        execptr += sprintf(execptr, "%hu", port);
        strcpy(execptr, execstring_port + 5);

    } else if(execstring_ip < execstring_port) {
        execptr += sprintf(execptr, "%s", ipc);
        execptr += sprintf(execptr, "%s", execstring_ip + 3);
        execptr += sprintf(execptr, "%hu", port);
        strcpy(execptr, execstring_port + 5);

    } else if(execstring_port < execstring_ip) {
        execptr += sprintf(execptr, "%hu", port);
        execptr += sprintf(execptr, "%s", execstring_port + 5);
        execptr += sprintf(execptr, "%s", ipc);
        strcpy(execptr, execstring_ip + 3);
    }

    printf("   Execute: \"%s\"\n", tmpexec);
    system(tmpexec);
}



void send_tl(int sock, u8 query, u8 countryt, u16 country, u8 type, u8 pass, u8 usert, u16 user, u8 namet, char *name) {
    u8      len,
            namelen;

#pragma pack(1)
    struct tspeak_webquery {
        u8      query;
        u8      countryt;
        u16     country;
        u8      pass;
        u8      usert;
        u16     user;
        u8      type;
        u8      namet;
    } tw;
#pragma pack()

    namelen     = strlen(name) + 1;
    tw.query    = query;
    tw.countryt = countryt;
    tw.country  = country;
    tw.pass     = pass;
    tw.usert    = usert;
    tw.user     = user;
    tw.type     = type;
    tw.namet    = namet;

    len = sizeof(struct tspeak_webquery) + namelen;
    if(send(sock, &len,        1,       0) < 0) std_err();
    if(send(sock, (void *)&tw, sizeof(struct tspeak_webquery), 0) < 0) std_err();
    if(send(sock, name,        namelen, 0) < 0) std_err();
}



int recv_tl(int sock, u8 *data) {
    int     t;
    u8      rem,
            len;

    if(recv(sock, &len, 1, 0) <= 0) std_err();
    if(!len) return(0);

    for(rem = len; rem; data += t, rem -= t) {
        t = recv(sock, data, rem, 0);
        if(t < 0) std_err();
        if(!t) return(0);
    }

    return(len);
}



char *get_country(u16 *country_num, char *country) {
    int     i;
    char    tmp[6],
            *ret,
            *country_list[] = {
                "1000=Africa",
                "1100=Eastern Africa",
                "1116=Burundi",
                "1121=Comoros",
                "1125=Djibouti",
                "1128=Eritrea",
                "1129=Ethiopia",
                "1135=Kenya",
                "1139=Madagascar",
                "1140=Malawi",
                "1143=Mauritius",
                "1145=Mozambique",
                "1149=Runion",
                "1150=Rwanda",
                "1154=Seychelles",
                "1156=Somalia",
                "1160=Tanzania",
                "1163=Uganda",
                "1166=Zambia",
                "1167=Zimbabwe",
                "1200=Middle Africa",
                "1217=Cameroon",
                "1219=Central African Republic",
                "1220=Chad",
                "1222=Congo",
                "1227=Equatorial Guinea",
                "1230=Gabon",
                "1252=Sao Tome and Principe",
                "1300=Northern Africa",
                "1311=Algeria",
                "1326=Egypt",
                "1338=Libyan Arab Jamahiriya",
                "1344=Morocco",
                "1358=Sudan",
                "1362=Tunisia",
                "1364=Western Sahara",
                "1400=Southern Africa",
                "1412=Angola",
                "1414=Botswana",
                "1436=Lesotho",
                "1446=Namibia",
                "1451=Saint Helena",
                "1457=South Africa",
                "1459=Swaziland",
                "1470=Bouvet Island",
                "1471=French Southern Territories",
                "1472=Heard Island and McDonald Islands",
                "1473=Mayotte",
                "1500=Western Africa",
                "1513=Benin",
                "1515=Burkina Faso",
                "1518=Cape Verde",
                "1524=Cote d'Ivoire",
                "1531=Gambia",
                "1532=Ghana",
                "1533=Guinea",
                "1534=Guinea-Bissau",
                "1537=Liberia",
                "1541=venezuela",
                "1542=Mauritania",
                "1547=Niger",
                "1548=Nigeria",
                "1553=Senegal",
                "1555=Sierra Leone",
                "1561=Togo",
                "2000=Americas",
                "2100=Caribbean",
                "2111=Anguilla",
                "2112=Antigua and Barbuda",
                "2114=Aruba",
                "2115=Bahamas",
                "2116=Barbados",
                "2121=Virgin Islands (British)",
                "2122=Cayman Islands",
                "2126=Cuba",
                "2127=Dominica",
                "2128=Dominican Republic",
                "2134=Grenada",
                "2135=Guadeloupe",
                "2138=Haiti",
                "2140=Jamaica",
                "2141=Martinique",
                "2143=Montserrat",
                "2144=Netherlands Antilles",
                "2149=Puerto Rico",
                "2150=Saint Kitts and Nevis",
                "2151=Saint Lucia",
                "2153=Saint Vincent and the Grenadines",
                "2155=Trinidad and Tobago",
                "2156=Turks and Caicos Islands",
                "2158=Virgin Islands (United States)",
                "2200=Central America",
                "2217=Belize",
                "2225=Costa Rica",
                "2230=El Salvador",
                "2236=Guatemala",
                "2239=Honduras",
                "2242=Mexico",
                "2245=Nicaragua",
                "2246=Panama",
                "2300=South America",
                "2313=Argentina",
                "2319=Bolivia",
                "2320=Brazil",
                "2323=Chile",
                "2324=Colombia",
                "2329=Ecuador",
                "2331=Falkland Islands (Malvinas)",
                "2332=French Guiana",
                "2337=Guyana",
                "2347=Paraguay",
                "2348=Peru",
                "2354=Suriname",
                "2359=Uruguay",
                "2360=Venezuela",
                "2370=South Georgia and the South Sandwich Islands",
                "2400=Northern America",
                "2418=Bermuda",
                "2421=Canada",
                "2433=Greenland",
                "2452=Saint Pierre and Miquelon",
                "2457=United States",
                "3000=Asia",
                "3100=Eastern Asia",
                "3119=China",
                "3122=Hong Kong",
                "3128=Japan",
                "3131=North Korea",
                "3132=South Korea",
                "3137=Macao",
                "3140=Mongolia",
                "3180=Taiwan",
                "3181=Tibet",
                "3200=South-central Asia",
                "3211=Afghanistan",
                "3215=Bangladesh",
                "3216=Bhutan",
                "3223=India",
                "3225=Iran",
                "3230=Kazakhstan",
                "3234=Kyrgyzstan",
                "3239=Maldives",
                "3242=Nepal",
                "3244=Pakistan",
                "3250=Sri Lanka",
                "3252=Tajikistan",
                "3255=Turkmenistan",
                "3257=Uzbekistan",
                "3270=British Indian Ocean Territory",
                "3400=South-eastern Asia",
                "3417=Brunei Darussalam",
                "3418=Cambodia",
                "3423=East Timor",
                "3424=Indonesia",
                "3435=Lao People's Democratic Republic",
                "3438=Malaysia",
                "3441=Myanmar",
                "3446=Philippines",
                "3449=Singapore",
                "3453=Thailand",
                "3458=Viet Nam",
                "3471=Christmas Island",
                "3472=Cocos (Keeling) Islands",
                "3500=Western Asia",
                "3512=Armenia",
                "3513=Azerbaijan",
                "3514=Bahrain",
                "3520=Cyprus",
                "3521=Georgia",
                "3526=Iraq",
                "3527=Israel",
                "3529=Jordan",
                "3533=Kuwait",
                "3536=Lebanon",
                "3543=Oman",
                "3547=Qatar",
                "3548=Saudi Arabia",
                "3551=Syrian Arab Republic",
                "3554=Turkey",
                "3556=United Arab Emirates",
                "3559=Yemen",
                "4000=Europe",
                "4100=Eastern Europe",
                "4114=Belarus",
                "4117=Bulgaria",
                "4120=Czech Republic",
                "4132=Hungary",
                "4143=Moldova",
                "4147=Poland",
                "4149=Romania",
                "4150=Russian Federation",
                "4152=Slovakia",
                "4158=Ukraine",
                "4200=Northern Europe",
                "4218=Channel Islands",
                "4222=Denmark",
                "4223=Estonia",
                "4224=Faroe Islands",
                "4225=Finland",
                "4233=Iceland",
                "4234=Ireland",
                "4235=Isle of Man",
                "4237=Latvia",
                "4239=Lithuania",
                "4246=Norway",
                "4255=Svalbard and Jan Mayen Islands",
                "4256=Sweden",
                "4259=United Kingdom",
                "4281=Northern Ireland",
                "4300=Southern Europe",
                "4311=Albania",
                "4312=Andorra",
                "4316=Bosnia and Herzegovina",
                "4319=Croatia",
                "4329=Gibraltar",
                "4330=Greece",
                "4331=Holy See",
                "4336=Italy",
                "4341=Macedonia",
                "4342=Malta",
                "4348=Portugal",
                "4351=San Marino",
                "4353=Slovenia",
                "4354=Spain",
                "4361=Yugoslavia",
                "4380=Kosovo",
                "4400=Western Europe",
                "4413=Austria",
                "4415=Belgium",
                "4426=France",
                "4428=Germany",
                "4438=Liechtenstein",
                "4440=Luxembourg",
                "4444=Monaco",
                "4445=Netherlands",
                "4457=Switzerland",
                "5000=Oceania",
                "5100=Australia and New Zealand",
                "5112=Australia",
                "5122=New Zealand",
                "5124=Norfolk Island",
                "5200=Melanesia",
                "5214=Fiji",
                "5221=New Caledonia",
                "5227=Papua New Guinea",
                "5230=Solomon Islands",
                "5234=Vanuatu",
                "5300=Micronesia",
                "5316=Guam",
                "5317=Kiribati",
                "5318=Marshall Islands",
                "5319=Micronesia",
                "5320=Nauru",
                "5325=Northern Mariana Islands",
                "5326=Palau",
                "5400=Polynesia",
                "5411=American Samoa",
                "5413=Cook Islands",
                "5415=French Polynesia",
                "5423=Niue",
                "5428=Pitcairn",
                "5429=Samoa",
                "5431=Tokelau",
                "5432=Tonga",
                "5433=Tuvalu",
                "5435=Wallis and Futuna Islands",
                "6000=Antartica",
                NULL
    };

    ret = NULL;

    if(country) {
        *country_num = 0;
        for(i = 0; country_list[i]; i++) {
            if(stristr(country_list[i] + 5, country)) {
                *country_num = atoi(country_list[i]);
                break;
            }
        }
        return(country);
    } else {
        sprintf(tmp, "%hu", *country_num);
        for(i = 0; country_list[i]; i++) {
            if(!memcmp(country_list[i], tmp, 4)) return(country_list[i] + 5);
        }
        return("Unknown");
    }
}



u32 resolv(char *host) {
    struct  hostent *hp;
    u32     host_ip;

    host_ip = inet_addr(host);
    if(host_ip == INADDR_NONE) {
        hp = gethostbyname(host);
        if(!hp) {
            printf("\nError: Unable to resolv hostname (%s)\n", host);
            exit(1);
        } else host_ip = *(u32 *)hp->h_addr;
    }
    return(host_ip);
}



#ifndef WIN32
    void std_err(void) {
        perror("\nError");
        exit(1);
    }
#endif


