/*
    Copyright 2009-2011 Luigi Auriemma

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    http://www.gnu.org/licenses/gpl-2.0.txt
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <ctype.h>
#include <sys/stat.h>
#include <errno.h>
#include <zlib.h>
#include "blowfish_ttarch.h"

#ifdef WIN32
    #include <windows.h>
    #include <direct.h>
    #define PATHSLASH   '\\'
    #define make_dir(x) mkdir(x)
#else
    #include <dirent.h>
    #define stricmp     strcasecmp
    #define strnicmp    strncasecmp
    #define stristr     strcasestr
    #define PATHSLASH   '/'
    #define make_dir(x) mkdir(x, 0755)
#endif

typedef uint8_t     u8;
typedef uint32_t    u32;



#define VER         "0.1.12b"
#define DEFAULT_VERSION 1



int     ttarch_tot_idx  = 0,
        ttarch_chunks_b = 0,
        ttarch_rem      = 0,
        ttarch_offset   = 0,
        ttarch_baseoff  = 0,
        chunksz         = 0x10000;
u32     *ttarch_chunks  = NULL;

int     list_only       = 0,
        force_overwrite = 0,
        extracted_files = 0,
        meta_extract    = 0,
        old_mode        = 0,    // boring old mode
        version         = DEFAULT_VERSION,
        xmode           = 1,
        verbose         = 0;
u8      *filter_files   = NULL,
        *mykey          = NULL,
        *dump_table     = NULL;

typedef struct {
    u8      *name;
    int     offset; // unused at the moment
    int     size;
} files_t;
typedef struct {
    int     old_mode;
    u8      *key;
    u8      *name;
} gamekeys_t;
gamekeys_t  gamekeys[] = {
    //{ 0, "\x96\xCA\x99\xA0\x85\xCF\x98\x8A\xE4\xDB\xE2\xCD\xA6\xA9\xB8\xC4\xD8\x81\x8E\xE3\xA9\xD6\x99\xD4\xAC\x93\xDE\xC2\xA2\xE0\xA5\x99\xCC\xB4\x89\xB0\xCD\xCF\x9A\xCF\xDA\xBB\xEA\xDB\xA4\x9B\xC6\xB3\xAA\xD0\x95\x8A\xD1\xDE\xAB", "Wallace & Gromit demo" },
    { 0, "\x96\xCA\x99\xA0\x85\xCF\x98\x8A\xE4\xDB\xE2\xCD\xA6\x96\x83\x88\xC0\x8B\x99\xE3\x9E\xD8\x9B\xB6\xD7\x90\xDC\xBE\xAD\x9D\x91\x65\xB6\xA6\x9E\xBB\xC2\xC6\x9E\xB3\xE7\xE3\xE5\xD5\xAB\x63\x82\xA0\x9C\xC4\x92\x9F\xD1\xD5\xA4", "Wallace & Gromit: Episode 1: Fright of the Bumblebees" },
    { 0, "\x96\xCA\x99\xA0\x85\xCF\x98\x8A\xE4\xDB\xE2\xCD\xA6\x96\x83\x89\xC0\x8B\x99\xE3\x9E\xD8\x9B\xB6\xD7\x90\xDC\xBE\xAD\x9D\x91\x66\xB6\xA6\x9E\xBB\xC2\xC6\x9E\xB3\xE7\xE3\xE5\xD5\xAB\x63\x82\xA1\x9C\xC4\x92\x9F\xD1\xD5\xA4", "Wallace & Gromit: Episode 2: The Last Resort" },
    { 0, "\x96\xCA\x99\xA0\x85\xCF\x98\x8A\xE4\xDB\xE2\xCD\xA6\x96\x83\x8A\xC0\x8B\x99\xE3\x9E\xD8\x9B\xB6\xD7\x90\xDC\xBE\xAD\x9D\x91\x67\xB6\xA6\x9E\xBB\xC2\xC6\x9E\xB3\xE7\xE3\xE5\xD5\xAB\x63\x82\xA2\x9C\xC4\x92\x9F\xD1\xD5\xA4", "Wallace & Gromit: Episode 3: Muzzled" },
    { 0, "\x8f\xd8\x98\x99\x96\xbc\xa2\xae\xd7\xde\xc5\xd3\x9d\xca\xc5\xa7\xd8\x95\x92\xe9\x8d\xe4\xa1\xd4\xd7\x71\xde\xc0\x9e\xde\xb1\xa3\xca\xaa\xa4\x9f\xd0\xce\x9e\xde\xc5\xe3\xe3\xd1\xa9\x82\xc1\xda\xaa\xd5\x76\xa2\xdb\xd7\xb1", "Telltale Texas Hold'em" },
    { 0, "\x81\xd8\x9b\x99\x55\xe2\x65\x73\xb4\xdb\xe3\xc9\x63\xdb\x85\x87\xab\x99\x9b\xdc\x6e\xeb\x68\x9f\xa7\x90\xdd\xba\x6a\xe2\x93\x64\xa1\xb4\xa0\xb4\x92\xd9\x6b\x9c\xb7\xe3\xe6\xd1\x68\xa8\x84\x9f\x87\xd2\x94\x98\xa1\xe8\x71", "Bone: Out From Boneville" },
    { 0, "\x81\xD8\x9B\x99\x56\xE2\x65\x73\xB4\xDB\xE3\xC9\x64\xDB\x85\x87\xAB\x99\x9B\xDC\x6F\xEB\x68\x9F\xA7\x90\xDD\xBA\x6B\xE2\x93\x64\xA1\xB4\xA0\xB4\x93\xD9\x6B\x9C\xB7\xE3\xE6\xD1\x69\xA8\x84\x9F\x87\xD2\x94\x98\xA2\xE8\x71", "Bone: The Great Cow Race" },
    { 0, "\x92\xCA\x9A\x81\x85\xE4\x64\x73\xA3\xBF\xD6\xD1\x7F\xC6\xCB\x88\x99\x5B\x80\xD8\xAA\xC2\x97\xE7\x96\x51\xA0\xA8\x9A\xD9\xAE\x95\xD7\x76\x62\x80\xB4\xC4\xA6\xB9\xD6\xEC\xA9\x9C\x68\x85\xB3\xDC\x92\xC4\x9E\x64\xA0\xA3\x92", "Sam & Max: Episode 101 - Culture Shock" },
    { 0, "\x92\xCA\x9A\x81\x85\xE4\x64\x73\xA4\xBF\xD6\xD1\x7F\xC6\xCB\x88\x99\x01\x80\xD8\xAA\xC2\x97\xE7\x96\x51\xA1\xA8\x9A\xD9\xAE\x95\xD7\x76\x62\x81\xB4\xC4\xA6\xB9\xD6\xEC\xA9\x9C\x69\x85\xB3\xDC\x92\xC4\x9E\x64\xA0\xA4\x92", "Sam & Max: Episode 102 - Situation: Comedy" },
    { 0, "\x92\xca\x9a\x81\x85\xe4\x64\x73\xa5\xbf\xd6\xd1\x7f\xc6\xcb\x88\x99\x5d\x80\xd8\xaa\xc2\x97\xe7\x96\x51\xa2\xa8\x9a\xd9\xae\x95\xd7\x76\x62\x82\xb4\xc4\xa6\xb9\xd6\xec\xa9\x9c\x6a\x85\xb3\xdc\x92\xc4\x9e\x64\xa0\xa5\x92", "Sam & Max: Episode 103 - The Mole, The Mob, and the Meatball" },
    { 0, "\x92\xCA\x9A\x81\x85\xE4\x64\x73\xA6\xBF\xD6\xD1\x7F\xC6\xCB\x88\x99\x5E\x80\xD8\xAA\xC2\x97\xE7\x96\x51\xA3\xA8\x9A\xD9\xAE\x95\xD7\x76\x62\x83\xB4\xC4\xA6\xB9\xD6\xEC\xA9\x9C\x6B\x85\xB3\xDC\x92\xC4\x9E\x64\xA0\xA6\x92", "Sam & Max: Episode 104 - Abe Lincoln Must Die!" },
    { 0, "\x92\xCA\x9A\x81\x85\xE4\x64\x73\xA7\xBF\xD6\xD1\x7F\xC6\xCB\x88\x99\x5F\x80\xD8\xAA\xC2\x97\xE7\x96\x51\xA4\xA8\x9A\xD9\xAE\x95\xD7\x76\x62\x84\xB4\xC4\xA6\xB9\xD6\xEC\xA9\x9C\x6C\x85\xB3\xDC\x92\xC4\x9E\x64\xA0\xA7\x92", "Sam & Max: Episode 105 - Reality 2.0" },
    { 0, "\x92\xCA\x9A\x81\x85\xE4\x64\x73\xA8\xBF\xD6\xD1\x7F\xC6\xCB\x88\x99\x60\x80\xD8\xAA\xC2\x97\xE7\x96\x51\xA5\xA8\x9A\xD9\xAE\x95\xD7\x76\x62\x85\xB4\xC4\xA6\xB9\xD6\xEC\xA9\x9C\x6D\x85\xB3\xDC\x92\xC4\x9E\x64\xA0\xA8\x92", "Sam & Max: Episode 106 - Bright Side of the Moon" },
    { 0, "\x92\xca\x9a\x81\x85\xe4\x65\x73\xa3\xbf\xd6\xd1\x7f\xc6\xcb\x89\x99\x5b\x80\xd8\xaa\xc2\x97\xe7\x97\x51\xa0\xa8\x9a\xd9\xae\x95\xd7\x77\x62\x80\xb4\xc4\xa6\xb9\xd6\xec\xaa\x9c\x68\x85\xb3\xdc\x92\xc4\x9e\x65\xa0\xa3\x92", "Sam & Max: Episode 201 - Ice Station Santa" },
    { 0, "\x92\xCA\x9A\x81\x85\xE4\x65\x73\xA4\xBF\xD6\xD1\x7F\xC6\xCB\x89\x99\x01\x80\xD8\xAA\xC2\x97\xE7\x97\x51\xA1\xA8\x9A\xD9\xAE\x95\xD7\x77\x62\x81\xB4\xC4\xA6\xB9\xD6\xEC\xAA\x9C\x69\x85\xB3\xDC\x92\xC4\x9E\x65\xA0\xA4\x92", "Sam & Max: Episode 202 - Moai Better Blues" },
    { 0, "\x92\xCA\x9A\x81\x85\xE4\x65\x73\xA5\xBF\xD6\xD1\x7F\xC6\xCB\x89\x99\x5D\x80\xD8\xAA\xC2\x97\xE7\x97\x51\xA2\xA8\x9A\xD9\xAE\x95\xD7\x77\x62\x82\xB4\xC4\xA6\xB9\xD6\xEC\xAA\x9C\x6A\x85\xB3\xDC\x92\xC4\x9E\x65\xA0\xA5\x92", "Sam & Max: Episode 203 - Night of the Raving Dead" },
    { 0, "\x92\xCA\x9A\x81\x85\xE4\x65\x73\xA6\xBF\xD6\xD1\x7F\xC6\xCB\x89\x99\x5E\x80\xD8\xAA\xC2\x97\xE7\x97\x51\xA3\xA8\x9A\xD9\xAE\x95\xD7\x77\x62\x83\xB4\xC4\xA6\xB9\xD6\xEC\xAA\x9C\x6B\x85\xB3\xDC\x92\xC4\x9E\x65\xA0\xA6\x92", "Sam & Max: Episode 204 - Chariots of the Dogs" },
    { 0, "\x92\xca\x9a\x81\x85\xe4\x65\x73\xa7\xbf\xd6\xd1\x7f\xc6\xcb\x89\x99\x5f\x80\xd8\xaa\xc2\x97\xe7\x97\x51\xa4\xa8\x9a\xd9\xae\x95\xd7\x77\x62\x84\xb4\xc4\xa6\xb9\xd6\xec\xaa\x9c\x6c\x85\xb3\xdc\x92\xc4\x9e\x65\xa0\xa7\x92", "Sam & Max: Episode 205 - What's New, Beelzebub" },
    { 0, "\x87\xD8\x9A\x99\x97\xE0\x94\xB5\xA3\x9C\xA6\xAC\xA1\xD2\xB8\xCA\xDD\x8B\x9F\xA8\x6D\xA6\x7E\xDE\xD2\x86\xE2\xC9\x9A\xDE\x92\x64\x90\x8D\xA1\xBC\xC6\xD6\xAD\xCD\xE7\xA5\xA8\x9D\x7F\xA1\xBF\xD4\xB8\xD7\x87\xA5\xA1\xA2\x70", "Strong Bad: Episode 1 - Homestar Ruiner" },
    { 0, "\x87\xd8\x9a\x99\x97\xe0\x94\xb5\xa3\x9c\xa7\xac\xa1\xd2\xb8\xca\xdd\x8b\x9f\xa8\x6d\xa7\x7e\xde\xd2\x86\xe2\xc9\x9a\xde\x92\x64\x91\x8d\xa1\xbc\xc6\xd6\xad\xcd\xe7\xa5\xa8\x9e\x7f\xa1\xbf\xd4\xb8\xd7\x87\xa5\xa1\xa2\x71", "Strong Bad: Episode 2 - Strong Badia the Free" },
    { 0, "\x87\xD8\x9A\x99\x97\xE0\x94\xB5\xA3\x9C\xA8\xAC\xA1\xD2\xB8\xCA\xDD\x8B\x9F\xA8\x6D\xA8\x7E\xDE\xD2\x86\xE2\xC9\x9A\xDE\x92\x64\x92\x8D\xA1\xBC\xC6\xD6\xAD\xCD\xE7\xA5\xA8\x9F\x7F\xA1\xBF\xD4\xB8\xD7\x87\xA5\xA1\xA2\x72", "Strong Bad: Episode 3 - Baddest of the Bands" },
    { 0, "\x87\xd8\x9a\x99\x97\xe0\x94\xb5\xa3\x9c\xa9\xac\xa1\xd2\xb8\xca\xdd\x8b\x9f\xa8\x6d\xa9\x7e\xde\xd2\x86\xe2\xc9\x9a\xde\x92\x64\x93\x8d\xa1\xbc\xc6\xd6\xad\xcd\xe7\xa5\xa8\xa0\x7f\xa1\xbf\xd4\xb8\xd7\x87\xa5\xa1\xa2\x73", "Strong Bad: Episode 4 - Daneresque 3" },
    { 0, "\x87\xd8\x9a\x99\x97\xe0\x94\xb5\xa3\x9c\xaa\xac\xa1\xd2\xb8\xca\xdd\x8b\x9f\xa8\x6d\xaa\x7e\xde\xd2\x86\xe2\xc9\x9a\xde\x92\x64\x94\x8d\xa1\xbc\xc6\xd6\xad\xcd\xe7\xa5\xa8\xa1\x7f\xa1\xbf\xd4\xb8\xd7\x87\xa5\xa1\xa2\x74", "Strong Bad: Episode 5 - 8-Bit Is Enough" },
    { 1, "\x34\x24\x6C\x33\x43\x72\x6C\x75\x64\x32\x65\x53\x57\x69\x45\x32\x4F\x61\x63\x39\x6C\x75\x74\x78\x6C\x37\x32\x52\x2D\x2A\x38\x49\x31\x71\x4F\x34\x6F\x61\x6A\x6C\x5F\x24\x65\x23\x69\x61\x63\x70\x34\x2A\x75\x46\x6C\x65\x30", "CSI 3 - Dimensions of Murder" },
    { 0, "\x82\xBC\x76\x68\x83\xB0\x78\x90\xC1\xAF\xC8\xAD\x66\xC4\x97\x9C\xB6\x79\x70\xCA\x86\xA9\x95\xB3\xAA\x6E\xBE\x98\x8C\xB5\x95\x93\xA3\x8A\x7F\x9E\xA4\xB6\x82\xA0\xD4\xB8\xBD\xB9\x86\x75\xA5\xB8\x79\xC2\x6A\x78\xBD\xC1\x82", "CSI 4 - Hard Evidence (demo)" },
    { 0, "\x8C\xD8\x9B\x9F\x89\xE5\x7C\xB6\xDE\xCD\xE3\xC8\x63\x95\x84\xA4\xD8\x98\x98\xDC\xB6\xBE\xA9\xDB\xC6\x8F\xD3\x86\x69\x9D\xAE\xA3\xCD\xB0\x97\xC8\xAA\xD6\xA5\xCD\xE3\xD8\xA9\x9C\x68\x7F\xC1\xDD\xB0\xC8\x9F\x7C\xE3\xDE\xA0", "Tales of Monkey Island 101: Launch of the Screaming Narwhal" },
    { 0, "\x96\xCA\x99\xA0\x85\xCF\x98\x8A\xE4\xDB\xE2\xCD\xA6\x96\x83\x8B\xC0\x8B\x99\xE3\x9E\xD8\x9B\xB6\xD7\x90\xDC\xBE\xAD\x9D\x91\x68\xB6\xA6\x9E\xBB\xC2\xC6\x9E\xB3\xE7\xE3\xE5\xD5\xAB\x63\x82\xA3\x9C\xC4\x92\x9F\xD1\xD5\xA4", "Wallace & Gromit: Episode 4: The Bogey Man" },
    { 0, "\x8C\xD8\x9B\x9F\x89\xE5\x7C\xB6\xDE\xCD\xE3\xC8\x63\x95\x85\xA4\xD8\x98\x98\xDC\xB6\xBE\xA9\xDB\xC6\x8F\xD3\x86\x69\x9E\xAE\xA3\xCD\xB0\x97\xC8\xAA\xD6\xA5\xCD\xE3\xD8\xA9\x9C\x69\x7F\xC1\xDD\xB0\xC8\x9F\x7C\xE3\xDE\xA0", "Tales of Monkey Island 102: The Siege of Spinner Cay" },
    { 0, "\x8C\xD8\x9B\x9F\x89\xE5\x7C\xB6\xDE\xCD\xE3\xC8\x63\x95\x86\xA4\xD8\x98\x98\xDC\xB6\xBE\xA9\xDB\xC6\x8F\xD3\x86\x69\x9F\xAE\xA3\xCD\xB0\x97\xC8\xAA\xD6\xA5\xCD\xE3\xD8\xA9\x9C\x6A\x7F\xC1\xDD\xB0\xC8\x9F\x7C\xE3\xDE\xA0", "Tales of Monkey Island 103: Lair of the Leviathan" },
    { 0, "\x82\xBC\x76\x69\x54\x9C\x86\x90\xD7\xDA\xEA\xA7\x85\xAE\x88\x87\x99\x7D\x7A\xDC\xAB\xEA\x79\xC2\xAE\x56\x9F\x85\x8C\xB9\xC6\xA2\xD4\x88\x85\x98\x96\x93\x69\xBF\xC2\xD9\xE6\xE1\x7A\x85\x9B\xA4\x75\x93\x79\x80\xD5\xE0\xB4", "CSI 5 - Deadly Intent" },
    { 0, "\x8c\xd8\x9b\x9f\x89\xe5\x7c\xb6\xde\xcd\xe3\xc8\x63\x95\x87\xa4\xd8\x98\x98\xdc\xb6\xbe\xa9\xdb\xc6\x8f\xd3\x86\x69\xa0\xae\xa3\xcd\xb0\x97\xc8\xaa\xd6\xa5\xcd\xe3\xd8\xa9\x9c\x6b\x7f\xc1\xdd\xb0\xc8\x9f\x7c\xe3\xde\xa0", "Tales of Monkey Island 104: The Trial and Execution of Guybrush Threepwood" },
    { 0, "\x82\xbc\x76\x68\x67\xbf\x7c\x77\xb5\xbf\xbe\x98\x75\xb8\x9c\x8b\xac\x7d\x76\xab\x80\xc8\x7f\xa3\xa8\x74\xb8\x89\x7c\xbf\xaa\x68\xa2\x98\x7b\x83\xa4\xb6\x82\xa0\xb8\xc7\xc1\xa0\x7a\x85\x9b\xa3\x88\xb6\x6f\x67\xb3\xc5\x88", "CSI 4 - Hard Evidence" },
    { 0, "\x8c\xd8\x9b\x9f\x89\xe5\x7c\xb6\xde\xcd\xe3\xc8\x63\x95\x88\xa4\xd8\x98\x98\xdc\xb6\xbe\xa9\xdb\xc6\x8f\xd3\x86\x69\xa1\xae\xa3\xcd\xb0\x97\xc8\xaa\xd6\xa5\xcd\xe3\xd8\xa9\x9c\x6c\x7f\xc1\xdd\xb0\xc8\x9f\x7c\xe3\xde\xa0", "Tales of Monkey Island 105: Rise of the Pirate God" },
    { 0, "\x82\xBC\x76\x69\x68\xD1\xA0\xB2\xB5\xBF\xBE\x99\x76\xCA\xC0\xC6\xAC\x7D\x76\xAC\x81\xDA\xA3\xDE\xA8\x74\xB8\x8A\x7D\xD1\xCE\xA3\xA2\x98\x7B\x84\xA5\xC8\xA6\xDB\xB8\xC7\xC1\xA1\x7B\x97\xBF\xDE\x88\xB6\x6F\x68\xB4\xD7\xAC", "CSI 5 - Deadly Intent (demo)" },
    { 0, "\x92\xCA\x9A\x81\x85\xE4\x66\x73\xA3\xBF\xD6\xD1\x7F\xC6\xCB\x8A\x99\x5B\x80\xD8\xAA\xC2\x97\xE7\x98\x51\xA0\xA8\x9A\xD9\xAE\x95\xD7\x78\x62\x80\xB4\xC4\xA6\xB9\xD6\xEC\xAB\x9C\x68\x85\xB3\xDC\x92\xC4\x9E\x66\xA0\xA3\x92", "Sam & Max: Episode 301 - The Penal Zone" },
    { 0, "\x92\xCA\x9A\x81\x85\xE4\x66\x73\xA4\xBF\xD6\xD1\x7F\xC6\xCB\x8A\x99\x01\x80\xD8\xAA\xC2\x97\xE7\x98\x51\xA1\xA8\x9A\xD9\xAE\x95\xD7\x78\x62\x81\xB4\xC4\xA6\xB9\xD6\xEC\xAB\x9C\x69\x85\xB3\xDC\x92\xC4\x9E\x66\xA0\xA4\x92", "Sam & Max: Episode 302 - The Tomb of Sammun-Mak" },
    { 0, "\x92\xCA\x9A\x81\x85\xE4\x66\x73\xA5\xBF\xD6\xD1\x7F\xC6\xCB\x8A\x99\x5D\x80\xD8\xAA\xC2\x97\xE7\x98\x51\xA2\xA8\x9A\xD9\xAE\x95\xD7\x78\x62\x82\xB4\xC4\xA6\xB9\xD6\xEC\xAB\x9C\x6A\x85\xB3\xDC\x92\xC4\x9E\x66\xA0\xA5\x92", "Sam & Max: Episode 303 - They Stole Max's Brain!" },
    { 0, "\x86\xDB\x96\x97\x8F\xD8\x98\x74\xA2\x9D\xBC\xD6\x9B\xC8\xBE\xC3\xCE\x5B\x5D\xA8\x84\xE7\x9F\xD2\xD0\x8D\xD4\x86\x69\x9D\xA8\xA6\xC8\xA8\x9D\xBB\xC6\x94\x69\x9D\xBC\xE6\xE1\xCF\xA2\x9E\xB7\xA0\x75\x94\x6D\xA5\xD9\xD5\xAA", "Puzzle Agent - The Mystery of Scoggins" },
    { 0, "\x92\xCA\x9A\x81\x85\xE4\x66\x73\xA6\xBF\xD6\xD1\x7F\xC6\xCB\x8A\x99\x5E\x80\xD8\xAA\xC2\x97\xE7\x98\x51\xA3\xA8\x9A\xD9\xAE\x95\xD7\x78\x62\x83\xB4\xC4\xA6\xB9\xD6\xEC\xAB\x9C\x6B\x85\xB3\xDC\x92\xC4\x9E\x66\xA0\xA6\x92", "Sam & Max: Episode 304 - Beyond the Alley of the Dolls" },
    { 0, "\x92\xCA\x9A\x81\x85\xE4\x66\x73\xA7\xBF\xD6\xD1\x7F\xC6\xCB\x8A\x99\x5F\x80\xD8\xAA\xC2\x97\xE7\x98\x51\xA4\xA8\x9A\xD9\xAE\x95\xD7\x78\x62\x84\xB4\xC4\xA6\xB9\xD6\xEC\xAB\x9C\x6C\x85\xB3\xDC\x92\xC4\x9E\x66\xA0\xA7\x92", "Sam & Max: Episode 305 - The City That Dares Not Sleep" },
    { 0, "\x82\xCE\x99\x99\x86\xDE\x9C\xB7\xEB\xBC\xE4\xCF\x97\xD7\x96\xBC\xD5\x8F\x8F\xE9\xA6\xE9\xAF\xBF\xD4\x8C\xD4\xC7\x7C\xD1\xCD\x99\xC1\xB7\x9B\xC3\xDA\xB3\xA8\xD7\xDA\xE6\xBB\xD1\xA3\x97\xB4\xE1\xAE\xD7\x9F\x83\xDF\xDD\xA4", "Poker Night at the Inventory" },
    { 0, "\x82\xBC\x76\x6A\x54\x9C\x76\x96\xBB\xA2\xA5\x94\x75\xB8\x9C\x8D\x99\x5A\x70\xCA\x86\xAB\x66\x9F\xA8\x74\xB8\x8B\x69\x9C\xA4\x87\xA8\x7B\x62\x7F\xA4\xB6\x82\xA2\xA5\xA4\xBB\xBF\x80\x68\x82\x9F\x88\xB6\x6F\x69\xA0\xA2\x82", "CSI 6 - Fatal Conspiracy" },
    { 0, "\x81\xCA\x90\x9F\x78\xDB\x87\xAB\xD7\xB2\xEA\xD8\xA7\xD7\xB8\x88\x99\x5B\x6F\xD8\xA0\xE0\x8A\xDE\xB9\x89\xD4\x9B\xAE\xE0\xD6\xA6\xC4\x76\x62\x80\xA3\xC4\x9C\xD7\xC9\xE3\xCC\xD4\x9C\x78\xC7\xE3\xBA\xD5\x8B\x64\xA0\xA3\x81", "Back To The Future: Episode 1 - It's About Time" },
    { 0, "\x81\xca\x90\x9f\x78\xdb\x87\xab\xd7\xb2\xea\xd8\xa7\xd7\xb8\x88\x99\x01\x6f\xd8\xa0\xe0\x8a\xde\xb9\x89\xd4\x9b\xae\xe0\xd6\xa6\xc4\x76\x62\x81\xa3\xc4\x9c\xd7\xc9\xe3\xcc\xd4\x9c\x78\xc7\xe3\xba\xd5\x8b\x64\xa0\xa4\x81", "Back To The Future: Episode 2 - Get Tannen!" },
    { 0, "\x81\xCA\x90\x9F\x78\xDB\x87\xAB\xD7\xB2\xEA\xD8\xA7\xD7\xB8\x88\x99\x5D\x6F\xD8\xA0\xE0\x8A\xDE\xB9\x89\xD4\x9B\xAE\xE0\xD6\xA6\xC4\x76\x62\x82\xA3\xC4\x9C\xD7\xC9\xE3\xCC\xD4\x9C\x78\xC7\xE3\xBA\xD5\x8B\x64\xA0\xA5\x81", "Back To The Future: Episode 3 - Citizen Brown" },
    { 0, "\x87\xCE\x90\xA8\x93\xDE\x64\x73\xA3\xB4\xDA\xC7\xA6\xD4\xC5\x88\x99\x5B\x75\xDC\xA0\xE9\xA5\xE1\x96\x51\xA0\x9D\x9E\xCF\xD5\xA3\xD1\x76\x62\x80\xA9\xC8\x9C\xE0\xE4\xE6\xA9\x9C\x68\x7A\xB7\xD2\xB9\xD2\x98\x64\xA0\xA3\x87", "Hector: Episode 1 - We Negotiate with Terrorists" },
    { 0, "\x81\xCA\x90\x9F\x78\xDB\x87\xAB\xD7\xB2\xEA\xD8\xA7\xD7\xB8\x88\x99\x5E\x6F\xD8\xA0\xE0\x8A\xDE\xB9\x89\xD4\x9B\xAE\xE0\xD6\xA6\xC4\x76\x62\x83\xA3\xC4\x9C\xD7\xC9\xE3\xCC\xD4\x9C\x78\xC7\xE3\xBA\xD5\x8B\x64\xA0\xA6\x81", "Back To The Future: Episode 4 - Double Visions" },
    { 0, "\x81\xCA\x90\x9F\x78\xDB\x87\xAB\xD7\xB2\xEA\xD8\xA7\xD7\xB8\x88\x99\x5F\x6F\xD8\xA0\xE0\x8A\xDE\xB9\x89\xD4\x9B\xAE\xE0\xD6\xA6\xC4\x76\x62\x84\xA3\xC4\x9C\xD7\xC9\xE3\xCC\xD4\x9C\x78\xC7\xE3\xBA\xD5\x8B\x64\xA0\xA7\x81", "Back To The Future: Episode 5 - OUTATIME" },
    { 0, "\x86\xDB\x96\x97\x8F\xD8\x98\x74\xA2\x9E\xBC\xD6\x9B\xC8\xBE\xC3\xCE\x5B\x5D\xA9\x84\xE7\x9F\xD2\xD0\x8D\xD4\x86\x69\x9E\xA8\xA6\xC8\xA8\x9D\xBB\xC6\x94\x69\x9E\xBC\xE6\xE1\xCF\xA2\x9E\xB7\xA0\x75\x95\x6D\xA5\xD9\xD5\xAA", "Puzzle Agent 2" },
    { 0, NULL, NULL }
};



int ttarch_import(FILE *fdo, u8 *fname);
int pad_it(int num, int pad);
int rebuild_it(FILE *fdo);
files_t *add_files(u8 *fname, int fsize, int *ret_files);
int recursive_dir(u8 *filedir);
u32 crypt_it(FILE *fd, u8 *fname, u32 offset, int wanted_size, int encrypt);
u8 *string2key(u8 *data);
void ttarch_extract(FILE *fd, u8 *input_fname);
int ttarch_meta_crypt(u8 *data, int datalen, int encrypt);
int ttarch_fseek(FILE *stream, int offset, int origin);
int ttarch_fread(void *ptr, int size, FILE *stream);
void xor(u8 *data, int datalen, int xornum);
void blowfish(u8 *data, int datalen, int encrypt);
int ttarch_dumpa(u8 *fname, u8 *data, int size, int already_decrypted);
int unzip(u8 *in, int insz, u8 *out, int outsz);
int check_wildcard(u8 *fname, u8 *wildcard);
u8 *create_dir(u8 *name);
int check_overwrite(u8 *fname);
void myalloc(u8 **data, int wantsize, int *currsize);
int getxx(u8 *tmp, int bytes);
int fgetxx(FILE *fd, int bytes);
int myfr(FILE *fd, u8 *data, int size);
int putxx(u8 *data, u32 num, int bytes);
int fputxx(FILE *fd, u32 num, int bytes);
void dumpa(u8 *fname, u8 *data, int size);
int myfw(FILE *fd, u8 *data, int size);
int get_num(u8 *str);
void std_err(void);



int main(int argc, char *argv[]) {
    FILE    *fd;
    int     i,
            gamenum,
            rebuild         = 0,
            decrypt_only    = -1,
            decrypt_onlysz  = -1,
            encrypt_only    = -1,
            encrypt_onlysz  = -1,
            force_old_mode  = 0;
    u8      *fname,
            *fdir,
            *ext,
            *p;

    setbuf(stdout, NULL);

    fputs("\n"
        "Telltale TTARCH files extractor/rebuilder "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", stdout);

    if(argc < 4) {
        printf("\n"
            "Usage: %s [options] <gamenum> <file.TTARCH> <output_folder>\n"
            "\n"
            "Options:\n"
            "-l      list the files without extracting them, you can use . as output folder\n"
            "-f W    filter the files to extract using the W wildcard, example -f \"*.mp3\"\n"
            "-o      if the output files already exist this option will overwrite them\n"
            "        automatically without asking the user's confirmation\n"
            "-m      automatically extract the data inside the meta files, for example the\n"
            "        DDS from FONT and D3DTX files and OGG from AUD and so on... USEFUL!\n"
            "-b      rebuild option, instead of extracting it performs the building of the\n"
            "        ttarch archive, example: ttarchext -b 24 output.ttarch input_folder\n"
            "        do NOT use -m in the extraction if you want to rebuild the archive!\n"
            "-k K    specify a custom key in hexadecimal (like 0011223344), use gamenum 0\n"
            "-d OFF  perform only the blowfish decryption of the input file from offset OFF\n"
            "-D O S  as above but the decryption is performed only on the piece of file\n"
            "        from offset O for a total of S bytes, the rest will remain as is\n"
            "-e OFF  perform the blowfish encryption of the input file (debug)\n"
            "-E O S  as above but the encryption is performed only on the piece of file\n"
            "        from offset O for a total of S bytes, the rest will remain as is\n"
            "-V VER  force a specific version number, needed ONLY with -d and -e when\n"
            "        handling archives of version 7 (like Tales of Monkey Island)\n"
            "-O      force the old mode format (needed sometimes with old archives, debug)\n"
            "-x      for versions >= 7 only in rebuild mode, if the game crashes when uses\n"
            "        the rebuilt archive try to rebuild it adding this -x option\n"
            "-T F    dump the decrypted name table in the file F (debug)\n"
            "\n", argv[0]);

        printf("Games (gamenum):\n");
        for(i = 0; gamekeys[i].name; i++) {
            printf(" %-3d %s\n", i, gamekeys[i].name);
        }
        printf("\n"
            "Examples:\n"
            "  ttarchext.exe -m 24 c:\\4_MonkeyIsland101_pc_tx.ttarch \"c:\\new folder\"\n"
            "  ttarchext.exe -l 24 c:\\4_MonkeyIsland101_pc_tx.ttarch \"c:\\new folder\"\n"
            "  ttarchext.exe -b -V 7 24 c:\\1_output.ttarch c:\\1_input_folder\n"
            "\n"
            "the tool can work also on single encrypted files like the prop files usually\n"
            "located in the main folder or the various aud files, examples:\n"
            "  ttarchext.exe -V 7 24 c:\\ttg_splash_a_telltale.d3dtx \"c:\\new folder\"\n"
            "  ttarchext.exe -V 7 -e 0 24 \"c:\\new folder\\ttg_splash_a_telltale.d3dtx\" c:\\\n"
            "\n");
        exit(1);
    }

    argc -= 3;
    for(i = 1; i < argc; i++) {
        if(((argv[i][0] != '-') && (argv[i][0] != '/')) || (strlen(argv[i]) != 2)) {
            printf("\nError: wrong argument (%s)\n", argv[i]);
            exit(1);
        }
        switch(argv[i][1]) {
            case 'l': list_only         = 1;                    break;
            case 'f': filter_files      = argv[++i];            break;
            case 'o': force_overwrite   = 1;                    break;
            case 'm': meta_extract      = 1;                    break;
            case 'b': rebuild           = 1;                    break;
            case 'k': mykey             = argv[++i];            break;
            case 'd': decrypt_only      = get_num(argv[++i]);   break;
            case 'D': {
                decrypt_only            = get_num(argv[++i]);
                decrypt_onlysz          = get_num(argv[++i]);
                break;
            }
            case 'e': encrypt_only      = get_num(argv[++i]);   break;
            case 'E': {
                encrypt_only            = get_num(argv[++i]);
                encrypt_onlysz          = get_num(argv[++i]);
                break;
            }
            case 'V': version           = get_num(argv[++i]);   break;
            case 'O': force_old_mode    = 1;                    break;
            case 'x': xmode             = 0;                    break;
            case 'T': dump_table        = argv[++i];            break;
            case 'v': verbose           = 1;                    break;
            default: {
                printf("\nError: wrong argument (%s)\n", argv[i]);
                exit(1);
            }
        }
    }
    for(i = 0; i < 3; i++) {    // check, it's easy to forget some parameters
        if((argv[argc + i][0] == '-') && (strlen(argv[argc + i]) == 2)) {
            fprintf(stderr, "\n"
                "Error: seems that you have missed one of the needed parameters.\n"
                "       launch this tool without arguments for a quick help\n");
            exit(1);
        }
    }

    gamenum = atoi(argv[argc]);
    fname   = argv[argc + 1];
    fdir    = argv[argc + 2];

    if(rebuild) {
        printf("- start building of %s\n", fname);
        if(check_overwrite(fname) < 0) exit(1);
        fd = fopen(fname, "wb");
    } else {
        printf("- open file %s\n", fname);
        fd = fopen(fname, "rb");
    }
    if(!fd) std_err();

    if(!list_only) {
        printf("- set output folder %s\n", fdir);
        if(chdir(fdir) < 0) std_err();
    }

    if(mykey) {
        printf("- set custom blowfish key\n");
        mykey = string2key(mykey);
    } else {
        for(i = 0; i < gamenum; i++) {
            if(!gamekeys[i].name) break;
        }
        if(!gamekeys[i].name) {
            printf("\nError: the number of key you choosed is too big\n");
            exit(1);
        }
        printf("- set the blowfish key of \"%s\"\n", gamekeys[i].name);
        mykey    = gamekeys[i].key;
        old_mode = gamekeys[i].old_mode;
    }
    if(force_old_mode) old_mode = 1;
    //blowfish(NULL, 0, mykey, strlen(mykey), 0);   // no longer used

    p = strrchr(fname, '\\');
    if(!p) p = strrchr(fname, '/');
    if(p) fname = p + 1;

    if(rebuild) {
        extracted_files = rebuild_it(fd);
        printf("\n- %d files found\n", extracted_files);
    } else {
        if(decrypt_only >= 0) {
            i = crypt_it(fd, fname, decrypt_only, decrypt_onlysz, 0);
            printf("\n- decrypted %d bytes from offset 0x%08x\n", i, decrypt_only);
        } else if(encrypt_only >= 0) {
            i = crypt_it(fd, fname, encrypt_only, encrypt_onlysz, 1);
            printf("\n- encrypted %d bytes from offset 0x%08x\n", i, encrypt_only);
        } else {
            ext = strrchr(fname, '.');
            if(ext && !stricmp(ext, ".ttarch")) {
                ttarch_extract(fd, fname);
            } else {
                ttarch_extract(fd, fname);
            }
            printf("\n- %d files found\n", extracted_files);
        }
    }

    fclose(fd);
    printf("- done\n");
    return(0);
}



int ttarch_import(FILE *fdo, u8 *fname) {
    static int  buffsz  = 0;
    static u8   *buff   = NULL;
    struct stat xstat;
    FILE    *fd;
    int     size;

    printf("- import %s\n", fname);
    fd = fopen(fname, "rb");
    if(!fd) std_err();
    fstat(fileno(fd), &xstat);
    size = xstat.st_size;
    if(buffsz < size) {
        buffsz = size;
        buff = realloc(buff, buffsz);
        if(!buff) std_err();
    }
    myfr(fd, buff, size);
    fclose(fd);

    ttarch_meta_crypt(buff, size, 1);
    myfw(fdo, buff, size);
    return(size);
}



int pad_it(int num, int pad) {
    int     t;

    t = num % pad;
    if(t) num += pad - t;
    return(num);
}



int rebuild_it(FILE *fdo) {
    int     i,
            tmp,
            off,
            tot         = 0,
            totdirs     = 0,
            info_size,
            data_size;
    files_t *files      = NULL;
    u8      filedir[4096],
            *folders[]  = { NULL },
            *info_table = NULL,
            *p;

    if(version == DEFAULT_VERSION) {
        printf("\n"
            "Error: you must use the -V option with the rebuild one\n"
            "       for example -V 7 for Monkey Island, you can see the version of the\n"
            "       original archive when you extract or list it\n");
        exit(1);
    }

    printf("- start recursive scanning\n");
    strcpy(filedir, ".");
    recursive_dir(filedir);
    files = add_files(NULL, 0, &tot);
    printf("- collected %d files\n", tot);

    info_size = 0;
    data_size = 0;

    info_size += 4;     // folders
    for(i = 0; folders[i]; i++) {
        info_size += 4 + strlen(folders[i]);
        totdirs++;
    }
    info_size += 4;     // files
    for(i = 0; i < tot; i++) {
        info_size += 4 + strlen(files[i].name) + 4 + 4 + 4;
        data_size += files[i].size;
    }
    if(version <= 2) {
        info_size += 4 + 4 + 4 + 4;
    }
    info_size = pad_it(info_size, 8);   // 8 for blow_fish
    info_table = calloc(info_size, 1);
    if(!info_table) std_err();

    printf("- start building of the file\n");
    fputxx(fdo, version,    4); // version
    fputxx(fdo, 1,          4); // info_mode
    fputxx(fdo, 2,          4); // type3
    if(version >= 3) {
        fputxx(fdo, 1,      4); // files_mode
    }
    if(version >= 3) {
        fputxx(fdo, 0,      4); // ttarch_tot_idx
        fputxx(fdo, data_size, 4);
        if(version >= 4) {
            fputxx(fdo, 0,  4);
            fputxx(fdo, 0,  4);
            if(version >= 7) {
                fputxx(fdo, xmode,  4);
                fputxx(fdo, xmode,  4);
                fputxx(fdo, 0x40,   4);
                if(version >= 8) {
                    fputxx(fdo, 0,  1);
                }
            }
        }
    }
    fputxx(fdo, info_size,  4);

    off = 0;
    p = info_table;
    p += putxx(p, totdirs,  4); // folders
    for(i = 0; i < totdirs; i++) {
        p += putxx(p, strlen(folders[i]), 4);
        p += sprintf(p, "%s", folders[i]);
    }
    p += putxx(p, tot,      4); // files
    for(i = 0; i < tot; i++) {
        p += putxx(p, strlen(files[i].name), 4);
        p += sprintf(p, "%s", files[i].name);
        p += putxx(p, 0,    4);
        p += putxx(p, off,  4);
        p += putxx(p, files[i].size, 4);
        off += files[i].size;
    }
    if(version <= 2) {
        p += putxx(p, info_size + 4, 4);
        p += putxx(p, data_size, 4);
        p += putxx(p, 0xfeedface, 4);
        p += putxx(p, 0xfeedface, 4);
    }
    if(pad_it(p - info_table, 8) != info_size) {
        printf("\nError: problem in the info_size calculated by ttarchext (%d, %d)\n", p - info_table, info_size);
        exit(1);
    }

    blowfish(info_table, info_size, 1);
    myfw(fdo, info_table, info_size);

    tmp = 0;
    for(i = 0; i < tot; i++) {
        tmp += ttarch_import(fdo, files[i].name);
    }
    if(tmp != data_size) {
        printf("\nError: problem in the data_size calculated by ttarchext (%d, %d)\n", tmp, data_size);
        exit(1);
    }

    return(tot);
}



files_t *add_files(u8 *fname, int fsize, int *ret_files) {
    static int      filesi  = 0,
                    filesn  = 0;
    static files_t  *files  = NULL;

    if(ret_files) {
        *ret_files = filesi;
        return(files);
    }

    if(filesi >= filesn) {
        filesn += 1024;
        files = realloc(files, sizeof(files_t) * filesn);
        if(!files) std_err();
    }
    files[filesi].name   = strdup(fname);
    files[filesi].offset = 0;
    files[filesi].size   = fsize;
    filesi++;
    return(NULL);
}



int recursive_dir(u8 *filedir) {
    int     plen,
            ret     = -1;

#ifdef WIN32
    static int      winnt = -1;
    OSVERSIONINFO   osver;
    WIN32_FIND_DATA wfd;
    HANDLE  hFind;

    if(winnt < 0) {
        osver.dwOSVersionInfoSize = sizeof(OSVERSIONINFO);
        GetVersionEx(&osver);
        if(osver.dwPlatformId >= VER_PLATFORM_WIN32_NT) {
            winnt = 1;
        } else {
            winnt = 0;
        }
    }

    plen = strlen(filedir);
    strcpy(filedir + plen, "\\*.*");
    plen++;

    if(winnt) { // required to avoid problems with Vista and Windows7!
        hFind = FindFirstFileEx(filedir, FindExInfoStandard, &wfd, FindExSearchNameMatch, NULL, 0);
    } else {
        hFind = FindFirstFile(filedir, &wfd);
    }
    if(hFind == INVALID_HANDLE_VALUE) return(0);
    do {
        if(!strcmp(wfd.cFileName, ".") || !strcmp(wfd.cFileName, "..")) continue;

        strcpy(filedir + plen, wfd.cFileName);

        if(wfd.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) {
            if(recursive_dir(filedir) < 0) goto quit;
        } else {
            add_files(filedir + 2, wfd.nFileSizeLow, NULL);
        }
    } while(FindNextFile(hFind, &wfd));
    ret = 0;

quit:
    FindClose(hFind);
#else
    struct  stat    xstat;
    struct  dirent  **namelist;
    int     n,
            i;

    n = scandir(filedir, &namelist, NULL, NULL);
    if(n < 0) {
        if(stat(filedir, &xstat) < 0) {
            printf("**** %s", filedir);
            std_err();
        }
        add_files(filedir + 2, xstat.st_size, NULL);
        return(0);
    }

    plen = strlen(filedir);
    strcpy(filedir + plen, "/");
    plen++;

    for(i = 0; i < n; i++) {
        if(!strcmp(namelist[i]->d_name, ".") || !strcmp(namelist[i]->d_name, "..")) continue;
        strcpy(filedir + plen, namelist[i]->d_name);

        if(stat(filedir, &xstat) < 0) {
            printf("**** %s", filedir);
            std_err();
        }
        if(S_ISDIR(xstat.st_mode)) {
            if(recursive_dir(filedir) < 0) goto quit;
        } else {
            add_files(filedir + 2, xstat.st_size, NULL);
        }
        free(namelist[i]);
    }
    ret = 0;

quit:
    for(; i < n; i++) free(namelist[i]);
    free(namelist);
#endif
    filedir[plen - 1] = 0;
    return(ret);
}



u32 crypt_it(FILE *fd, u8 *fname, u32 offset, int wanted_size, int encrypt) {
    static int  buffsz  = 0;
    static u8   *buff   = NULL;

    u32     size;

    fname = create_dir(fname);
    if(check_overwrite(fname) < 0) exit(1);

    fseek(fd, 0, SEEK_END);
    size = ftell(fd);
    if(offset > size) exit(1);

    if(wanted_size < 0) {
        if(fseek(fd, offset, SEEK_SET)) std_err();
        size -= offset;
    } else {
        fseek(fd, 0, SEEK_SET);
    }

    myalloc(&buff, size, &buffsz);
    myfr(fd, buff, size);

    if(wanted_size < 0) {
        blowfish(buff, size, encrypt);
    } else {
        blowfish(buff + offset, wanted_size, encrypt);
    }

    dumpa(fname, buff, size);
    return(size);
}



u8 *string2key(u8 *data) {
    int     i,
            n;
    u8      *ret;

    ret = strdup(data);
    for(i = 0; *data; i++) {
        while(*data && ((*data <= ' ') || (*data == '\\') || (*data == 'x'))) data++;
        if(sscanf(data, "%02x", &n) != 1) break;
        ret[i] = n;
        data += 2;
    }
    ret[i] = 0; // key must be NULL delimited
    return(ret);
}



u32 ttarch_get32(FILE *fd, u8 **data) {
    u32     ret;

    if(fd) {
        ret = fgetxx(fd, 4);
    } else {
        ret = getxx(*data, 4);
        *data += 4;
    }
    return(ret);
}



u8 *ttarch_getname(FILE *fd, u8 **data) {
    static int  buffsz  = 0;
    static u8   *buff   = NULL;
    int     namesz;

    namesz = ttarch_get32(fd, data);
    myalloc(&buff, namesz + 1, &buffsz);
    if(fd) {
        myfr(fd, buff, namesz);
    } else {
        memcpy(buff, *data, namesz);
        *data += namesz;
    }
    buff[namesz] = 0;
    return(buff);
}



void ttarch_extract(FILE *fd, u8 *input_fname) {
    static int  buffsz  = 0;
    static u8   *buff   = NULL;

    u32     info_mode   = 0,
            files_mode  = 0,
            type3       = 0,
            info_zsize  = 0,
            info_size   = 0,
            data_size   = 0,
            dummy,
            folders,
            files,
            offset,
            size,
            zero;
    int     i;
    u8      *info_table = NULL,
            *t          = NULL,
            *name,
            *p;

    p = strrchr(input_fname, '.');
    if(p && stricmp(p, ".ttarch")) {    // simple decryption of a single file... just for fun
        fseek(fd, 0, SEEK_END);
        size = ftell(fd);
        fseek(fd, 0, SEEK_SET);

        myalloc(&buff, size, &buffsz);
        myfr(fd, buff, size);

        name = strdup(input_fname);
        printf("- decrypt the single input file as %s\n", name);
        if(list_only) {
            // do nothing, it's only a listing
        } else {
            ttarch_dumpa(name, buff, size, 0);
        }
        return;
    }

    if(old_mode) {  // boring old mode
        folders = ttarch_get32(fd, NULL);
        for(i = 0; i < folders; i++) {
            name = ttarch_getname(fd, NULL);
        }
        files = ttarch_get32(fd, NULL);
        for(i = 0; i < files; i++) {
            name   = ttarch_getname(fd, NULL);
            zero   = ttarch_get32(fd, NULL);
            offset = ttarch_get32(fd, NULL);
            size   = ttarch_get32(fd, NULL);
        }
        info_size = fgetxx(fd, 4);
        fseek(fd, 0, SEEK_SET);
        goto goto_extract;
    }

    version   = fgetxx(fd, 4);
    if((version < 1) || (version > 9)) {
        printf("\nError: version %d is not supported yet\n", version);
        exit(1);
    }
    printf("- version    %d\n", version);

    info_mode = fgetxx(fd, 4);
    if(/*(info_mode < 0) ||*/ (info_mode > 1)) {
        printf("\nError: info_mode %d is not supported yet\n", info_mode);
        exit(1);
    }
    printf("- info_mode  %d\n", info_mode);

    type3     = fgetxx(fd, 4);
    printf("- type3      %d\n", type3);

    if(version >= 3) {
        files_mode = fgetxx(fd, 4);
    }
    if(/*(files_mode < 0)  || */ (files_mode > 2)) {
        printf("\nError: files_mode %d is not supported yet\n", files_mode);
        exit(1);
    }
    printf("- files_mode %d\n", files_mode);

    if(version >= 3) {
        ttarch_tot_idx = fgetxx(fd, 4);
        if(ttarch_tot_idx) {
            printf("- found %d compressed chunks\n", ttarch_tot_idx);
            ttarch_chunks = calloc(ttarch_tot_idx, 4);
            if(!ttarch_chunks) std_err();
            for(i = 0; i < ttarch_tot_idx; i++) {
                ttarch_chunks[i] = fgetxx(fd, 4);
            }
        }

        data_size = fgetxx(fd, 4);  // the size of the field where are stored all the files contents
        if(version >= 4) {
            dummy = fgetxx(fd, 4);
            dummy = fgetxx(fd, 4);
            if(version >= 7) {
                dummy = fgetxx(fd, 4);
                dummy = fgetxx(fd, 4);
                chunksz = fgetxx(fd, 4);
                chunksz *= 1024;
                printf("- set chunk size to 0x%x bytes\n", chunksz);
                if(version >= 8) dummy = fgetxx(fd, 1);
                if(version >= 9) dummy = fgetxx(fd, 4);
            }
        }
    }

    info_size  = fgetxx(fd, 4);
    if((version >= 7) && files_mode >= 2) {
        info_zsize = fgetxx(fd, 4);
    }
goto_extract:
    printf("- info_table has a size of %d bytes\n", info_size);
    info_table = malloc(info_size);
    if(!info_table) std_err();

    if((version >= 7) && (files_mode >= 2)) {
        t = malloc(info_zsize);
        myfr(fd, t, info_zsize);
        unzip(t, info_zsize, info_table, info_size);
        free(t);
    } else {
        myfr(fd, info_table, info_size);
    }

    if(info_mode >= 1) {
        printf("- decrypt info_table\n");
        blowfish(info_table, info_size, 0);
    }

    ttarch_baseoff = ftell(fd);
    printf("- set files base offset 0x%08x\n", ttarch_baseoff);

    if(files_mode >= 2) {   // not verified
        if(info_mode > 0) ttarch_chunks_b = 1;
    }
    printf("- filesystem compression: %s\n", ttarch_tot_idx  ? "on" : "off");
    printf("- filesystem encryption:  %s\n", ttarch_chunks_b ? "on" : "off");

    t = info_table;
    if(dump_table) {
        dumpa(dump_table, info_table, info_size);
    }

    printf("- retrieve folders:\n");
    folders = ttarch_get32(NULL, &t);
    for(i = 0; i < folders; i++) {
        name = ttarch_getname(NULL, &t);
        printf("  %s\n", name);
    }

    printf("\n"
        "  offset   filesize   filename\n"
        "------------------------------\n");

    files = ttarch_get32(NULL, &t);
    for(i = 0; i < files; i++) {
        name   = ttarch_getname(NULL, &t);
        zero   = ttarch_get32(NULL, &t);
        offset = ttarch_get32(NULL, &t);
        size   = ttarch_get32(NULL, &t);

        if(zero) {  // this value is just ignored in the current versions of ttarch, maybe it's for a future usage?
            printf("\nError: this file has an unknown ZERO value, contact me\n");
            exit(1);
        }

        if(filter_files && (check_wildcard(name, filter_files) < 0)) continue;
        printf("  %08x %-10u %s\n", ttarch_tot_idx ? offset : (ttarch_baseoff + offset), size, name);
        if(list_only) {
            extracted_files++;
            continue;
        }

        myalloc(&buff, size, &buffsz);

        ttarch_fseek(fd, offset, SEEK_SET);
        ttarch_fread(buff, size, fd);

        ttarch_dumpa(name, buff, size, 0);
    }
}



int ttarch_meta_crypt(u8 *data, int datalen, int encrypt) {
#define SET_BLOCKS(X,Y,Z) { \
            blocks_size     = X; \
            blocks_crypt    = Y; \
            blocks_clean    = Z; \
        }

    u32     file_type       = 0;
    int     i,
            blocks_size     = 0,
            blocks_crypt    = 0,
            blocks_clean    = 0,
            blocks,
            rem_blocks,
            meta            = 1;
    u8      *p,
            *l;

    if(datalen < 4) return(file_type);

    p = data;
    l = data + datalen;
    file_type = ttarch_get32(NULL, &p);

    switch(file_type) {
        case 0x4D424553: SET_BLOCKS(0x40,  0x40, 0x64)  break;
        case 0x4D42494E:                                break;
        case 0xFB4A1764: SET_BLOCKS(0x80,  0x20, 0x50)  break;
        case 0xEB794091: SET_BLOCKS(0x80,  0x20, 0x50)  break;
        case 0x64AFDEFB: SET_BLOCKS(0x80,  0x20, 0x50)  break;
        case 0x64AFDEAA: SET_BLOCKS(0x100, 0x8,  0x18)  break;
        case 0x4D545245:                                break;
        default:         meta = 0;                      break;  // is not a meta stream file
    }

    if(blocks_size) {   // meta, just the same result
        blocks     = (datalen - 4) / blocks_size;
        rem_blocks = (datalen - 4) % blocks_size;

        for(i = 0; i < blocks; i++) {
            if(!(i % blocks_crypt)) {
                blowfish(p, blocks_size, encrypt);
            } else if(!(i % blocks_clean) && (i > 0)) {
                // skip this block
            } else {
                xor(p, blocks_size, 0xff);
            }
            p += blocks_size;
        }
    }

    return(meta);
}



int ttarch_fseek(FILE *stream, int offset, int origin) {
    int     i,
            idx,
            off = 0;

    if(!ttarch_tot_idx) return(fseek(stream, ttarch_baseoff + offset, origin));

    idx = offset / chunksz;
    if(idx > ttarch_tot_idx) return(-1);
    for(i = 0; i < idx; i++) {
        off += ttarch_chunks[i];
    }
    ttarch_rem    = offset % chunksz;
    ttarch_offset = offset;
    return(fseek(stream, ttarch_baseoff + off, origin));
}



int ttarch_fread(void *ptr, int size, FILE *stream) {
    static  u8  *in     = NULL,
                *out    = NULL;
    int     i,
            idx,
            len,
            currsz;

    if(!ttarch_tot_idx) {
        myfr(stream, ptr, size);
        if(ttarch_chunks_b) blowfish(ptr, size, 0);
        return(size);
    }

    if(!in || !out) {
        in  = malloc(chunksz);
        out = malloc(chunksz);
        if(!in || !out) std_err();
    }

    ttarch_fseek(stream, ttarch_offset, SEEK_SET);

    currsz = 0;
    for(idx = ttarch_offset / chunksz; idx < ttarch_tot_idx; idx++) {
        if(currsz >= size) break;
        myfr(stream, in, ttarch_chunks[idx]);
        if(ttarch_chunks_b) blowfish(in, ttarch_chunks[idx], 0);
        if(ttarch_chunks[idx] == chunksz) {
            len = chunksz;
            memcpy(out, in, len);
        } else {
            len = unzip(in, ttarch_chunks[idx], out, chunksz);
        }
        if(ttarch_rem) {
            if(ttarch_rem > len) {
                ttarch_rem -= len;
                continue;
            }
            len -= ttarch_rem;
            //memmove(out, out + ttarch_rem, len);  // for unknown reasons sometimes memmove makes a bad job
            for(i = 0; i < len; i++) {
                out[i] = out[ttarch_rem + i];
            }
            ttarch_rem = 0;
        }
        currsz += len;
        if(currsz > size) {
            len -= (currsz - size);
            currsz = size;
        }
        memcpy(ptr, out, len);
        ptr += len;
        if(currsz >= size) break;
    }
    ttarch_offset += currsz;
    ttarch_rem    = ttarch_offset % chunksz;
    return(currsz);
}



void xor(u8 *data, int datalen, int xornum) {
    int     i;

    for(i = 0; i < datalen; i++) {
        data[i] ^= xornum;
    }
}



void blowfish(u8 *data, int datalen, int encrypt) {
    static  blf_ctx *blowfish_ctx = NULL;

    if(!blowfish_ctx) { // init
        blowfish_ctx = malloc(sizeof(blf_ctx));
        if(!blowfish_ctx) std_err();
        if(version >= 7) {
            blf_key7(blowfish_ctx, mykey, strlen(mykey));
        } else {
            blf_key(blowfish_ctx, mykey, strlen(mykey));
        }
    }

    if(encrypt) {
        if(version >= 7) {
            blf_enc7(blowfish_ctx, (void *)data, datalen / 8);
        } else {
            blf_enc(blowfish_ctx, (void *)data, datalen / 8);
        }
    } else {
        if(version >= 7) {
            blf_dec7(blowfish_ctx, (void *)data, datalen / 8);
        } else {
            blf_dec(blowfish_ctx, (void *)data, datalen / 8);
        }
    }
}



u8 *scan_search(u8 *buff, int *buffsz, u8 *find, int findsz) {
#define MAX_SCAN_SIZE   4096
#define MAX_SCAN_CRYPT  8   // blowfish size
    int     i,
            tmpsz;
    u8      tmp[MAX_SCAN_CRYPT];

    if(findsz > *buffsz) return(NULL);
    if(findsz > MAX_SCAN_CRYPT) {
        printf("\nError: you need to modify MAX_SCAN_CRYPT in scan_search\n");
        exit(1);
    }

    tmpsz = MAX_SCAN_SIZE;   // it's not needed to scan the whole file
    if(*buffsz < tmpsz) tmpsz = *buffsz;
    tmpsz -= findsz;

    for(i = 0; i <= tmpsz; i++) {
        if(!memcmp(buff + i, find, findsz)) {
            buff    += i;
            *buffsz -= i;
            return(buff);
        }
    }

    for(i = 0; i <= tmpsz; i++) {
        memcpy(tmp, buff + i, MAX_SCAN_CRYPT);
        blowfish(tmp, MAX_SCAN_CRYPT, 0);
        if(!memcmp(tmp, find, findsz)) {
            buff    += i;
            *buffsz -= i;
            blowfish(buff, 0x800, 0);
            return(buff);
        }
    }
    return(NULL);
}



u8 *ttarch_meta_dump(u8 *ext, u8 *data, int *datalen) { // completely experimental and horrible, ignores the classes
    u32     file_type;
    int     size;
    u8      *p,
            *ret;

    p    = data;
    size = *datalen;
    if(!meta_extract) return(data);
    if(size < 4) return(data);

    file_type = ttarch_get32(NULL, &p);
    size -= 4;

    ret = NULL;
    if(!stricmp(ext, ".font") || !stricmp(ext, ".d3dtx")) {
        ret = scan_search(p, &size, "DDS ", 4);
        if(ret) strcpy(ext, ".dds");
    } else if(!stricmp(ext, ".aud")) {
        ret = scan_search(p, &size, "OggS", 4);
        if(ret) strcpy(ext, ".ogg");
    } else if(!stricmp(ext, ".lenc")) {
        blowfish(data, size, 0);
        strcpy(ext, ".lua");
    }
    if(ret && (size >= 0)) {
        *datalen = size;
    } else {
        ret = data;
    }
    return(ret);
}



int ttarch_dumpa(u8 *fname, u8 *data, int size, int already_decrypted) {
    int     meta;
    u8      *ext,
            *p;

    if(!already_decrypted) {
        meta = ttarch_meta_crypt(data, size, 0);
    }
    ext = strrchr(fname, '.');
    if(ext) {
        if(meta_extract) {
            data = ttarch_meta_dump(ext, data, &size);
        }
    }

    // the following is a set of filename cleaning instructions to avoid that files or data with special names are not saved
    if(fname) {
        if(fname[1] == ':') fname += 2;
        for(p = fname; *p && (*p != '\n') && (*p != '\r'); p++) {
            if(strchr("?%*:|\"<>", *p)) {    // invalid filename chars not supported by the most used file systems
                *p = '_';
            }
        }
        *p = 0;
        for(p--; (p >= fname) && ((*p == ' ') || (*p == '.')); p--) *p = 0;   // remove final spaces and dots
    }

    //if(filter_files && (check_wildcard(fname, filter_files) < 0)) return(0);
    //printf("  %08x %-10u %s\n", offset, size, fname);

    fname = create_dir(fname);
    if(check_overwrite(fname) < 0) return(0);

    dumpa(fname, data, size);

    extracted_files++;
    return(0);
}



int unzip(u8 *in, int insz, u8 *out, int outsz) {
    static z_stream *z_zlib     = NULL;
    static z_stream *z_deflate  = NULL;
    z_stream *z;

#define UNZIP_INIT(X,Y) \
    if(!z_##X) { \
        z_##X = malloc(sizeof(z_stream)); \
        if(!z_##X) std_err(); \
        z_##X->zalloc = (alloc_func)0; \
        z_##X->zfree  = (free_func)0; \
        z_##X->opaque = (voidpf)0; \
        if(inflateInit2(z_##X, Y)) { \
            printf("\nError: "#X" initialization error\n"); \
            exit(1); \
        } \
    }
#define UNZIP_END(X) \
        if(z_##X) { \
            inflateEnd(z_##X); \
            free(z_##X); \
            z_##X = NULL; \
        }

    if(!insz || !outsz) return(0);
    if(!in && !out) {
        UNZIP_END(zlib)
        UNZIP_END(deflate)
        return(-1);
    }

    UNZIP_INIT(zlib,    15)
    UNZIP_INIT(deflate, -15)
    z = z_zlib;
redo:
    inflateReset(z);

    z->next_in   = in;
    z->avail_in  = insz;
    z->next_out  = out;
    z->avail_out = outsz;
    if(inflate(z, Z_FINISH) != Z_STREAM_END) {
        if(z == z_zlib) {
            z = z_deflate;
            goto redo;
        }
        printf("\nError: the compressed zlib/deflate input is wrong or incomplete\n");
        exit(1);
    }
    return(z->total_out);
}



int check_wildcard(u8 *fname, u8 *wildcard) {
    u8      *f,
            *w,
            *a;

    if(!wildcard) return(0);
    f = fname;
    w = wildcard;
    a = NULL;
    while(*f || *w) {
        if(!*w && !a) return(-1);
        if(*w == '?') {
            if(!*f) break;
            w++;
            f++;
        } else if(*w == '*') {
            w++;
            a = w;
        } else {
            if(!*f) break;
            if(tolower(*f) != tolower(*w)) {
                if(!a) return(-1);
                f++;
                w = a;
            } else {
                f++;
                w++;
            }
        }
    }
    if(*f || *w) return(-1);
    return(0);
}



u8 *create_dir(u8 *fname) {
    u8      *p,
            *l;

    p = strchr(fname, ':'); // unused
    if(p) {
        *p = '_';
        fname = p + 1;
    }
    for(p = fname; *p && strchr("\\/. \t:", *p); p++) *p = '_';
    fname = p;

    for(p = fname; ; p = l + 1) {
        for(l = p; *l && (*l != '\\') && (*l != '/'); l++);
        if(!*l) break;
        *l = 0;

        if(!strcmp(p, "..")) {
            p[0] = '_';
            p[1] = '_';
        }

        make_dir(fname);
        *l = PATHSLASH;
    }
    return(fname);
}



int check_overwrite(u8 *fname) {
    FILE    *fd;
    u8      ans[16];

    if(force_overwrite) return(0);
    if(!fname) return(0);
    fd = fopen(fname, "rb");
    if(!fd) return(0);
    fclose(fd);
    printf("- the file \"%s\" already exists\n  do you want to overwrite it (y/N/all)? ", fname);
    fgets(ans, sizeof(ans), stdin);
    if(tolower(ans[0]) == 'y') return(0);
    if(tolower(ans[0]) == 'a') {
        force_overwrite = 1;
        return(0);
    }
    return(-1);
}



void myalloc(u8 **data, int wantsize, int *currsize) {
    if(!wantsize) return;
    if(wantsize <= *currsize) {
        if(*currsize > 0) return;
    }
    *data = realloc(*data, wantsize);
    if(!*data) std_err();
    *currsize = wantsize;
}



int getxx(u8 *tmp, int bytes) {
    u32     num;
    int     i;

    num = 0;
    for(i = 0; i < bytes; i++) {
        num |= (tmp[i] << (i << 3));
    }
    return(num);
}



int fgetxx(FILE *fd, int bytes) {
    int     ret;
    u8      tmp[bytes];

    myfr(fd, tmp, bytes);
    ret = getxx(tmp, bytes);
    if(verbose) printf("  %08x: %08x\n", ((int)ftell(fd)) - bytes, ret);
    return(ret);
}



int myfr(FILE *fd, u8 *data, int size) {
    int     len;

    len = fread(data, 1, size, fd);
    if(len != size) {
        printf("\nError: incomplete input file, can't read %u bytes\n", size - len);
        exit(1);
    }
    return(len);
}



int putxx(u8 *data, u32 num, int bytes) {
    int     i;

    for(i = 0; i < bytes; i++) {
        data[i] = num >> (i << 3);
    }
    return(bytes);
}



int fputxx(FILE *fd, u32 num, int bytes) {
    u8      tmp[bytes];

    putxx(tmp, num, bytes);
    myfw(fd, tmp, bytes);
    return(bytes);
}



void dumpa(u8 *fname, u8 *data, int size) {
    FILE    *fdo;

    fdo = fopen(fname, "wb");
    if(!fdo) std_err();
    myfw(fdo, data, size);
    fclose(fdo);
}



int myfw(FILE *fd, u8 *data, int size) {
    int     len;

    len = fwrite(data, 1, size, fd);
    if(len != size) {
        printf("\nError: impossible to write %u bytes\n", size - len);
        exit(1);
    }
    return(len);
}



int get_num(u8 *str) {
    int     offset;

    if(!strncmp(str, "0x", 2) || !strncmp(str, "0X", 2)) {
        sscanf(str + 2, "%x", &offset);
    } else {
        sscanf(str, "%u", &offset);
    }
    return(offset);
}



void std_err(void) {
    perror("Error");
    exit(1);
}


