/*
    Copyright 2004,2005,2006,2007,2008 Luigi Auriemma

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    http://www.gnu.org/licenses/gpl-2.0.txt
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <time.h>
#include "ubi_gschat.h"

#ifdef WIN32
    #include <winsock.h>
    #include "winerr.h"

    #define close   closesocket
    #define sleep   Sleep
    #define ONESEC  1000
#else
    #include <unistd.h>
    #include <sys/socket.h>
    #include <sys/types.h>
    #include <arpa/inet.h>
    #include <netinet/in.h>
    #include <netdb.h>
    #include <pthread.h>

    #define ONESEC  1
#endif

#ifdef WIN32
    #define quick_thread(NAME, ARG) DWORD WINAPI NAME(ARG)
    #define thread_id   DWORD
#else
    #define quick_thread(NAME, ARG) void *NAME(ARG)
    #define thread_id   pthread_t
#endif

thread_id quick_threadx(void *func, void *data) {
    thread_id       tid;
#ifdef WIN32
    if(!CreateThread(NULL, 0, func, data, 0, &tid)) return(0);
#else
    pthread_attr_t  attr;
    if(pthread_attr_init(&attr)) return(0);
    pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_DETACHED);
    pthread_attr_setstacksize(&attr, 1<<18); //PTHREAD_STACK_MIN);
    if(pthread_create(&tid, &attr, func, data)) return(0);
#endif
    return(tid);
}

typedef uint8_t     u8;
typedef uint16_t    u16;
typedef uint32_t    u32;



#define VER             "0.2.2"
#define BUFFSZ          0xffff
#define PORT            6668
#define LOGFILE         "ubichat_irc.log"
#define LOGIT(X,Y,Z)    if(logfile) { \
                            fprintf(fdlog, "\n"X"\n%.*s", Y, Z); \
                            fflush(fdlog); \
                        }



int bind_job(u32 iface, u16 port);
void calculate_peer(void);
quick_thread(client, int sock);
u32 resolv(char *host);
void std_err(void);



struct  sockaddr_in peer;
FILE    *fdlog       = NULL;
int     logfile      = 0;
u16     ubiport      = PORT;
u8      ubihost[128] = "",
        ubihostnum   = 0;



int main(int argc, char *argv[]) {
    struct  sockaddr_in peer1;
    struct  tm  *tmx;
    time_t  datex;
    u32     lhost;
    int     sdl,    // listening
            sda,    // accept
            i,
            psz;
    u16     lport   = PORT;
    u8      *p;

#ifdef WIN32
    WSADATA    wsadata;
    WSAStartup(MAKEWORD(1,0), &wsadata);
#endif

    setbuf(stdout, NULL);

    printf("\n"
        "UBI.COM gschat IRC proxy "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n"
        "HOW TO USE\n"
        "==========\n"
        " 1) launch this tool (done)\n"
        " 2) connect the IRC client to this host, 127.0.0.1 by default\n"
        " 3) if you have problems retrieving the channels to join take a look here:\n"
        "    http://aluigi.org/papers/ubichan.txt\n"
        "\n"
        "Optional command-line arguments:\n"
        "-l         enable log file ("LOGFILE") in append mode\n"
        "-o         as above but to stdout\n"
        "-b IP[:P]  IP or hostname of the local interface to bind, default is localhost\n"
        "           only. Use -b 0 to bind any interface of your computer or -b 0:6667\n"
        "           to bind the port 6667 instead of the default %hu\n"
        "-s H[:P]   specify the Ubi.com IRC server H and its port P to which you want\n"
        "           to connect, default is %s:%hu\n"
        "\n", lport, ubihost, ubiport);

    lhost = inet_addr("127.0.0.1");

    for(i = 1; i < argc; i++) {
        if(((argv[i][0] != '-') && (argv[i][0] != '/')) || (strlen(argv[i]) != 2)) {
            printf("\nError: wrong argument (%s)\n", argv[i]);
            exit(1);
        }
        switch(argv[i][1]) {
            case '-':
            case 'h':
            case 'v':
            case '?': return(0); break;
            case 'l': {
                logfile = 1;
                } break;
            case 'o': {
                logfile = -1;
                } break;
            case 'b': {
                i++;
                p = strchr(argv[i], ':');
                if(p) {
                    *p = 0;
                    lport = atoi(p + 1);
                }
                lhost = resolv(argv[i]);
                printf("- bind local interface:    %s : %hu\n",
                    (lhost != INADDR_ANY) ? inet_ntoa(*(struct in_addr *)&lhost) : "ANY",
                    lport);
                } break;
            case 's': {
                sprintf(ubihost, "%.*s", sizeof(ubihost), argv[++i]);
                p = strchr(ubihost, ':');
                if(p) {
                    *p = 0;
                    ubiport = atoi(p + 1);
                }
                } break;
            default: {
                printf("\nError: wrong command-line argument (%s)\n", argv[i]);
                exit(1);
                } break;
        }
    }

    calculate_peer();
    printf("- bind TCP port            %hu\n", lport);

    if(logfile < 0) {
        fdlog = stdout;
    } else if(logfile) {
        printf("- Open log file            "LOGFILE" ... ");
        fdlog = fopen(LOGFILE, "ab");
        if(!fdlog) std_err();
        printf("ok\n");

        time(&datex);
        tmx = localtime(&datex);
        fprintf(fdlog,
            "----------------------------------\n"
            "Log file appended at:\n"
            "  %02d/%02d/%02d   (dd/mm/yy)\n"
            "  %02d:%02d:%02d\n"
            "----------------------------------\n"
            "\n",
            tmx->tm_mday, tmx->tm_mon + 1, tmx->tm_year % 100,
            tmx->tm_hour, tmx->tm_min, tmx->tm_sec);
        fflush(fdlog);
    }

    sdl = bind_job(lhost, lport);

    if(lhost != INADDR_ANY) {
        printf("\n"
            "- Connect your IRC clients to %s on port %hu\n",
            inet_ntoa(*(struct in_addr *)&lhost), lport);
    } else {
        printf("\n"
            "- Now the IRC clients can connect to this host on port %hu\n",
            lport);
    }
    printf("  This server accepts multiple clients simultaneously:\n\n");

    for(;;) {
        psz = sizeof(struct sockaddr_in);
        sda = accept(sdl, (struct sockaddr *)&peer1, &psz);
        if(sda < 0) {
            printf("- accept() failed, continue within one second\n");
            close(sdl);
            sleep(ONESEC);
            sdl = bind_job(INADDR_ANY, PORT);
            continue;
        }

        printf("  connection from   %s : %hu\n",
            inet_ntoa(peer1.sin_addr),
            ntohs(peer1.sin_port));

        if(logfile > 0) {
            fprintf(fdlog, "\n"
                "------------------------------------------\n"
                "Connection from   %s : %hu\n"
                "------------------------------------------\n",
                inet_ntoa(peer1.sin_addr),
                ntohs(peer1.sin_port));
        }

        if(!quick_threadx(client, (void *)sda)) close(sda);
    }

    close(sdl);
    return(0);
}



int bind_job(u32 iface, u16 port) {
    struct  sockaddr_in peerx;
    int     sdl,
            on = 1;

    peerx.sin_addr.s_addr = iface;
    peerx.sin_port        = htons(port);
    peerx.sin_family      = AF_INET;

    sdl = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
    if(sdl < 0) std_err();
    if(setsockopt(sdl, SOL_SOCKET, SO_REUSEADDR, (char *)&on, sizeof(on))
      < 0) std_err();
    if(bind(sdl, (struct sockaddr *)&peerx, sizeof(struct sockaddr_in))
      < 0) std_err();
    listen(sdl, SOMAXCONN);
    return(sdl);
}



void calculate_peer(void) {
    if(!ubihost[0] || !strncmp(ubihost, "gsxirc", 6)) {
        if(ubihostnum == 0) {
            ubihostnum = 2;
        } else if(ubihostnum == 2) {
            ubihostnum = 1;
        } else if(ubihostnum == 1) {
            ubihostnum = 3;
        } else {
            ubihostnum++;
        }
        sprintf(ubihost, "gsxirc%02d.gs.mdc.ubisoft.com", ubihostnum);
    }
    printf("- resolv hostname:         %s ... ", ubihost);
    peer.sin_addr.s_addr = resolv(ubihost);    /* connect to server */
    printf("ok\n");

    if(peer.sin_addr.s_addr == inet_addr("127.0.0.1")) {
        printf("\n"
            "Error: the server's IP is a local IP (%s), check your hosts file (probably you\n"
            "       have used ubichat_proxy and forgot to remove the server's entry):\n"
            "\n"
            " Windows 95/98/Me          c:\\windows\\hosts\n"
            " Windows NT/2000           c:\\winnt\\system32\\drivers\\etc\\hosts\n"
            " Windows XP                c:\\windows\\system32\\drivers\\etc\\hosts\n"
            " Unix/Linux                /etc/hosts\n"
            "\n"
            "Press ENTER to exit\n",
            inet_ntoa(peer.sin_addr));
        fgetc(stdin);
        exit(1);
    }

    peer.sin_port        = htons(ubiport);
    peer.sin_family      = AF_INET;

    printf("- server:                  %s : %hu\n", inet_ntoa(peer.sin_addr), ntohs(peer.sin_port));
}



quick_thread(client, int sock) {
    fd_set  rset;
    int     sd      = -1,
            selsock,
            t,
            slen,
            rlen;
    u16     rsize;
    u8      *rbuff  = NULL,
            *sbuff  = NULL,
            *sbuffp;

    sd = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
    if(sd < 0) std_err();
redo_connection:
    if(connect(sd, (struct sockaddr *)&peer, sizeof(peer)) < 0) {
        printf("- the server %s:%hu seems down, I try the next\n", ubihost, ubiport);
        calculate_peer();
        goto redo_connection;
    }

    slen  = 0;
    rlen  = 0;
    rsize = 0;
    sbuff = malloc(2 + BUFFSZ);
    rbuff = malloc(BUFFSZ);
    if(!sbuff || !rbuff) std_err();
    sbuffp = sbuff + 2; // required

    selsock = ((sock > sd) ? sock : sd) + 1;

    for(;;) {
        FD_ZERO(&rset);
        FD_SET(sock, &rset);
        FD_SET(sd, &rset);
        if(select(selsock, &rset, NULL, NULL, NULL)
          < 0) std_err();

        if(FD_ISSET(sd, &rset)) {
            if(!rsize) {
                if(recv(sd, rbuff + rlen, 1, 0) <= 0) break;
                rlen++;
                if(rlen == 2) {
                    rsize = (rbuff[0] << 8) | rbuff[1];
                    rlen  = 0;
                }
            } else {
                t = recv(sd, rbuff + rlen, rsize - rlen, 0);
                if(t <= 0) break;
                rlen += t;
                if(rlen == rsize) {
                    rsize = ubi_gschat_dec(rbuff, rsize);
                    LOGIT("### SERVER", rsize, rbuff)
                    if(send(sock, rbuff, rsize, 0) != rsize) break;
                    rlen  = 0;
                    rsize = 0;
                }
            }
        } else if(FD_ISSET(sock, &rset)) {
            if(recv(sock, sbuffp + slen, 1, 0) <= 0) break;
            slen++;

            if((slen >= (BUFFSZ - 2)) || (sbuffp[slen - 1] == '\n')) {
                LOGIT("### CLIENT", slen, sbuffp)

                t = ubi_gschat_enc(sbuffp, slen);   // t is the new padded size
                sbuffp[t]     = slen;               // slen is the original size
                sbuffp[t + 1] = slen >> 8;
                slen = t + 2;                       // now slen is the total pad + 2 size

                sbuff[0] = slen >> 8;               // the server wants all the data in one time
                sbuff[1] = slen;
                slen += 2;
                if(send(sd, sbuff, slen, 0) != slen) break;
                slen = 0;
            }
        }
    }

    if(sbuff) free(sbuff);
    if(rbuff) free(rbuff);
    if(sd > 0) close(sd);
    close(sock);

    printf("  disconnected\n");
    if(logfile > 0) fprintf(fdlog, "### DISCONNECTED ###\n");
    return(0);
}



u32 resolv(char *host) {
    struct  hostent *hp;
    u32     host_ip;

    host_ip = inet_addr(host);
    if(host_ip == INADDR_NONE) {
        hp = gethostbyname(host);
        if(!hp) {
            printf("\nError: Unable to resolv hostname (%s)\n", host);
            exit(1);
        } else host_ip = *(u32 *)hp->h_addr;
    }
    return(host_ip);
}



#ifndef WIN32
    void std_err(void) {
        perror("\nError");
        exit(1);
    }
#endif


