/*
    Copyright 2004,2005,2006,2007,2008 Luigi Auriemma

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    http://www.gnu.org/licenses/gpl-2.0.txt
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <pcap.h>
#include "show_dump.h"
#include "ip.h"
#include "ubi_algo.h"
#include "ether_hdrlen.h"

#ifdef WIN32
    #include <winsock.h>
#else
    #include <unistd.h>
    #include <sys/socket.h>
    #include <sys/types.h>
    #include <sys/param.h>
    #include <arpa/inet.h>
    #include <netinet/in.h>
    #include <netdb.h>
    #include <sys/stat.h>
#endif

typedef uint8_t     u8;
typedef uint16_t    u16;
typedef uint32_t    u32;



#define VER         "0.2"
#define UBIFILE     "ubisniff.log"
#define USERINFO    "\x62\x00\x00\x00\x01\x13"
#define SNIFFPORT   40000
#define MAXHOSTS    200
#define MAXBUFFSZ   0x7fff  // 0x3fff is enough



void pcap_error(pcap_if_t *alldevs, char *err);
pcap_t *pcap_get_device(void);
void pcaploop(u8 *ignore, const struct pcap_pkthdr *hdr, const u8 *pkt);
void ubi_do_it(u8 flags, u32 saddr, u16 sport, u32 daddr, u16 dport, u8 *data, int datalen);
u32 resolv(char *host);



FILE    *fdlog;
int     if_offset;



int main(int argc, char *argv[]) {
    pcap_t  *fp;

#ifdef WIN32
    WSADATA wsadata;
    WSAStartup(MAKEWORD(1,0), &wsadata);
#endif

    setbuf(stdout, NULL);

    fputs("\n"
        "Ubi.com real-time packets decoder "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", stderr);

    printf("\n"
        "All the ingoing and outgoing TCP data from port %hu until port %hu will be\n"
        "captured and decoded in real-time.\n"
        "The output is in hex and byte format and is displayed to screen and appended to\n"
        "the log file %s too\n"
        "Due to the port based identification of the hosts is better if you run this\n"
        "tool without other network related programs running (except GS4 naturally)\n"
        "\n", SNIFFPORT, SNIFFPORT + MAXHOSTS, UBIFILE);

    fp = pcap_get_device();

    fdlog = fopen(UBIFILE, "ab");
    if(!fdlog) {
        printf("\nError: impossible to create or open the log file "UBIFILE"\n");
        exit(1);
    }
    fputs("\n\n===\nNEW DUMP\n===\n\n", fdlog);
    fflush(fdlog);

    printf("- ready:\n\n");
    pcap_loop(fp, 0, pcaploop, NULL);

    return(0);
}



void pcap_error(pcap_if_t *alldevs, char *err) {
    fprintf(stderr, "\nError: %s\n", err);
    if(alldevs) pcap_freealldevs(alldevs);
    exit(1);
}



pcap_t *pcap_get_device(void) {
    pcap_if_t   *alldevs,
                *d;
    pcap_t  *fp;
    int     i,
            inum;
    char    ans[64],
            errbuf[PCAP_ERRBUF_SIZE];

    if(pcap_findalldevs(&alldevs, errbuf) < 0) pcap_error(alldevs, errbuf);

    fprintf(stderr, "- interfaces:\n\n");
    for(i = 0, d = alldevs; d; d = d->next) {
        fprintf(stderr, "%-2d %s\n", ++i, d->name);
        if(d->description) fprintf(stderr, "   %s\n", d->description);
    }

    if(i > 1) {
        fprintf(stderr, "\n- enter the interface number (1 - %d): ", i);
        fgets(ans, sizeof(ans), stdin);
        sscanf(ans, "%d", &inum);
        if(inum < 1 || inum > i) pcap_error(alldevs, "interface number out of range");
    } else if(!i) {
        pcap_error(alldevs,
            "No interfaces found or you don't have the needed permissions.\n"
            "       Make sure Pcap/WinPcap is installed and you are root or admin");
    } else {
        inum = 1;
    }

    for(inum--, d = alldevs, i = 0; i < inum; i++, d = d->next);
    fprintf(stderr, "- adapter to use: %s\n", d->name);

    fp = pcap_open_live(d->name, 0xffff, 1, 100, errbuf);
    if(!fp) pcap_error(alldevs, errbuf);

    pcap_freealldevs(alldevs);

    if_offset = pcap_hdrlen(pcap_datalink(fp));
    if(if_offset < 0) pcap_error(alldevs, "pcap interface not supported by this tool");

    return(fp);
}



void pcaploop(u8 *ignore, const struct pcap_pkthdr *hdr, const u8 *pkt) {
    struct  iphdr   *ip;
    struct  tcphdr  *tcp;
    int     iflen,
            iplen,
            tcplen,
            len;
    u8      *data;

    iflen  = ntohs(*(u16 *)(pkt + if_offset - 2));
    ip     = (struct iphdr *)(pkt + if_offset + ether_hdrlen(iflen));
    if((ip->ihl_ver >> 4) != 4) return;
    iplen  = (ip->ihl_ver & 15) << 2;

    if(ip->protocol != IPPROTO_TCP) return;

    tcp    = (struct tcphdr *)((u8 *)ip + iplen);
    tcplen = tcp->doff >> 2;

    data   = (u8 *)((u8 *)tcp + tcplen);
    len    = ntohs(ip->tot_len) - (data - (u8 *)ip);
    ubi_do_it(tcp->flags, ip->saddr, tcp->source, ip->daddr, tcp->dest, data, len);
}



typedef struct {
    //u32     sniffhost;
    int     bufflen;
    int     buffsize;
    int     headlen;
    u8      head[6];
    u8      *buff;
} sniffhost_t;



void ubi_do_it(u8 flags, u32 saddr, u16 sport, u32 daddr, u16 dport, u8 *data, int datalen) {
    int     n,
            x;
    u8      tmp[80],
            *p,
            *l;
    static  sniffhost_t *sniffhost  = NULL;

    if(!sniffhost) {
        sniffhost = calloc(sizeof(sniffhost_t), MAXHOSTS);
        if(!sniffhost) return;
    }

    sport = ntohs(sport);
    dport = ntohs(dport);

    // I filter the ports, not the hosts... gsarouter02.gs.mdc.ubisoft.com
    if((sport >= SNIFFPORT) && (sport < (SNIFFPORT + MAXHOSTS))) {
        x = sport - SNIFFPORT;
    } else if((dport >= SNIFFPORT) && (dport < (SNIFFPORT + MAXHOSTS))) {
        x = dport - SNIFFPORT;
    } else {
        return;
    }

    n = sprintf(tmp, "\n%s:%u --> ", inet_ntoa(*(struct in_addr *)&saddr), sport);
    sprintf(tmp + n, "%s:%u\n", inet_ntoa(*(struct in_addr *)&daddr), dport);
    fputs(tmp, stdout);
    fputs(tmp, fdlog);

    if(flags & TH_FIN) {
        printf("- disconnected\n");
        sniffhost[x].bufflen  = 0;
        sniffhost[x].buffsize = 0;
        sniffhost[x].headlen  = 0;
        return;
    }
    // if(!(flags & TH_PSH)) return; // some packets arrive without PSH???
    if(datalen <= 0) return;

    p = data;
    l = data + datalen;
    while(p < l) {
        if(sniffhost[x].buffsize <= 0) {
            while(sniffhost[x].headlen < 6) {
                if(p >= l) break;
                sniffhost[x].head[sniffhost[x].headlen++] = *p++;
            }
            if(sniffhost[x].headlen == 6) {
                sniffhost[x].buffsize  = ((sniffhost[x].head[0] << 16) | (sniffhost[x].head[1] << 8) | sniffhost[x].head[2]) - 6;
                sniffhost[x].bufflen   = 0;
                sniffhost[x].headlen   = 0;
                if(sniffhost[x].buffsize > MAXBUFFSZ) sniffhost[x].buffsize = -1;   // probably an error in the packet
                if(sniffhost[x].buffsize < 0) {
                    printf("- possible invalid packet, hexdump:\n");
                    show_dump(data, datalen, stdout);
                    break;
                }
            }
        }
        if(sniffhost[x].buffsize <= 0) continue;
        if(!sniffhost[x].buff) {
            sniffhost[x].buff = malloc(MAXBUFFSZ);
            if(!sniffhost[x].buff) return;
        }
        while(sniffhost[x].bufflen < sniffhost[x].buffsize) {
            if(p >= l) break;
            sniffhost[x].buff[sniffhost[x].bufflen++] = *p++;
        }
        if(sniffhost[x].bufflen == sniffhost[x].buffsize) {
            ubi_algo(sniffhost[x].buff, sniffhost[x].buffsize);
            show_dump(sniffhost[x].buff, sniffhost[x].buffsize, stdout);
            show_dump(sniffhost[x].buff, sniffhost[x].buffsize, fdlog);
            sniffhost[x].bufflen  = 0;
            sniffhost[x].buffsize = 0;
            sniffhost[x].headlen  = 0;
        }
    }

    fflush(fdlog);
}



u32 resolv(char *host) {
    struct  hostent *hp;
    u32     host_ip;

    host_ip = inet_addr(host);
    if(host_ip == INADDR_NONE) {
        hp = gethostbyname(host);
        if(!hp) {
            printf("\nError: Unable to resolve hostname (%s)\n", host);
            exit(1);
        } else host_ip = *(u32 *)hp->h_addr;
    }
    return(host_ip);
}


