/*

    Copyright 2004,2005,2006 Luigi Auriemma

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    http://www.gnu.org/licenses/gpl.txt

*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include "umodcrc.h"

#ifdef WIN32
    #include <direct.h>
    #include <malloc.h>

    #define u_int  unsigned int
    #define u_char  unsigned char
#else
    #include <unistd.h>
    #include <sys/stat.h>
#endif



#define VER         "0.3.3"
#define BUFFSZ      32768
#define NAMESZ      256
#define UMODSIGN    0x9fe3c5a3



long fread_index(FILE *fd);
void create_dir(char *name);
void getfile(FILE *fdin, u_char *fname, u_int size);
void std_err(void);
void io_err(void);



struct end {
    unsigned int   sign;
    unsigned int   indexstart;
    unsigned int   indexend;
    unsigned int   one;
    unsigned int   crc;
} end;



int main(int argc, char *argv[]) {
    FILE    *fd,
            *fd2;
    u_char  fname[NAMESZ + 1],
            *destdir = ".";  // option: destination directory, default current
    u_int  offset,
            size,
            flag,
            tot,
            crc;
    int     value,
            i,
            skipped   = 0,
            list      = 0,  // option: listing of files only
            overwrite = 0,  // option: confirmation for files overwriting
            chksum    = 0,  // option: checksum only
            ignore    = 0,  // option: ignore package errors
            corrcrc   = 0,  // option: write a correct crc
            all       = 0;  // option: all the files
    struct  stat    xstat;


    setbuf(stdout, NULL);

    fputs("\n"
        "UMOD extractor "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", stdout);

    if(argc < 2) {
        printf("\nUsage: %s [options] <file.umod>\n"
            "\n"
            "Options:\n"
            "-l       lists all the files inside the package without extracting them\n"
            "-d DIR   destination directory where will be extracted the files\n"
            "         By default the files will be extracted in the current directory\n"
            "         (I higly suggest to use this option)\n"
            "-o       overwrites the existing files without asking user's confirmation\n"
            "-c       calculates the package checksum, shows if it is the same of that\n"
            "         contained in the package and then exits\n"
            "-C       overwrites the current package checksum with one correct (useless)\n"
            "-i       ignores package errors. By default the program exits if the\n"
            "         checksum and the sign are incorrects, so this option is used to\n"
            "         force the extraction (not suggested)\n"
            "-a       extracts all the files. By default the files with flag 3\n"
            "         (manifest.ini for example) are skipped because not needed\n"
            "\n", argv[0]);
        exit(1);
    }

    argc--;
    for(i = 1; i < argc; i++) {
        switch(argv[i][1]) {
            case 'l': list = 1; break;
            case 'd': destdir = argv[++i]; break;
            case 'o': overwrite = 1; break;
            case 'c': chksum = 1; break;
            case 'C': corrcrc = 1; break;
            case 'i': ignore = 1; break;
            case 'a': all = 1; break;
            default: {
                printf("\nError: wrong argument (%s)\n", argv[i]);
                exit(1);
            }
        }
    }

        /* used for index */
    printf("- open input file \"%s\"\n", argv[argc]);
    fd = fopen(argv[argc], "rb");
    if(!fd) std_err();

        /* used for files */
    fd2 = fopen(argv[argc], "rb");
    if(!fd2) std_err();

        /* crc calculation */
    fputs("- calculate CRC (needs a bit of time)\n", stdout);
    crc = umodcrc(NULL, fd);

        /* change destination directory */
    if(!list && !chksum && !corrcrc) {
        printf("- enter in directory \"%s\": ", destdir);
        if(chdir(destdir) < 0) std_err();
        fputs("ok\n", stdout);
    }

    if(fseek(fd, -sizeof(end), SEEK_END)
      < 0) std_err();

    if(fread(&end, sizeof(end), 1, fd) != 1) io_err();

    printf("- Correct CRC: %08x\n"
           "- Package CRC: %08x\n",
            crc,
            end.crc);

    if(crc != end.crc) {
        fputs("\nError: Wrong checksum, package is corrupted!\n", stdout);
        if(!ignore && !list && !corrcrc) exit(1);
        if(corrcrc) {
            fclose(fd);
            fclose(fd2);
            fd = fopen(argv[argc], "r+b");
            if(!fd) std_err();
            fseek(fd, -4, SEEK_END);
            if(fwrite(&crc, 4, 1, fd) != 1) io_err();
            fclose(fd);
            printf("\nChecksum successfully changed\n");
            exit(0);
        }
    } else {
        if(corrcrc) {
            printf("\nChecksum is correct so will not be changed\n");
            exit(0);
        }
    }

    if(chksum) exit(0);

    if(end.sign != UMODSIGN) {
        fputs("\nError: Wrong package sign (it is not a correct umod package)\n", stdout);
        if(!ignore && !list) exit(1);
    }

    if(fseek(fd, end.indexstart, SEEK_SET)
      < 0) std_err();

    tot = fread_index(fd);      // total number of files

    for(i = 0; i < tot; i++) {
        value = fread_index(fd);
        if(fread(fname, value, 1, fd) != 1) io_err();
        if(fread(&offset, 4, 1, fd) != 1) io_err();
        if(fread(&size, 4, 1, fd) != 1) io_err();
        if(fread(&flag, 4, 1, fd) != 1) io_err();

        printf("\n"
            "File:   %s\n"
            "Offset: %08x   Size: %u\n",
            fname, offset, size);

        if(list) continue;

        if(!all) {
            if(flag == 3) {
                fputs("setup file, not needed\n", stdout);
                skipped++;
                continue;
            }
        }

            /* security and unix/win management */
        create_dir(fname);

        if(!overwrite) {
            if(!stat(fname, &xstat)) {
                fputs("File already exists, do you want to overwrite it (y/n)? ", stdout);
                fflush(stdin);
                do {
                    value = fgetc(stdin);
                } while(value == '\n');

                if((value != 'y') && (value != 'Y')) {
                    fputs("  *SKIPPED*\n", stdout);
                    skipped++;
                    continue;
                }
                fputs("  *OVERWRITTEN*\n", stdout);
            }
        }

        if(fseek(fd2, offset, SEEK_SET) < 0) std_err();

        getfile(fd2, fname, size);
    }

    fclose(fd);
    fclose(fd2);
    if(!list) {
        printf("\n%u files extracted\n\n", tot - skipped);
    } else {
        printf("\n%u files in the package\n\n", tot);
    }

    return(0);
}



void getfile(FILE *fdin, u_char *fname, u_int size) {
    FILE    *fdout;
    int     len;
    u_char  *buff;


    fdout = fopen(fname, "wb");
    if(!fdout) std_err();

    buff = malloc(BUFFSZ);
    if(!buff) std_err();

    len = BUFFSZ;
    while(size) {
        if(len > size) len = size;

        if(fread(buff, len, 1, fdin) != 1) io_err();

        if(fwrite(buff, len, 1, fdout) != 1) io_err();

        size -= len;
    }

    fclose(fdout);
    free(buff);
}



long fread_index(FILE *fd) {
    int     result = 0;
    u_char  b0,
            b1,
            b2,
            b3,
            b4;

    b0 = fgetc(fd);
    if(b0 & 0x40) {
        b1 = fgetc(fd);
        if(b1 & 0x80) {
            b2 = fgetc(fd);
            if(b2 & 0x80) {
                b3 = fgetc(fd);
                if(b3 & 0x80) {
                    b4 = fgetc(fd);
                    result = b4;
                }
                result = (result << 7) | (b3 & 0x7f);
            }
            result = (result << 7) | (b2 & 0x7f);
        }
        result = (result << 7) | (b1 & 0x7f);
    }
    result = (result << 6) | (b0 & 0x3f);
    if(b0 & 0x80) result = -result;

    return(result);
}



void create_dir(char *name) {
    char    *stri,
            *strf;

        /* security */
    stri = name;
    while(1) {
        if((*stri == '/') || (*stri == '\\') || (stri[1] == ':')) {
            *stri = '_';
        } else break;
        stri++;
    }

    stri = name;
    while(1) {
        strf = strchr(stri, '\\');
        if(!strf) {
            strf = strchr(stri, '/');
            if(!strf) break;
        }
        *strf = 0;

        if(*stri == '.') *stri = '_';   // security
        stri = strf + 1;
#ifdef WIN32
        mkdir(name);
        *strf = '\\';
#else
        mkdir(name, 0);
        *strf = '/';
#endif
    }
}



void std_err(void) {
    perror("\nError");
    exit(1);
}



void io_err(void) {
    fputs("\nError: I/O error, the file is incomplete or the disk space is finished\n", stdout);
    exit(1);
}



