/*

Unreal engine package file checker
by Luigi Auriemma
e-mail: aluigi@autistici.org
web:    aluigi.org


This utility check if a package file used by games based on Unreal
engine is correct.

Package file can be maps, sound, music and all the other type of
files used for contain data.
for example: *.uxx, *.unr, *.umx, *.uax, *.utx, *.u, *.u3d


    Copyright 2004,2005,2006 Luigi Auriemma

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    http://www.gnu.org/licenses/gpl.txt

UNIX VERSION

*/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dirent.h>
#include <sys/stat.h>


#define VER        "0.2"
#define SIGN        0x9e2a83c1
#define BUFFSZ        256


void unrcheck(unsigned char *filename, unsigned int totfsize);
void searchdir(unsigned char *dir);
void make_err(int errtype);
void read_import_table(FILE *fd);
unsigned int read_export_table(FILE *fd);
long read_index(FILE *fd);
void std_err(void);
void len_err(void);


#pragma pack(1)


struct GlobalHeader {
    unsigned int    Signature;
    unsigned int    PackageVersion;
    unsigned int    PackageFlags;
    unsigned int    NameCount;
    unsigned int    NameOffset;
    unsigned int    ExportCount;
    unsigned int    ExportOffset;
    unsigned int    ImportCount;
    unsigned int    ImportOffset;
} GlobalHeader;


struct Heritage {
    unsigned int    Count;
    unsigned int    Offset;
} Heritage;


struct Generation1 {
    unsigned char    GUID[16];
    unsigned int    Count;
} Generation1;


struct Generation2 {
    unsigned int    ExportCount;
    unsigned int    NameCount;
} Generation2;



u_int    total = 0,
    nototal = 0;



int main(int argc, char *argv[]) {
    setbuf(stdout, NULL);

    fputs("\n"
        "Unreal engine package file checker "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", stdout);


    if(argc < 2) {
        printf("\nUsage: %s <Unreal_package_file/directory>\n", argv[0]);
//NOTE
printf("\nNote: 1) for the moment use the utility ONLY on map files\n"
       "        2) the utility will stop automatically if found an error\n"
       "        3) the utility scans directories recursively\n"
       "        4) the utility doesn't look for file extensions so be sure that\n"
       "           the directory contain only package files\n"
       "        5) if you want to scan only specific extensions, I suggest to use:\n"
       "           `find /path -name \"*.ext\" -exec %s {} \\;'\n"
       "\n", argv[0]);
//NOTE
        exit(1);
    }

    searchdir(argv[1]);

    printf("\n\n"
        "----------------------------------\n"
        "Total files scanned:              %u\n"
        "Files that are not Package files: %u\n"
        "----------------------------------\n\n",
        total, nototal);

    return(0);
}


void unrcheck(unsigned char *filename, unsigned int totfsize) {
    FILE        *fd;
    int        i,
            err;
    unsigned int    nameoff,
            expoff,
            impoff,
            endoff,
            heroff;
    int         byte;
    unsigned char    buff[BUFFSZ];


    fd = fopen(filename, "rb");
    if(!fd) std_err();


    /*****************/
    /* INDEX READING */
    /*****************/

    printf("\n----------------[%u]----------------\n"
        "*FILENAME*      %s\n",
        total + 1,
        filename);

    err = fread(&GlobalHeader, 1, sizeof(GlobalHeader), fd);
    if((err < sizeof(GlobalHeader)) || (GlobalHeader.Signature != SIGN)) {
        nototal++;
        return;
    }


    printf("PackageVersion  %u\n"
        "NameCount       %u (offset 0x%08X)\n"
        "ExportCount     %u (offset 0x%08X)\n"
        "ImportCount     %u (offset 0x%08X)\n"
        "\n",
        GlobalHeader.PackageVersion,
        GlobalHeader.NameCount,
        GlobalHeader.NameOffset,
        GlobalHeader.ExportCount,
        GlobalHeader.ExportOffset,
        GlobalHeader.ImportCount,
        GlobalHeader.ImportOffset);

    if(!GlobalHeader.PackageVersion ||
       !GlobalHeader.NameCount ||
       !GlobalHeader.NameOffset ||
       !GlobalHeader.ExportCount ||
       !GlobalHeader.ExportOffset ||
       !GlobalHeader.ImportCount ||
       !GlobalHeader.ImportOffset) make_err(6);



    /**********************************/
    /* HERITAGE & GENERATION CHECKING */
    /**********************************/


    if(GlobalHeader.PackageVersion < 68) {
        err = fread(&Heritage, 1, sizeof(Heritage), fd);
        if(err < sizeof(Heritage)) len_err();
        err = fseek(fd, Heritage.Offset, SEEK_SET);
        if(err != 0) std_err();
        printf("Heritage checking   (0x%08X): ", ftell(fd));
        for(i = 0; i < Heritage.Count; i++) {
            err = fseek(fd, 16, SEEK_CUR);
            if(err != 0) std_err();
        }
    } else {
        err = fread(&Generation1, 1, sizeof(Generation1), fd);
        if(err < sizeof(Generation1)) len_err();
        printf("Generation checking (0x%08X): ", ftell(fd));
        for(i = 0; i < Generation1.Count; i++) {
            err = fread(&Generation2, 1, sizeof(Generation2), fd);
            if(err < sizeof(Generation2)) len_err();
        }
    }
    printf("OK\n");
    heroff = ftell(fd);



    /*****************/
    /* NAME CHECKING */
    /*****************/

    err = fseek(fd, GlobalHeader.NameOffset, SEEK_SET);
    if(err != 0) std_err();
    printf("Name checking       (0x%08X): ", ftell(fd));

    for(i = 0; i < GlobalHeader.NameCount; i++) {
        if(GlobalHeader.PackageVersion < 64) {
            while(!feof(fd)) {
                fread(&err, 1, 1, fd);
                if(err == 0) break;
            }
        } else {
            byte = read_index(fd);
            /* NEGATIVE BUG CHECKING */
            if(byte & 0x80000000) make_err(8);
            if(byte < BUFFSZ) {
                err = fread(buff, 1, byte, fd);
                if(err < byte) len_err();
                buff[byte] = 0;

                if(strlen(buff) != (byte - 1)) make_err(1);
            } else {
                if(byte > (totfsize - ftell(fd))) make_err(7);
                err = fseek(fd, byte, SEEK_CUR);
                if(err != 0) std_err();
            }
        }
        err = fseek(fd, 4, SEEK_CUR);
        if(err != 0) std_err();
    }
    printf("OK\n");
    nameoff = ftell(fd);



    /*******************/
    /* EXPORT CHECKING */
    /*******************/

    err = fseek(fd, GlobalHeader.ExportOffset, SEEK_SET);
    if(err != 0) std_err();
    printf("Export checking     (0x%08X): ", ftell(fd));

    for(i = 0; i < GlobalHeader.ExportCount; i++) {
        nameoff += read_export_table(fd);
    }
    printf("OK\n");
    expoff = ftell(fd);


    /*******************/
    /* IMPORT CHECKING */
    /*******************/

    err = fseek(fd, GlobalHeader.ImportOffset, SEEK_SET);
    if(err != 0) std_err();
    printf("Import checking     (0x%08X): ", ftell(fd));

    for(i = 0; i < GlobalHeader.ImportCount; i++) {
        read_import_table(fd);
    }
    printf("OK\n");
    impoff = ftell(fd);



    /********************************/
    /* CHECK FILE "SIZE CHECKPOINTS */
    /********************************/

    err = fseek(fd, 0, SEEK_END);
    if(err != 0) std_err();
    endoff = ftell(fd);
    fclose(fd);

/*
    printf("\nOffsets of the sections:\n");

    printf("Heritage   %08X: ", heroff);
    if((heroff != GlobalHeader.NameOffset) &&
       (heroff != GlobalHeader.ExportOffset) &&
       (heroff != GlobalHeader.ImportOffset) &&
       (heroff != endoff)) make_err(3);
       else printf("OK\n");

    printf("Name       %08X: ", nameoff);
    if((nameoff != heroff) &&
       (nameoff != GlobalHeader.ExportOffset) &&
       (nameoff != GlobalHeader.ImportOffset) &&
       (nameoff != endoff)) make_err(3);
       else printf("OK\n");

    printf("Export     %08X: ", expoff);
    if((expoff != heroff) &&
       (expoff != GlobalHeader.NameOffset) &&
       (expoff != GlobalHeader.ImportOffset) &&
       (expoff != endoff)) make_err(4);
       else printf("OK\n");

    printf("Import     %08X: ", impoff);
    if((impoff != heroff) &&
       (impoff != GlobalHeader.ExportOffset) &&
       (impoff != GlobalHeader.NameOffset) &&
       (impoff != endoff)) make_err(5);
       else printf("OK\n");
*/
    printf("\n\nPackage file seems to be OK!\n");
    total++;
}


void make_err(int errtype) {
    char    *err;

    switch(errtype) {
        case 0: err = "The size of a name is different from the real or is equal to 0 (happen with packages version minor than 64; sometimes contain music data).\nDon't scan this file"; break;
        case 1: err = "The size of a name is different from the real (faked or not standard)"; break;
        case 2: err = "Incorrect signature (not a real package)"; break;
        case 3: err = "Name section has incorrect size"; break;
        case 4: err = "Export section has incorrect size"; break;
        case 5: err = "Import section has incorrect size"; break;
        case 6: err = "One of the package tables doesn't exist"; break;
        case 7: err = "An index number is faked (major than total file size)"; break;
        case 8: err = "Negative sign bug used. This file is hacked!"; break;
        default: err = "unknown"; break;
    }

    printf("\n\n"
        "*********************************************\n"
        "* Package could be modified or not standard *\n"
        "*********************************************\n"
        "\nError: %s\n\n"
        "NOTE: Delete or move the file in a temporary directory and relaunch the utility for scan the other package files\n\n", err);
    exit(1);
}


void searchdir(unsigned char *dir) {
    unsigned char    fullname[BUFSIZ];
    struct    stat    xstat;
    struct    dirent    **namelist;
    int        n,
            i;

    stat(dir, &xstat);
    if(S_ISDIR(xstat.st_mode)) {
        n = scandir(dir, &namelist, 0, alphasort);
        if(n >= 0) {
            for(i = 2; i < n; i++) {
                sprintf(fullname, "%s/%s", dir, namelist[i]->d_name);
                stat(fullname, &xstat);
                if(!S_ISDIR(xstat.st_mode)) unrcheck(fullname, xstat.st_size);

                if(S_ISDIR(xstat.st_mode)) searchdir(fullname);
            }
        }
    } else unrcheck(dir, xstat.st_size);
}


void read_import_table(FILE *fd) {
    int    err;

    read_index(fd);            /* ClassPackage */
    read_index(fd);            /* ClassName */
    err = fseek(fd, 4, SEEK_CUR);    /* Package */
    if(err != 0) std_err();
    read_index(fd);            /* Object Name */
}


unsigned int read_export_table(FILE *fd) {
    int     size;
    int    err;

    read_index(fd);            /* Class */
    read_index(fd);            /* Super */
    err = fseek(fd, 4, SEEK_CUR);    /* Package */
    if(err != 0) std_err();
    read_index(fd);            /* ObjectName */
    err = fseek(fd, 4, SEEK_CUR);    /* ObjectFlags */
    if(err != 0) std_err();
    size = read_index(fd);        /* SerialSize */
    read_index(fd);        /* SerialOffset */

    return(size);
}


long read_index(FILE *fd) {
    int         result = 0;
    unsigned char    b0,
            b1,
            b2,
            b3,
            b4;
    int        err;

    err = fread(&b0, 1, 1, fd);
    if(err < 1) len_err();
    if(b0 & 0x40) {
        err = fread(&b1, 1, 1, fd);
        if(err < 1) len_err();
        if(b1 & 0x80) {
            err = fread(&b2, 1, 1, fd);
            if(err < 1) len_err();
            if(b2 & 0x80) {
                err = fread(&b3, 1, 1, fd);
                if(err < 1) len_err();
                if(b3 & 0x80) {
                    err = fread(&b4, 1, 1, fd);
                    if(err < 1) len_err();
                    result = b4;
                }
                result = (result << 7) | (b3 & 0x7f);
            }
            result = (result << 7) | (b2 & 0x7f);
        }
        result = (result << 7) | (b1 & 0x7f);
    }
    result = (result << 6) | (b0 & 0x3f);
    if(b0 & 0x80) result = -result;

    return(result);
}


void std_err(void) {
    perror("\nError");
    exit(1);
}


void len_err(void) {
    printf("\nError: File is corrupted\n");
    exit(1);
}
