/*

Unreal engine package file checker
by Luigi Auriemma
e-mail: aluigi@autistici.org
web:    aluigi.org


This utility check if a package file used by games based on Unreal
engine is correct.

Package file can be maps, sound, music and all the other type of
files used for contain data.
for example: *.uxx, *.unr, *.umx, *.uax, *.utx, *.u, *.u3d


    Copyright 2004,2005,2006 Luigi Auriemma

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    http://www.gnu.org/licenses/gpl.txt

WIN32 VERSION

*/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <windows.h>
#include <direct.h>
#include <errno.h>
#include <direct.h>
#include <sys/stat.h>


#define VER        "0.2"
#define SIGN        0x9e2a83c1
#define BUFFSZ        256
#define FNAMESZ        81920


char *get_file(void);
int unrcheck(FILE *fd, char *filename);
void make_err(int errtype, char *fname);
void read_import_table(FILE *fd);
unsigned int read_export_table(FILE *fd);
long read_index(FILE *fd);
void msgbox_err(void);
void len_err(void);


struct GlobalHeader {
    unsigned int    Signature;
    unsigned int    PackageVersion;
    unsigned int    PackageFlags;
    unsigned int    NameCount;
    unsigned int    NameOffset;
    unsigned int    ExportCount;
    unsigned int    ExportOffset;
    unsigned int    ImportCount;
    unsigned int    ImportOffset;
} GlobalHeader;


struct Heritage {
    unsigned int    Count;
    unsigned int    Offset;
} Heritage;


struct Generation1 {
    unsigned char    GUID[16];
    unsigned int    Count;
} Generation1;


struct Generation2 {
    unsigned int    ExportCount;
    unsigned int    NameCount;
} Generation2;


char    *directory;
int    total = 0;


int main(void) {
    FILE        *fd;
    char        tmp[256],
            *filename;
    int        errnum = 0,
            err;




    setbuf(stdout, NULL);

    fputs("\n"
        "Unreal engine package file checker "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n"
        "\nNote: 1) for the moment use the utility ONLY on map files\n"
        "        2) the utility will stop automatically if found an error\n"
        "        3) the utility scans directories recursively\n"
        "        4) the utility doesn't look for file extensions so be sure that\n"
        "           the directory contain only package files\n"
        "        5) if you want to scan only specific extensions, I suggest to use:\n"
        "           `find /path -name \"*.ext\" -exec %s {} \\;'\n"
        "\n", stdout);




    filename = get_file();

    while(1) {
        if(*filename == '\0') break;

        fd = fopen(filename, "rb");
        if(!fd) msgbox_err();

        err = unrcheck(fd, filename);
        if(err < 0) errnum++;

        fclose(fd);
        filename += strlen(filename) + 1;
    }

    if(errnum == 0) {
        sprintf(tmp,
            "All files are ok!\n"
            "-------------------------\n"
            "Total files scanned:     %d\n"
            "-------------------------\n"
            "\n", total);
         MessageBox(0, tmp , "Success", MB_OK | MB_ICONINFORMATION | MB_TASKMODAL);
    } else {
        snprintf(tmp, sizeof(tmp) - 1, "%d package files have problems\n", errnum);
        MessageBox(0, tmp , "Problems...", MB_OK | MB_ICONINFORMATION | MB_TASKMODAL);
    }

    return(0);
}


char *get_file(void) {
    OPENFILENAME    ofn;
    static char    filename[FNAMESZ];
    int        err;

    memset(&ofn, 0, sizeof(ofn));
    ofn.lStructSize  = sizeof(ofn);
    ofn.hwndOwner    = NULL;
    ofn.lpstrFilter  = "Unreal Package Files (*.u*)\0*.u*\0All Package Files (*.*)\0*.*\0";
    ofn.nFilterIndex = 2;
    ofn.lpstrFile    = filename;
    ofn.nMaxFile     = sizeof(filename);
    ofn.lpstrTitle   = "Unreal engine package file checker: select files";
    ofn.Flags        = OFN_PATHMUSTEXIST | OFN_FILEMUSTEXIST | OFN_ALLOWMULTISELECT | OFN_LONGNAMES | OFN_EXPLORER | OFN_ENABLEHOOK;

    err = GetOpenFileName(&ofn);
    if(err == 0) exit(1);

    filename[ofn.nFileOffset - 1] = 0;
    chdir(filename);
    directory = malloc(strlen(filename));
    strcpy(directory, filename);

    return(filename + ofn.nFileOffset);
}


int unrcheck(FILE *fd, char *filename) {
    int        i,
            err;
    int         byte;
    unsigned char    buff[BUFFSZ];
    unsigned int    nameoff,
            expoff,
            impoff,
            endoff,
            heroff,
            totfsize;
    struct    stat    xstat;
    char    filedir[BUFSIZ];

    sprintf(filedir, "%s\\%s", directory, filename);
    stat(filedir, &xstat);
    totfsize = xstat.st_size;



    /*****************/
    /* INDEX READING */
    /*****************/

    err = fread(&GlobalHeader, 1, sizeof(GlobalHeader), fd);
    if(err < sizeof(GlobalHeader)) len_err();
    if(GlobalHeader.Signature != SIGN) {
        make_err(2, filename);
        return(-1);
    }

    printf("PackageVersion  %u\n"
        "NameCount       %u (offset 0x%08X)\n"
        "ExportCount     %u (offset 0x%08X)\n"
        "ImportCount     %u (offset 0x%08X)\n"
        "\n",
        GlobalHeader.PackageVersion,
        GlobalHeader.NameCount,
        GlobalHeader.NameOffset,
        GlobalHeader.ExportCount,
        GlobalHeader.ExportOffset,
        GlobalHeader.ImportCount,
        GlobalHeader.ImportOffset);


    if(!GlobalHeader.PackageVersion ||
       !GlobalHeader.NameCount ||
       !GlobalHeader.NameOffset ||
       !GlobalHeader.ExportCount ||
       !GlobalHeader.ExportOffset ||
       !GlobalHeader.ImportCount ||
       !GlobalHeader.ImportOffset) {
        make_err(6, filename);
        return(-1);
    }



    /**********************************/
    /* HERITAGE & GENERATION CHECKING */
    /**********************************/


    if(GlobalHeader.PackageVersion < 68) {
        err = fread(&Heritage, 1, sizeof(Heritage), fd);
        if(err < sizeof(Heritage)) len_err();
        err = fseek(fd, Heritage.Offset, SEEK_SET);
        if(err != 0) msgbox_err();
        printf("Heritage checking   (0x%08X): ", ftell(fd));
        for(i = 0; i < Heritage.Count; i++) fseek(fd, 16, SEEK_CUR);
    } else {
        err = fread(&Generation1, 1, sizeof(Generation1), fd);
        if(err < sizeof(Generation1)) len_err();
        printf("Generation checking (0x%08X): ", ftell(fd));
        for(i = 0; i < Generation1.Count; i++) {
            err = fread(&Generation2, 1, sizeof(Generation2), fd);
            if(err < sizeof(Generation2)) len_err();
        }
    }
    printf("OK\n");
    heroff = ftell(fd);



    /*****************/
    /* NAME CHECKING */
    /*****************/

    err = fseek(fd, GlobalHeader.NameOffset, SEEK_SET);
    if(err != 0) msgbox_err();
    printf("Name checking       (0x%08X): ", ftell(fd));

    for(i = 0; i < GlobalHeader.NameCount; i++) {
        if(GlobalHeader.PackageVersion < 64) {
            while(!feof(fd)) {
                fread(&err, 1, 1, fd);
                if(err == 0) break;
            }
        } else {
            byte = read_index(fd);
            /* NEGATIVE BUG CHECKING */
            if(byte & 0x80000000) {
                make_err(8, filename);
                return(-1);
            }
            if(byte < BUFFSZ) {
                err = fread(buff, 1, byte, fd);
                if(err < byte) len_err();
                buff[byte] = 0;
                    if(GlobalHeader.PackageVersion >= 64) {
                    if(strlen(buff) != (byte - 1)) {
                        make_err(1, filename);
                        return(-1);
                    }
                } else {
                    if(strlen(buff) != byte) {
                        make_err(0, filename);
                        return(-1);
                    }
                }
            } else {
                if(byte > (totfsize - ftell(fd))) {
                    make_err(7, filename);
                    return(-1);
                }
                err = fseek(fd, byte, SEEK_CUR);
                if(err != 0) msgbox_err();
            }
        }
        err = fseek(fd, 4, SEEK_CUR);
        if(err != 0) msgbox_err();
    }
    printf("OK\n");
    nameoff = ftell(fd);



    /*******************/
    /* EXPORT CHECKING */
    /*******************/

    err = fseek(fd, GlobalHeader.ExportOffset, SEEK_SET);
    if(err != 0) msgbox_err();
    printf("Export checking     (0x%08X): ", ftell(fd));

    for(i = 0; i < GlobalHeader.ExportCount; i++) {
        nameoff += read_export_table(fd);
    }
    printf("OK\n");
    expoff = ftell(fd);



    /*******************/
    /* IMPORT CHECKING */
    /*******************/

    err = fseek(fd, GlobalHeader.ImportOffset, SEEK_SET);
    if(err != 0) msgbox_err();
    printf("Import checking     (0x%08X): ", ftell(fd));

    for(i = 0; i < GlobalHeader.ImportCount; i++) {
        read_import_table(fd);
    }
    printf("OK\n");
    impoff = ftell(fd);



    /********************************/
    /* CHECK FILE "SIZE CHECKPOINTS */
    /********************************/

    err = fseek(fd, 0, SEEK_END);
    if(err != 0) msgbox_err();
    endoff = ftell(fd);
    fclose(fd);

/*
    if((heroff != GlobalHeader.NameOffset) &&
       (heroff != GlobalHeader.ExportOffset) &&
       (heroff != GlobalHeader.ImportOffset) &&
       (heroff != endoff)) {
        make_err(3, filename);
        return(-1);
    }

    if((nameoff != heroff) &&
       (nameoff != GlobalHeader.ExportOffset) &&
       (nameoff != GlobalHeader.ImportOffset) &&
       (nameoff != endoff)) {
        make_err(3, filename);
        return(-1);
    }

    if((expoff != heroff) &&
       (expoff != GlobalHeader.NameOffset) &&
       (expoff != GlobalHeader.ImportOffset) &&
       (expoff != endoff)) {
        make_err(4, filename);
        return(-1);
    }

    if((impoff != heroff) &&
       (impoff != GlobalHeader.ExportOffset) &&
       (impoff != GlobalHeader.NameOffset) &&
       (impoff != endoff)) {
        make_err(5, filename);
        return(-1);
    }
*/

    printf("\n\nPackage file seems to be OK!\n");
    total++;
    return(0);
}



void make_err(int errtype, char *fname) {
    char    *err;

    switch(errtype) {
        case 0: err = "The size of a name is different from the real or is equal to 0 (happen with packages version minor than 64; sometimes contain music data)"; break;
        case 1: err = "The size of a name is different from the real (faked or not standard)"; break;
        case 2: err = "Incorrect signature (not a real package)"; break;
        case 3: err = "Name section has incorrect size"; break;
        case 4: err = "Export section has incorrect size"; break;
        case 5: err = "Import section has incorrect size"; break;
        case 6: err = "One of the package tables doesn't exist"; break;
        case 7: err = "An index number is faked (major than total file size)"; break;
        case 8: err = "Negative sign bug used. This file is HACKED!!!"; break;
        default: err = "Error not recognized"; break;
    }

    MessageBox(0, err, fname, MB_OK | MB_ICONERROR | MB_TASKMODAL);
}


void read_import_table(FILE *fd) {
    int    err;

    read_index(fd);            /* ClassPackage */
    read_index(fd);            /* ClassName */
    err = fseek(fd, 4, SEEK_CUR);    /* Package */
    if(err != 0) msgbox_err();
    read_index(fd);            /* Object Name */
}


unsigned int read_export_table(FILE *fd) {
    int     size;
    int    err;

    read_index(fd);            /* Class */
    read_index(fd);            /* Super */
    err = fseek(fd, 4, SEEK_CUR);    /* Package */
    if(err != 0) msgbox_err();
    read_index(fd);            /* ObjectName */
    err = fseek(fd, 4, SEEK_CUR);    /* ObjectFlags */
    if(err != 0) msgbox_err();
    size = read_index(fd);        /* SerialSize */
    read_index(fd);            /* SerialOffset */

    return(size);
}


long read_index(FILE *fd) {
    int         result = 0;
    unsigned char    b0,
            b1,
            b2,
            b3,
            b4;
    int        err;

    err = fread(&b0, 1, 1, fd);
    if(err < 1) len_err();
    if(b0 & 0x40) {
        err = fread(&b1, 1, 1, fd);
        if(err < 1) len_err();
        if(b1 & 0x80) {
            err = fread(&b2, 1, 1, fd);
            if(err < 1) len_err();
            if(b2 & 0x80) {
                err = fread(&b3, 1, 1, fd);
                if(err < 1) len_err();
                if(b3 & 0x80) {
                    err = fread(&b4, 1, 1, fd);
                    if(err < 1) len_err();
                    result = b4;
                }
                result = (result << 7) | (b3 & 0x7f);
            }
            result = (result << 7) | (b2 & 0x7f);
        }
        result = (result << 7) | (b1 & 0x7f);
    }
    result = (result << 6) | (b0 & 0x3f);
    if(b0 & 0x80) result = -result;

    return(result);
}


void len_err(void) {
    MessageBox(0, strerror(errno), "I/O Error", MB_OK | MB_ICONERROR | MB_TASKMODAL);
    exit(1);
}


void msgbox_err(void) {
    MessageBox(0, strerror(errno), "I/O Error", MB_OK | MB_ICONERROR | MB_TASKMODAL);
    exit(1);
}


