/*

Unreal engine index reader/maker
by Luigi Auriemma
e-mail: aluigi@autistici.org
web:    aluigi.org


This utility can be used for create an Unreal index number or convert
an index number in a long type number


If you don't know what is an Unreal index number (or compact idices)
please read "UT Package File format" by Antonio Cordero:
http://www.acordero.org

However, quickly, it is a long type number that has its bits compacted
for occupy less space than a "normal" long type number.

LINUX & WIN32 VERSION

*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>


#define VER        "0.1a"


int read_index(unsigned char *index_num);
void make_index(int number, unsigned char *index_num);


int main(int argc, char *argv[]) {
    int         tmp[5],
                result = 0;
    unsigned char    bytes[] = "\x00\x00\x00\x00\x00\x00";

    setbuf(stdout, NULL);

    printf("\n"
        "Unreal engine index reader/maker %s\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", VER);

    if(argc < 3) {
        printf("\nUsage: %s [options]\n\n"
            " Options:\n"
            "-l NUM\tlong type number       --> index type number (hex)\n"
            "-x NUM\tlong type number (hex) --> index type number (hex)\n"
            "-i HEX\tindex type number      --> long type number\n"
            "\n"
            " Example:\n"
            "unrindex -l 130000000\n"
            "unrindex -i 4092FD7B\n"
            "\n", argv[0]);
        exit(1);
    }

    if(!argv[2]) {
        printf("\nError: you must insert a number\n");
        exit(1);
    }

    switch(argv[1][1]) {
        case 'x': {
            sscanf(argv[2], "%x", &result);
            make_index(result, bytes);
            printf("Index number: 0x%02hhX%02hhX%02hhX%02hhX%02hhX\n",
                bytes[0],
                bytes[1],
                bytes[2],
                bytes[3],
                bytes[4]);
            }; break;
        case 'l': {
            result = atoi(argv[2]);
            make_index(result, bytes);
            printf("Index number: 0x%02hhX%02hhX%02hhX%02hhX%02hhX\n",
                bytes[0],
                bytes[1],
                bytes[2],
                bytes[3],
                bytes[4]);
            }; break;
        case 'i': {
            sscanf(argv[2], "%02X%02X%02X%02X%02X",   // sscanf and hh don't work
                &tmp[0], &tmp[1], &tmp[2], &tmp[3], &tmp[4]);
            bytes[0] = tmp[0];
            bytes[1] = tmp[1];
            bytes[2] = tmp[2];
            bytes[3] = tmp[3];
            bytes[4] = tmp[4];

            result = read_index(bytes);
            printf("%d\n", result);
            }; break;
        default: {
            printf("\nError: Wrong argument\n");
            exit(1);
            }; break;
    }

    return(0);
}


int read_index(unsigned char *index_num) {
    int         result = 0;
    unsigned char    b0 = index_num[0],
            b1 = index_num[1],
            b2 = index_num[2],
            b3 = index_num[3],
            b4 = index_num[4];

    if(b0 & 0x40) {
        if(b1 & 0x80) {
            if(b2 & 0x80) {
                if(b3 & 0x80) {
                    result = b4;
                }
                result = (result << 7) | (b3 & 0x7f);
            }
            result = (result << 7) | (b2 & 0x7f);
        }
        result = (result << 7) | (b1 & 0x7f);
    }
    result = (result << 6) | (b0 & 0x3f);
    if(b0 & 0x80) result = -result;

    return(result);
}


void make_index(int number, unsigned char *index_num) {
    int    sign = 0;    /* 0 = positive, 1 = negative */

    /* sign */
    if(number & 0x80000000) {
        number = labs(number);
        sign = 1;
    }

    index_num[0] = (number & 0x3f);
    number >>= 6;
    if(number) {
        index_num[0] += 0x40;
        index_num[1] = (number & 0x7f);
        number >>= 7;
        if(number) {
            index_num[1] += 0x80;
            index_num[2] = (number & 0x7f);
            number >>= 7;
            if(number) {
                index_num[2] += 0x80;
                index_num[3] = (number & 0x7f);
                number >>= 7;
                if(number) {
                    index_num[3] += 0x80;
                    index_num[4] = number;
                }
            }
        }
    }
    if(sign) index_num[0] += 0x80;
}

