/*
    Copyright 2005,2006,2007,2008 Luigi Auriemma

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    http://www.gnu.org/licenses/gpl-2.0.txt
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <sys/stat.h>
#include <ctype.h>
#include <zlib.h>

#ifdef WIN32
    #include <direct.h>
#else
    #include <unistd.h>
    #include <sys/types.h>
#endif

typedef uint8_t     u8;
typedef uint16_t    u16;
typedef uint32_t    u32;



#define VER             "0.1.3"
#define VIRTOOLS_SIGN   "Nemo Fi\0"
#define BUFFSZ          4096
#define FILE1           "components"
#define FILE2           "objects"
#define FREAD(A,B,C)    (fread(A, 1, B, C) != B)
#define FWRITE(A,B,C)   (fwrite(A, 1, B, C) != B)



int vxbg_extract(FILE *fd);
void virtools_compobj(FILE *fd, u8 *fname, u32 insz, u32 outsz);
u32 virtools_scan(FILE *fd);
char *virtdate(u32 num);
void check_badname(char *fname);
void getfile(FILE *fd, char *fname, u32 size, int offset);
void unzip(FILE *fd, char *fname, u32 *inlen, u32 *outlen);
void write_err(void);
void read_err(void);
void std_err(void);



typedef struct {
    u8      sign[8];
    u32     crc;
    u32     date;
    u32     plugin1;
    u32     plugin2;
    u32     flags;
    u32     compcsz;
    u32     objcsz;
    u32     objsz;
    u32     addpath;
    u32     components;
    u32     objects;
    u32     zero;
    u32     version;
    u32     compsz;
} vmo_t;

int     listonly    = 0;



int main(int argc, char *argv[]) {
    struct stat xstat;
    FILE    *fd;
    vmo_t   vmo;
    u32     i,
            len,
            off;
    u8      fname[BUFFSZ],
            ans[16],
            *dir = NULL;

    setbuf(stdout, NULL);

    fputs("\n"
        "Virtools files unpacker "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", stdout);

    if(argc < 2) {
        printf("\n"
            "Usage: %s [options] <file.EXT>\n"
            "\n"
            "Options:\n"
            "-l        lists the archived files\n"
            "-d DIR    folder in which extracting the files\n"
            "\n"
            "File can be VMO, NMS, NMO, CMO, VBF or any other of the same format\n"
            "Some big executables could contain VMO files, you can extract their resource\n"
            "with a resources extractor or passing them to this tool directly (autoscan)\n"
            "\n", argv[0]);
        exit(1);
    }

    argc--;
    for(i = 1; i < argc; i++) {
        switch(argv[i][1]) {
            case 'l': listonly  = 1;            break;
            case 'd': dir       = argv[++i];    break;
            default: {
                printf("\nError: wrong command-line argument (%s)\n\n", argv[i]);
                exit(1);
            } break;
        }
    }

    printf("- open file:        %s\n", argv[argc]);
    fd = fopen(argv[argc], "rb");
    if(!fd) std_err();

    if(dir) {
        printf("- change directory: %s\n", dir);
        if(chdir(dir) < 0) std_err();
    }

    if(FREAD(&vmo, sizeof(vmo), fd)) std_err();
    off = ftell(fd);

    if(!memcmp(&vmo, "VXBG", 4)) {
        i = vxbg_extract(fd);
        goto quit;
    }

    if(memcmp(vmo.sign, VIRTOOLS_SIGN, sizeof(vmo.sign))) {
        printf("- file seems invalid, its signature is \"%.*s\"\n", sizeof(vmo.sign), vmo.sign);
        printf("- do you want to scan it to see if it contains valid Virtools data (y/N)? ");
        fgets(ans, sizeof(ans), stdin);
        if(tolower(ans[0]) != 'y') exit(1);
        off = virtools_scan(fd);
        if(!off) {
            fclose(fd);
            printf("\nError: no valid signature found\n");
            exit(1);
        }
        printf("- Virtools signature found at offset 0x%08x\n", off);
        if(FREAD(&vmo, sizeof(vmo), fd)) std_err();
        off = ftell(fd);
    }

    printf(
        "- date              %s\n"
        "- components:       %u\n"
        "- objects:          %u\n"
        "- version:          %hhu.%hhu.%hhu.%hhu\n",
        virtdate(vmo.date),
        vmo.components,
        vmo.objects,
        (vmo.version >> 24) & 0xff, (vmo.version >> 16) & 0xff,
        (vmo.version >> 8)  & 0xff, vmo.version & 0xff);

    printf("\n- additional raw info:\n");
    printf("  signature         %.*s\n", sizeof(vmo.sign), vmo.sign);
    printf("  crc               %08x\n", vmo.crc);
    printf("  date              %08x\n", vmo.date);
    printf("  plugin1           %08x\n", vmo.plugin1);
    printf("  plugin2           %08x\n", vmo.plugin2);
    printf("  flags             %08x\n", vmo.flags);
    printf("  compcsz           %08x\n", vmo.compcsz);
    printf("  objcsz            %08x\n", vmo.objcsz);
    printf("  objsz             %08x\n", vmo.objsz);
    printf("  addpath           %08x\n", vmo.addpath);
    printf("  components        %08x\n", vmo.components);
    printf("  objects           %08x\n", vmo.objects);
    printf("  zero              %08x\n", vmo.zero);
    printf("  version           %08x\n", vmo.version);
    printf("  compsz            %08x\n", vmo.compsz);

    fstat(fileno(fd), &xstat);
    if((sizeof(vmo) + vmo.compcsz + vmo.objcsz) > xstat.st_size) {
        fclose(fd);
        printf("\nError: the size of components and objects is bigger than the size of the file\n");
        exit(1);
    }

    printf("\n"
        "  insize     outsize    filename\n"
        "  ------------------------------\n");

    if(listonly) {
        printf("  %-10u %-10u %s\n", vmo.compcsz, vmo.compsz, FILE1);
        printf("  %-10u %-10u %s\n", vmo.objcsz,  vmo.objsz,  FILE2);
    } else {
        virtools_compobj(fd, FILE1, vmo.compcsz, vmo.compsz);

        if(fseek(fd, off + vmo.compcsz, SEEK_SET)) std_err();

        virtools_compobj(fd, FILE2, vmo.objcsz, vmo.objsz);
    }
    if(fseek(fd, off + vmo.compcsz + vmo.objcsz, SEEK_SET)) std_err();

    for(i = 2; ; i++) {
        if(FREAD(&len,  4,   fd))   break;
        if(len >= sizeof(fname))    break;  // checks

        if(FREAD(fname, len, fd))   break;
        fname[len] = 0;
        if(!fname[0])               break;

        if(FREAD(&len,  4,   fd))   break;
        printf("             %-10u %s\n", len, fname);

        if(listonly) {
            if(fseek(fd, len, SEEK_CUR)) std_err();
        } else {
            check_badname(fname);
            getfile(fd, fname, len, -1);
        }
    }

quit:
    fclose(fd);
    printf("\n- %u files %s\n\n", i, listonly ? "listed" : "extracted");
    return(0);
}



int vxbg_extract(FILE *fd) {
    u32     offset,
            start,
            size;
    int     i,
            c,
            files   = 0;
    u8      fname[BUFFSZ + 1];

    fseek(fd, 4, SEEK_SET);
    if(FREAD(&offset, 4, fd)) goto quit;

    printf("\n"
        "  offset   size       filename\n"
        "  ----------------------------\n");

    offset += 8;
    for(start = offset; ftell(fd) < start; offset += size) {
        for(i = 0; i < BUFFSZ; i++) {
            c = fgetc(fd);
            fname[i] = c;
            if(!c) break;
            if(c < 0) goto quit;
        }
        if(FREAD(&size, 4, fd)) goto quit;

        printf("  %08x %-10u %s\n", offset, size, fname);
        if(!listonly) getfile(fd, fname, size, offset);
        files++;
    }
quit:
    return(files);
}



void virtools_compobj(FILE *fd, u8 *fname, u32 insz, u32 outsz) {
    u32     inlen,
            outlen;

    printf("  %-10u ", insz);

    if(insz != outsz) {
        unzip(fd, fname, &inlen, &outlen);
        printf("%-10u %s\n", outlen, fname);
        if(outlen != outsz) printf("- wrong output size, should be %u\n", outsz);
    } else {
        getfile(fd, fname, outsz, -1);
        printf("%-10u %s\n", insz, fname);
    }
}



u32 virtools_scan(FILE *fd) {
    u32     offset  = 0;
    int     len;
    u8      buff[BUFFSZ],
            *p,
            *l;

    for(;;) {
        len = fread(buff, 1, BUFFSZ, fd);
        if(len < 8) break;

        l = buff + len - 8;
        for(p = buff; p <= l; p++) {
            if(!memcmp(p, VIRTOOLS_SIGN, 8)) {
                offset = ftell(fd) - (len - (p - buff));
                break;
            }
        }
        if(offset) break;
        
        if(feof(fd)) break;
        fseek(fd, -7, SEEK_CUR);
    }

    fseek(fd, offset, SEEK_SET);
    return(offset);
}



char *virtdate(u32 num) {
    static char ret[32];
    static const char *months[12] = {
            "jan","feb","mar","apr","may","jun",
            "jul","aug","sep","oct","nov","dec" };
    char    mini[5];

    sprintf(mini,   // now I have doubts about this solution but it's old so probably it had a sense when I wrote it... mah
        "%x", ((num >> 16) & 0xff) - 1);

    sprintf(ret,
        "%x %s %x",
        (num >> 24) & 0xff,
        months[atoi(mini) % 12],
        num & 0xffff);
    return(ret);
}



void check_badname(char *fname) {
    if((*fname == '\\') || (*fname == '/') || strchr(fname, ':') || strstr(fname, "..")) {
        printf("\nError: malformed filename \"%s\"\n\n", fname);
        exit(1);
    }
}



int check_overwrite(u8 *fname) {
    FILE    *fd;
    u8      ans[16];

    fd = fopen(fname, "rb");
    if(!fd) return(0);
    fclose(fd);
    printf("  the file already exists, do you want to overwrite it (y/N)? ");
    fgets(ans, sizeof(ans), stdin);
    if(tolower(ans[0]) != 'y') return(-1);
    return(0);
}



void getfile(FILE *fd, char *fname, u32 size, int offset) {
    FILE    *fdo;
    int     len,
            oldoff  = 0;
    u8      buff[BUFFSZ];

    if(check_overwrite(fname) < 0) return;
    fdo = fopen(fname, "wb");
    if(!fdo) std_err();

    if(offset >= 0) {
        oldoff = ftell(fd);
        if(fseek(fd, offset, SEEK_SET)) std_err();
    }

    for(len = BUFFSZ; size; size -= len) {
        if(len > size) len = size;
        if(FREAD(buff,  len, fd)) read_err();
        if(FWRITE(buff, len, fdo)) write_err();
    }

    fclose(fdo);
    if(offset >= 0) {
        fseek(fd, oldoff, SEEK_SET);
    }
}



void std_zerr(z_stream *z) {
    fprintf(stderr, "\nError: zlib error, %s\n", z->msg ? z->msg : "");
    inflateEnd(z);
    exit(1);
}



void unzip(FILE *fd, char *fname, u32 *inlen, u32 *outlen) {
    z_stream    z;
    FILE    *fdo;
    int     zerr,
            len;
    u8      in[BUFFSZ],
            out[BUFFSZ];

    if(check_overwrite(fname) < 0) return;
    fdo = fopen(fname, "wb");
    if(!fdo) std_err();

    z.zalloc = (alloc_func)0;
    z.zfree  = (free_func)0;
    z.opaque = (voidpf)0;
    if(inflateInit2(&z, 15)) {
        printf("\nError: zlib initialization error\n");
        exit(1);
    }

    while((len = fread(in, 1, BUFFSZ, fd)) > 0) {
        z.next_in   = in;
        z.avail_in  = len;
        do {
            z.next_out  = out;
            z.avail_out = BUFFSZ;
            zerr = inflate(&z, Z_NO_FLUSH);
            fwrite(out, BUFFSZ - z.avail_out, 1, fdo);

            if(zerr == Z_STREAM_END) break;
            if(zerr != Z_OK) std_zerr(&z);
        } while(z.avail_in);
        if(zerr == Z_STREAM_END) break;
    }

    *inlen  = z.total_in;
    *outlen = z.total_out;
    inflateEnd(&z);
}



void write_err(void) {
    printf("\nError: impossible to write the output file, probably your disk space is finished\n\n");
    exit(1);
}



void read_err(void) {
    printf("\nError: the file contains unexpected data\n\n");
    exit(1);
}



void std_err(void) {
    perror("\nError");
    exit(1);
}


