/*

by Luigi Auriemma

source code of the autopatcher
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <windows.h>
#include <sys/stat.h>
#include <direct.h>
#include <errno.h>
#include "crc32.h"



#define FILENAME    "engine.dll"
#define FNAMEBCK    "engine.dll.backup"
#define CRC32O      0xFD134F0F
#define CRC32P      0x7E188902
#define VER         "0.2.1"
#define GAMEVER     "3369"
#define TITLE       "Unreal Tournament 2004 Alternative Network Project "VER
#define HEADER      "Unreal Tournament 2004 Alternative Network Project "VER"\n" \
                    "Autopatcher for UT2004 v"GAMEVER" Win32\n" \
                    "by Luigi Auriemma\n" \
                    "e-mail: aluigi@autistici.org\n" \
                    "web:    aluigi.org\n" \
                    "\n" \
                    "\n" \
                    "This patch will:\n" \
                    "- modify your "FILENAME" file saving the original as "FNAMEBCK"\n" \
                    "- add the demo cd-key "DEMOKEY" to your registry\n" \
                    "  your old cd-key (if exists) will be saved as CDKey-backup in your registry:\n" \
                    "  HKEY_LOCAL_MACHINE\\"REGKEY"\n" \
                    "\n" \
                    FILENAME" found in the following directory:\n" \
                    "\n" \
                    "\t%s\n" \
                    "\n" \
                    "Do you want to apply the patch?"
#define WRONGCRC    "Your file "FILENAME" is not the right version!\n" \
                    "You cannot apply this patch until you have an old version\n" \
                    "Go on http://www.ut2004.com and download the v"GAMEVER" patch\n"
#define PATCHED     "Your file "FILENAME" is already patched"
#define SUCCESS     "The file "FILENAME" has been successfully patched!\n" \
                    "The backup copy of the file is called "FNAMEBCK"\n" \
                    "The demo cd-key has been successfully setted\n" \
                    "\n" \
                    "Contact me for any comment, question or feedback"
#define NOFILE      "The file "FILENAME" has not be found so you must launch this " \
                    "patcher from the UT2004\\SYSTEM directory"
#define REGKEY      "Software\\Unreal Technology\\Installed Apps\\UT2004"
#define DEMOKEY     "UT2004-UTDEMO-UTDEMO-UT2004"
#define BUFFSZ      32768



char *get_folder(void);
void copiafile(char *inz, char *outz);
void patch_it(void);
void reg_it(void);
unsigned int crc_file(char *fname);
void std_err(char *err);



int main(void) {
    int             len;
    unsigned int    crc32csum;
    char            *buff,
                    *dir;


    dir = get_folder();

    len = sizeof(HEADER) + MAX_PATH;
    buff = malloc(len);
    if(!buff) std_err(NULL);

    snprintf(buff, len, HEADER, dir);

    if(MessageBox(0, buff, TITLE, MB_YESNO | MB_ICONINFORMATION | MB_TASKMODAL) == IDNO) return(0);

    free(buff);

    crc32csum = crc_file(FILENAME);
    if(crc32csum == CRC32P) {
        MessageBox(0, PATCHED, TITLE, MB_OK | MB_ICONINFORMATION | MB_TASKMODAL);
        return(0);
    } else if(crc32csum != CRC32O) {
        std_err(WRONGCRC);
    }

    copiafile(FILENAME, FNAMEBCK);

    patch_it();

    reg_it();

    MessageBox(0, SUCCESS, TITLE, MB_OK | MB_ICONINFORMATION | MB_TASKMODAL);

    return(0);
}



char *get_folder(void) {
    HKEY            key;
    DWORD           tmp;
    struct  stat    xstat;
    static char     folder[MAX_PATH];
    char            currdir[MAX_PATH];

    getcwd(currdir, MAX_PATH);

    if(!RegOpenKeyEx(HKEY_LOCAL_MACHINE, REGKEY, 0, KEY_SET_VALUE, &key)) {
        tmp = MAX_PATH;

    	if(!RegQueryValueEx(key, "Folder", NULL, NULL, folder, &tmp)) {
            RegCloseKey(key);
            strcat(folder, "\\System");
            chdir(folder);
        }
    }

    if(stat(FILENAME, &xstat) < 0) {
        chdir(currdir);
        if(stat(FILENAME, &xstat) < 0) {
            if((chdir("System") < 0) || (stat(FILENAME, &xstat) < 0)) {
                std_err(NOFILE);
            }
        }
    }

    getcwd(folder, MAX_PATH);
    return(folder);
}



void copiafile(char *inz, char *outz) {
    FILE            *fdin,
                    *fdout;
    int             len;
    unsigned char   *buff;

    fdin = fopen(inz, "rb");
    if(!fdin) std_err(NULL);

    fdout = fopen(outz, "wb");
    if(!fdout) std_err(NULL);

    buff = malloc(BUFFSZ);
    if(!buff) std_err(NULL);

    while((len = fread(buff, 1, BUFFSZ, fdin))) {
        if(fwrite(buff, len, 1, fdout) != 1) {
            std_err("Error while writing the backup copy of the file");
        }
    }

    fclose(fdin);
    fclose(fdout);
    free(buff);
}



void patch_it(void) {
    FILE            *fd;
    int             i;
    unsigned int    changes[][2] = {
        { 0x000F5648, 0xEB },
        { 0x000F5676, 0xEB },
        { 0x000F5677, 0x0F },
        { 0x000F58E0, 0xB0 },
        { 0x000F58E1, 0x01 },
        { 0x000F58E2, 0xC3 },
        { 0x00000000, 0x00 }    // delimiter
    };

    fd = fopen(FILENAME, "r+b");
    if(!fd) std_err(NULL);

    for(i = 0; changes[i][0]; i++) {
        if(fseek(fd, changes[i][0], SEEK_SET) < 0) std_err(NULL);
        fputc(changes[i][1], fd);
    }

    fclose(fd);
}



void reg_it(void) {
    HKEY            key;
    DWORD           tmp;
    char            cdkey[32];

    if(RegOpenKeyEx(HKEY_LOCAL_MACHINE, REGKEY, 0, KEY_SET_VALUE, &key)) {
        if(RegCreateKeyEx(
            HKEY_LOCAL_MACHINE,
            REGKEY,
            0,
            0,
            REG_OPTION_NON_VOLATILE,
            KEY_SET_VALUE,
            NULL,
            &key,
            &tmp)) {
            std_err("Impossible to access to the registry and create the key "REGKEY);
        }
    }

    tmp = sizeof(cdkey) - 1;
    if(!RegQueryValueEx(key, "CDKey", NULL, NULL, cdkey, &tmp)) {
        RegSetValueEx(key, "CDKey-backup", 0, REG_SZ, cdkey, tmp);
    }

    if(RegSetValueEx(key, "CDKey", 0, REG_SZ, DEMOKEY, sizeof(DEMOKEY) - 1)) {
        std_err("Impossible to set the demo key in "REGKEY);
    }

    RegCloseKey(key);
}



unsigned int crc_file(char *fname) {
    FILE            *fd;
    unsigned int    crc_32 = 0xffffffff;
    unsigned char   *buff;
    int             len;

    fd = fopen(fname, "rb");
    if(!fd) std_err(NULL);

    buff = malloc(BUFFSZ);
    if(!buff) std_err(NULL);

    while((len = fread(buff, 1, BUFFSZ, fd))) {
        crc_32 = crc32(crc_32, buff, len);
    }
    fclose(fd);

    free(buff);
    return(~crc_32);
}



void std_err(char *err) {
    if(!err) err = strerror(errno);
    MessageBox(0, err, TITLE, MB_OK | MB_ICONERROR | MB_TASKMODAL);
    exit(1);
}


